#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/domain/domain_fetcher.h>
#include <passport/infra/daemons/blackbox/src/protobuf/domain_lists.pb.h>

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/value.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <memory>

using namespace NPassport::NBb;
using namespace NPassport;

Y_UNIT_TEST_SUITE(PasspBbDomains) {
    static const char* const ADMIN_UID = "70502";
    static const char* const BORN_DATE = "0000-00-00 00:00:00";

    template <class T>
    void CreateRowImpl(NDbPool::TRow& row, T&& val) {
        row.emplace_back(std::move(val));
    }

    template <class T, class... Types>
    void CreateRowImpl(NDbPool::TRow& row, T&& val, Types&&... values) {
        row.emplace_back(std::move(val));
        CreateRowImpl(row, values...);
    }

    template <class... Types>
    NDbPool::TRow CreateRow(Types&&... values) {
        NDbPool::TRow row;
        CreateRowImpl(row, values...);
        return row;
    }

    Y_UNIT_TEST(domainMaster) {
        const TString name1 = "okna.ru";

        TDomain master({
            .Id = "129",
            .Master = "0",
            .Name = name1,
            .Ena = true,
            .AdminUid = ADMIN_UID,
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "Рога и копыта",
                .OrganizationId = "356",
            },
        });
        UNIT_ASSERT_VALUES_EQUAL("0", master.Master());
        UNIT_ASSERT(master.IsFound());
        UNIT_ASSERT(!master.IsSlave());
        UNIT_ASSERT_VALUES_EQUAL(name1, master.AsciiName());
        UNIT_ASSERT_VALUES_EQUAL("okna.ru", master.UtfName());
        UNIT_ASSERT_VALUES_EQUAL("129", master.Id());
        UNIT_ASSERT(master.Ena());
        UNIT_ASSERT(!master.Mx());
        UNIT_ASSERT_VALUES_EQUAL(ADMIN_UID, master.AdminUid());
        UNIT_ASSERT_VALUES_EQUAL("70500", master.DefaultUid());
        UNIT_ASSERT_VALUES_EQUAL("Рога и копыта", master.OrganizationName());
        UNIT_ASSERT_VALUES_EQUAL("356", master.OrganizationId());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves(), master.Slaves());
        master.AddSlave("300");
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves({"300"}), master.Slaves());
        master.AddSlave("300");
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves({"300"}), master.Slaves());
        master.DropSlave("300");
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves(), master.Slaves());
    }

    Y_UNIT_TEST(domainSlave) {
        const TString name2 = "xn--__-6kcilbdeuclcee7aehgfmf1ay6wqa.xn--p1ai";

        TDomain slave({
            .Id = "300",
            .Master = "129",
            .Name = name2,
            .Ena = false,
            .AdminUid = ADMIN_UID,
            .DefaultUid = "70500",
        });
        UNIT_ASSERT_VALUES_EQUAL("129", slave.Master());
        UNIT_ASSERT(slave.IsSlave());
        UNIT_ASSERT_VALUES_EQUAL(name2, slave.AsciiName());
        UNIT_ASSERT_VALUES_EQUAL("красивенький_модный_домен.рф", slave.UtfName());
        UNIT_ASSERT_VALUES_EQUAL("300", slave.Id());
        UNIT_ASSERT(!slave.Ena());
        UNIT_ASSERT(!slave.Mx());
        UNIT_ASSERT_VALUES_EQUAL(ADMIN_UID, slave.AdminUid());
        UNIT_ASSERT_VALUES_EQUAL("70500", slave.DefaultUid());
    }

    Y_UNIT_TEST(domainList) {
        const TString name1 = "okna.ru";
        const TString name2 = "xn--__-6kcilbdeuclcee7aehgfmf1ay6wqa.xn--p1ai";
        TDomain master({
            .Id = "129",
            .Master = "0",
            .Name = name1,
            .Ena = true,
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "Рога и копыта",
                .OrganizationId = "356",
            },
        });
        TDomain slave({
            .Id = "300",
            .Master = "129",
            .Name = name2,
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "Unused",
                .OrganizationId = "100",
            },
        });

        TDomainList l1;
        UNIT_ASSERT_VALUES_EQUAL(nullptr, l1.FindById("129"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, l1.FindById("300"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, l1.FindByName(name1));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, l1.FindByName(name2));

        l1.AddOrUpdateDomain(TDomain(slave));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, l1.FindById("129"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, l1.FindById("300"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, l1.FindByName(name1));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, l1.FindByName(name2));

        l1.AddOrUpdateDomain(TDomain(master));
        UNIT_ASSERT_EQUAL(master, *l1.FindById("129"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, l1.FindById("300"));
        UNIT_ASSERT_EQUAL(master, *l1.FindByName(name1));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, l1.FindByName(name2));

        l1.AddOrUpdateDomain(TDomain(slave));
        master.AddSlave(slave.Id());
        UNIT_ASSERT_EQUAL(master, *l1.FindById("129"));
        UNIT_ASSERT_EQUAL(slave, *l1.FindById("300"));
        UNIT_ASSERT_EQUAL(master, *l1.FindByName(name1));
        UNIT_ASSERT_EQUAL(slave, *l1.FindByName(name2));

        TDomainList l2(l1);
        UNIT_ASSERT_EQUAL(master, *l2.FindById("129"));
        UNIT_ASSERT_EQUAL(slave, *l2.FindById("300"));
        UNIT_ASSERT_EQUAL(master, *l2.FindByName(name1));
        UNIT_ASSERT_EQUAL(slave, *l2.FindByName(name2));

        l2.DeleteDomain(slave.Id());
        master.DropSlave(slave.Id());
        UNIT_ASSERT_EQUAL(master, *l2.FindById("129"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, l2.FindById("300"));
        UNIT_ASSERT_EQUAL(master, *l2.FindByName(name1));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, l2.FindByName(name2));

        l2.DeleteDomain(master.Id());
        UNIT_ASSERT_VALUES_EQUAL(nullptr, l2.FindById("129"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, l2.FindById("300"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, l2.FindByName(name1));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, l2.FindByName(name2));

        TDomainList l3(l1);
        l3.DeleteDomain(master.Id());
        UNIT_ASSERT_VALUES_EQUAL(nullptr, l3.FindById("129"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, l3.FindById("300"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, l3.FindByName(name1));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, l3.FindByName(name2));
    }

    Y_UNIT_TEST(domainCache) {
        const TString name1 = "okna.ru";
        const TString name2 = "xn--__-6kcilbdeuclcee7aehgfmf1ay6wqa.xn--p1ai";
        TDomain master({
            .Id = "129",
            .Master = "0",
            .Name = name1,
            .Ena = true,
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "Рога и копыта",
                .OrganizationId = "156",
            },
        });
        TDomain slave({
            .Id = "300",
            .Master = "129",
            .Name = name2,
            .DefaultUid = "70500",
        });
        master.AddSlave(slave.Id());

        TDomainCache cache;
        cache.List.AddOrUpdateDomain(TDomain(master));
        cache.List.AddOrUpdateDomain(TDomain(slave));
        cache.LastEventId = "100015";

        UNIT_ASSERT_VALUES_EQUAL("100015", cache.LastEventId);
        UNIT_ASSERT_EQUAL(master, *cache.List.FindById("129"));
        UNIT_ASSERT_EQUAL(slave, *cache.List.FindById("300"));
        UNIT_ASSERT_EQUAL(master, *cache.List.FindByName(name1));
        UNIT_ASSERT_EQUAL(slave, *cache.List.FindByName(name2));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindById("4"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindByName("123"));
    }

    Y_UNIT_TEST(domainBuilder_master) {
        const TString name1 = "okna.ru";
        TDomain master({
            .Id = "129",
            .Master = "0",
            .Name = name1,
            .Ena = true,
            .AdminUid = ADMIN_UID,
            .DefaultUid = "70500",
            .BornDate = BORN_DATE,
            .Options = {
                .OrganizationName = "Рога и копыта",
                .RawOptions = R"({"2" : "Рога и копыта"})",
            },
        });

        TDomainCache cache;
        TDomainListBuilder b(cache);
        b.SetLastEventId("100015");

        NDbPool::TRow row = CreateRow("129", "0", TString(name1), "1", "0", ADMIN_UID, "70500", BORN_DATE, R"({"2" : "Рога и копыта"})");
        UNIT_ASSERT_VALUES_EQUAL("129", b.AddRowAndGetDomId(std::move(row)));

        b.Finish();
        UNIT_ASSERT_VALUES_EQUAL(1, cache.List.size());
        UNIT_ASSERT_VALUES_EQUAL("100015", cache.LastEventId);
        UNIT_ASSERT_EQUAL(master, *cache.List.FindById("129"));
        UNIT_ASSERT_EQUAL(master, *cache.List.FindByName(name1));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindById("4"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindByName("123"));
    }

    Y_UNIT_TEST(domainBuilder_masterAndSlave) {
        const TString name1 = "okna.ru";
        const TString name2 = "xn--__-6kcilbdeuclcee7aehgfmf1ay6wqa.xn--p1ai";
        TDomain master({
            .Id = "129",
            .Master = "0",
            .Name = name1,
            .Ena = true,
            .AdminUid = ADMIN_UID,
            .DefaultUid = "70500",
            .BornDate = BORN_DATE,
            .Options = {
                .OrganizationName = "org_id",
                .RawOptions = R"({"2" : "org_id"})",
            },
        });
        TDomain slave({
            .Id = "300",
            .Master = "129",
            .Name = name2,
            .AdminUid = "70505",
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "unused value",
                .RawOptions = R"({"2" : "unused value"})",
            },
        });
        master.AddSlave(slave.Id());

        TDomainCache cache;
        TDomainListBuilder b(cache);
        b.SetLastEventId("100015");

        NDbPool::TRow row = CreateRow("129", "0", TString(name1), "1", "0", ADMIN_UID, "70500", BORN_DATE, R"({"2" : "org_id"})");
        UNIT_ASSERT_VALUES_EQUAL("129", b.AddRowAndGetDomId(std::move(row)));

        row = CreateRow("300", "129", TString(name2), "0", "0", "70505", "70500", BORN_DATE, R"({"2" : "unused value"})");
        UNIT_ASSERT_VALUES_EQUAL("300", b.AddRowAndGetDomId(std::move(row)));

        b.Finish();
        UNIT_ASSERT_VALUES_EQUAL("100015", cache.LastEventId);
        UNIT_ASSERT_VALUES_EQUAL(2, cache.List.size());
        UNIT_ASSERT_EQUAL(master, *cache.List.FindById("129"));
        UNIT_ASSERT_EQUAL(master, *cache.List.FindByName(name1));
        UNIT_ASSERT_EQUAL(slave, *cache.List.FindById("300"));
        UNIT_ASSERT_EQUAL(slave, *cache.List.FindByName(name2));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindById("4"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindByName("123"));
    }

    Y_UNIT_TEST(domainBuilder_slaveAndMaster) {
        const TString name1 = "okna.ru";
        const TString name2 = "xn--__-6kcilbdeuclcee7aehgfmf1ay6wqa.xn--p1ai";
        TDomain master({
            .Id = "1129",
            .Master = "0",
            .Name = name1,
            .Ena = true,
            .AdminUid = ADMIN_UID,
            .DefaultUid = "70500",
            .BornDate = BORN_DATE,
            .Options = {
                .OrganizationName = "org_id:-",
                .RawOptions = R"({"2" : "org_id:-"})",
            },
        });
        TDomain slave({
            .Id = "300",
            .Master = "1129",
            .Name = name2,
            .AdminUid = ADMIN_UID,
            .DefaultUid = "70500",
            .BornDate = BORN_DATE,
            .Options = {
                .RawOptions = "{}",
            },
        });
        master.AddSlave(slave.Id());

        TDomainCache cache;
        TDomainListBuilder b(cache);
        b.SetLastEventId("100015");

        NDbPool::TRow row = CreateRow("300", "1129", TString(name2), "0", "0", ADMIN_UID, "70500", BORN_DATE, R"({})");
        UNIT_ASSERT_VALUES_EQUAL("300", b.AddRowAndGetDomId(std::move(row)));

        row = CreateRow("1129", "0", TString(name1), "1", "0", ADMIN_UID, "70500", BORN_DATE, R"({"2" : "org_id:-"})");
        UNIT_ASSERT_VALUES_EQUAL("1129", b.AddRowAndGetDomId(std::move(row)));

        b.Finish();
        UNIT_ASSERT_VALUES_EQUAL("100015", cache.LastEventId);
        UNIT_ASSERT_VALUES_EQUAL(2, cache.List.size());
        UNIT_ASSERT_EQUAL(master, *cache.List.FindById("1129"));
        UNIT_ASSERT_EQUAL(master, *cache.List.FindByName(name1));
        UNIT_ASSERT_EQUAL(slave, *cache.List.FindById("300"));
        UNIT_ASSERT_EQUAL(slave, *cache.List.FindByName(name2));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindById("4"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindByName("123"));
    }

    Y_UNIT_TEST(domainBuilder_slave) {
        const TString name2 = "xn--__-6kcilbdeuclcee7aehgfmf1ay6wqa.xn--p1ai";
        TDomain slave({
            .Id = "300",
            .Master = "1129",
            .Name = name2,
            .DefaultUid = "70500",
        });

        TDomainCache cache;
        TDomainListBuilder b(cache);
        b.SetLastEventId("100015");

        NDbPool::TRow row = CreateRow("300", "1129", TString(name2), "0", "0", ADMIN_UID, "70500", BORN_DATE, R"({})");
        UNIT_ASSERT_VALUES_EQUAL("300", b.AddRowAndGetDomId(std::move(row)));

        b.Finish();
        UNIT_ASSERT_VALUES_EQUAL("100015", cache.LastEventId);
        UNIT_ASSERT_VALUES_EQUAL(0, cache.List.size());
    }

    Y_UNIT_TEST(domainBuilder_size) {
        TDomainCache cache;
        TDomainListBuilder b(cache);
        NDbPool::TRow row;
        NDbPool::TRow row2;
        row.resize(3);
        UNIT_ASSERT_EXCEPTION_CONTAINS(b.AddRowAndGetDomId(std::move(row)), yexception, "DomainListBuilder: response from db has invalid number of fields: 3");
        row2.resize(10);
        UNIT_ASSERT_EXCEPTION_CONTAINS(b.AddRowAndGetDomId(std::move(row2)), yexception, "DomainListBuilder: response from db has invalid number of fields: 10");

        NDbPool::TRow row3 = CreateRow("0", "129", "asdas", "0", "0", ADMIN_UID, "70501", BORN_DATE, "");
        UNIT_ASSERT_NO_EXCEPTION(b.AddRowAndGetDomId(std::move(row3)));
    }

    Y_UNIT_TEST(domainUpdater_addMaster) {
        const TString name1 = "okna.ru";
        TDomain master({
            .Id = "129",
            .Master = "0",
            .Name = name1,
            .Ena = true,
            .AdminUid = ADMIN_UID,
            .DefaultUid = "70500",
            .BornDate = BORN_DATE,
            .Options = {
                .OrganizationName = "Рога и копыта",
                .RawOptions = R"({"2" : "Рога и копыта"})",
            },
        });

        TDomainCache cache;
        TDomainListUpdater b(cache);

        NDbPool::TRow row = CreateRow("100015", "129", "0", TString(name1), "1", "0", ADMIN_UID, "70500", BORN_DATE, R"({"2" : "Рога и копыта"})");
        UNIT_ASSERT_VALUES_EQUAL((int)TDomainListUpdater::Event::GoNext,
                                 (int)b.AddRow(row));

        b.Finish();
        UNIT_ASSERT_VALUES_EQUAL("100015", cache.LastEventId);
        UNIT_ASSERT_VALUES_EQUAL(1, cache.List.size());
        UNIT_ASSERT_EQUAL(master, *cache.List.FindById("129"));
        UNIT_ASSERT_EQUAL(master, *cache.List.FindByName(name1));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindById("4"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindByName("123"));
    }

    Y_UNIT_TEST(domainUpdater_addMasterAndSlave) {
        const TString name1 = "okna.ru";
        const TString name2 = "xn--__-6kcilbdeuclcee7aehgfmf1ay6wqa.xn--p1ai";
        TDomain master({
            .Id = "129",
            .Master = "0",
            .Name = name1,
            .Ena = true,
            .AdminUid = ADMIN_UID,
            .DefaultUid = "70500",
            .BornDate = BORN_DATE,
            .Options = {
                .OrganizationName = "org_id",
                .RawOptions = R"({"2" : "org_id"})",
            },
        });
        TDomain slave({
            .Id = "300",
            .Master = "129",
            .Name = name2,
            .AdminUid = ADMIN_UID,
            .DefaultUid = "70500",
            .BornDate = BORN_DATE,
        });
        master.AddSlave(slave.Id());

        TDomainCache cache;
        TDomainListUpdater b(cache);

        NDbPool::TRow row = CreateRow("100015", "129", "0", TString(name1), "1", "0", ADMIN_UID, "70500", BORN_DATE, R"({"2" : "org_id"})");
        UNIT_ASSERT_VALUES_EQUAL((int)TDomainListUpdater::Event::GoNext,
                                 (int)b.AddRow(row));

        row = CreateRow("100016", "300", "129", TString(name2), "0", "0", ADMIN_UID, "70500", BORN_DATE, "");
        UNIT_ASSERT_VALUES_EQUAL((int)TDomainListUpdater::Event::GoNext,
                                 (int)b.AddRow(row));

        b.Finish();
        UNIT_ASSERT_VALUES_EQUAL("100016", cache.LastEventId);
        UNIT_ASSERT_VALUES_EQUAL(2, cache.List.size());
        UNIT_ASSERT_EQUAL(master, *cache.List.FindById("129"));
        UNIT_ASSERT_EQUAL(master, *cache.List.FindByName(name1));
        UNIT_ASSERT_EQUAL(slave, *cache.List.FindById("300"));
        UNIT_ASSERT_EQUAL(slave, *cache.List.FindByName(name2));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindById("4"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindByName("123"));
    }

    Y_UNIT_TEST(domainUpdater_addSlaveAndMaster) {
        const TString name1 = "okna.ru";
        const TString name2 = "xn--__-6kcilbdeuclcee7aehgfmf1ay6wqa.xn--p1ai";
        TDomain master({
            .Id = "129",
            .Master = "0",
            .Name = name1,
            .Ena = true,
            .AdminUid = ADMIN_UID,
            .DefaultUid = "70500",
            .BornDate = BORN_DATE,
            .Options = {
                .OrganizationName = "org_id:111bbbb",
                .RawOptions = R"({"2" : "org_id:111bbbb"})",
            },
        });
        TDomain slave({
            .Id = "300",
            .Master = "129",
            .Name = name2,
            .AdminUid = ADMIN_UID,
            .DefaultUid = "70500",
            .BornDate = BORN_DATE,
        });
        master.AddSlave(slave.Id());

        TDomainCache cache;
        TDomainListUpdater b(cache);

        NDbPool::TRow row = CreateRow("100015", "300", "129", TString(name2), "0", "0", ADMIN_UID, "70500", BORN_DATE, "");
        UNIT_ASSERT_VALUES_EQUAL((int)TDomainListUpdater::Event::GoNext,
                                 (int)b.AddRow(row));

        row = CreateRow("100016", "129", "0", TString(name1), "1", "0", ADMIN_UID, "70500", BORN_DATE, R"({"2" : "org_id:111bbbb"})");
        UNIT_ASSERT_VALUES_EQUAL((int)TDomainListUpdater::Event::GoNext,
                                 (int)b.AddRow(row));

        b.Finish();
        UNIT_ASSERT_VALUES_EQUAL("100016", cache.LastEventId);
        UNIT_ASSERT_VALUES_EQUAL(2, cache.List.size());
        UNIT_ASSERT_EQUAL(master, *cache.List.FindById("129"));
        UNIT_ASSERT_EQUAL(master, *cache.List.FindByName(name1));
        UNIT_ASSERT_EQUAL(slave, *cache.List.FindById("300"));
        UNIT_ASSERT_EQUAL(slave, *cache.List.FindByName(name2));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindById("4"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindByName("123"));
    }

    Y_UNIT_TEST(domainUpdater_addAndDeleteMaster) {
        const TString name1 = "okna.ru";
        const TString name2 = "xn--__-6kcilbdeuclcee7aehgfmf1ay6wqa.xn--p1ai";
        TDomain master({
            .Id = "129",
            .Master = "0",
            .Name = name1,
            .Ena = true,
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "Рога и копыта",
                .OrganizationId = "100",
            },
        });
        TDomain slave({
            .Id = "300",
            .Master = "129",
            .Name = name2,
            .DefaultUid = "70500",
            .Options = {
                .OrganizationId = "0",
            },
        });
        master.AddSlave(slave.Id());

        TDomainCache cache;
        cache.List.AddOrUpdateDomain(TDomain(master));
        TDomainListUpdater b(cache);

        NDbPool::TRow row = CreateRow("100016", "300", "129", TString(name2), "0", "0", ADMIN_UID, "70500", BORN_DATE, "");
        UNIT_ASSERT_VALUES_EQUAL((int)TDomainListUpdater::Event::GoNext,
                                 (int)b.AddRow(row));

        row = CreateRow("100017", "129", "", NDbPool::TValue("", true), "", "", "", "", "", "");
        UNIT_ASSERT_VALUES_EQUAL((int)TDomainListUpdater::Event::GoNext,
                                 (int)b.AddRow(row));

        b.Finish();
        UNIT_ASSERT_VALUES_EQUAL("100017", cache.LastEventId);
        UNIT_ASSERT_VALUES_EQUAL(0, cache.List.size());
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindById("129"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindByName(name1));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindById("300"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindByName(name2));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindById("4"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindByName("123"));
    }

    Y_UNIT_TEST(domainUpdater_updateMaster) {
        const TString name1 = "okna.ru";
        const TString name2 = "xn--__-6kcilbdeuclcee7aehgfmf1ay6wqa.xn--p1ai";
        TDomain master({
            .Id = "129",
            .Master = "0",
            .Name = name1,
            .Ena = true,
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "Рога и копыта",
                .OrganizationId = "100",
            },
        });
        TDomain slave({
            .Id = "300",
            .Master = "129",
            .Name = name2,
            .DefaultUid = "70500",
            .Options = {
                .OrganizationId = "0",
            },
        });
        master.AddSlave(slave.Id());

        TDomainCache cache;
        cache.List.AddOrUpdateDomain(TDomain(master));
        cache.List.AddOrUpdateDomain(TDomain(slave));
        TDomainListUpdater b(cache);

        NDbPool::TRow row = CreateRow("100016", "129", "0", TString(name1), "0", "0", ADMIN_UID, "70501", BORN_DATE, "");
        UNIT_ASSERT_VALUES_EQUAL((int)TDomainListUpdater::Event::GoNext,
                                 (int)b.AddRow(row));

        b.Finish();
        UNIT_ASSERT_VALUES_EQUAL("100016", cache.LastEventId);
        UNIT_ASSERT_VALUES_EQUAL(2, cache.List.size());

        UNIT_ASSERT_VALUES_EQUAL(cache.List.FindByName(name1), cache.List.FindById("129"));
        const TDomain* m = cache.List.FindById("129");
        UNIT_ASSERT_VALUES_EQUAL(name1, m->AsciiName());
        UNIT_ASSERT(!m->Ena());
        UNIT_ASSERT(!m->Mx());
        UNIT_ASSERT_VALUES_EQUAL(ADMIN_UID, m->AdminUid());
        UNIT_ASSERT_VALUES_EQUAL("70501", m->DefaultUid());
        UNIT_ASSERT(m->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves({"300"}), m->Slaves());

        UNIT_ASSERT_EQUAL(slave, *cache.List.FindById("300"));
        UNIT_ASSERT_EQUAL(slave, *cache.List.FindByName(name2));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindById("4"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindByName("123"));
    }

    Y_UNIT_TEST(domainUpdater_updateSlave) {
        const TString name1 = "okna.ru";
        const TString name2 = "xn--__-6kcilbdeuclcee7aehgfmf1ay6wqa.xn--p1ai";
        const TString name3 = "mail.ru";
        TDomain master({
            .Id = "129",
            .Master = "0",
            .Name = name1,
            .Ena = true,
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "org_id:1234",
                .OrganizationId = "1234",
            },
        });
        TDomain slave({
            .Id = "300",
            .Master = "129",
            .Name = name2,
            .DefaultUid = "70500",
        });
        master.AddSlave(slave.Id());

        TDomainCache cache;
        cache.List.AddOrUpdateDomain(TDomain(master));
        cache.List.AddOrUpdateDomain(TDomain(slave));
        TDomainListUpdater b(cache);

        NDbPool::TRow row = CreateRow("100016", "300", "129", TString(name3), "0", "0", ADMIN_UID, "70501", BORN_DATE, R"("2":"org_id:1234")");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext,
                          b.AddRow(row));

        b.Finish();
        UNIT_ASSERT_VALUES_EQUAL("100016", cache.LastEventId);
        UNIT_ASSERT_VALUES_EQUAL(2, cache.List.size());
        UNIT_ASSERT_EQUAL(master, *cache.List.FindById("129"));
        UNIT_ASSERT_EQUAL(master, *cache.List.FindByName(name1));

        UNIT_ASSERT_VALUES_EQUAL(cache.List.FindByName(name3), cache.List.FindById("300"));
        const TDomain* s = cache.List.FindById("300");
        UNIT_ASSERT_VALUES_EQUAL(name3, s->AsciiName());
        UNIT_ASSERT(!s->Ena());
        UNIT_ASSERT(!s->Mx());
        UNIT_ASSERT_VALUES_EQUAL(ADMIN_UID, s->AdminUid());
        UNIT_ASSERT_VALUES_EQUAL("70501", s->DefaultUid());
        UNIT_ASSERT(s->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves(), s->Slaves());

        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindById("4"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindByName("123"));
    }

    Y_UNIT_TEST(domainUpdater_deleteSwapped) {
        const TString name1 = "visaconcord.com";
        const TString name2 = "visaconcord.ru";
        TDomain master({
            .Id = "490847",
            .Master = "0",
            .Name = name1,
            .Ena = true,
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "some",
                .OrganizationId = "5",
            },
        });
        TDomain slave({
            .Id = "3427166",
            .Master = "490847",
            .Name = name2,
            .DefaultUid = "70500",
        });
        master.AddSlave(slave.Id());

        TDomainCache cache;
        cache.List.AddOrUpdateDomain(TDomain(master));
        cache.List.AddOrUpdateDomain(TDomain(slave));
        TDomainListUpdater b(cache);

        NDbPool::TRow row = CreateRow("100016", "490847", "0", TString(name2), "0", "0", ADMIN_UID, "70501", BORN_DATE, "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        row = CreateRow("100017", "3427166", "", NDbPool::TValue("", true), "", "", "", "", "", "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        b.Finish();
        UNIT_ASSERT_VALUES_EQUAL("100017", cache.LastEventId);
        UNIT_ASSERT_VALUES_EQUAL(1, cache.List.size());

        const TDomain* m = cache.List.FindById("490847");
        UNIT_ASSERT(m);
        UNIT_ASSERT_VALUES_EQUAL(name2, m->AsciiName());
        UNIT_ASSERT(!m->Ena());
        UNIT_ASSERT(!m->Mx());
        UNIT_ASSERT_VALUES_EQUAL(ADMIN_UID, m->AdminUid());
        UNIT_ASSERT_VALUES_EQUAL("70501", m->DefaultUid());
        UNIT_ASSERT(m->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves(), m->Slaves());

        m = cache.List.FindByName(name2);
        UNIT_ASSERT(m);
        UNIT_ASSERT_VALUES_EQUAL("490847", m->Id());

        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindById("3427166"));
        UNIT_ASSERT_VALUES_EQUAL(nullptr, cache.List.FindByName(name1));
    }

    Y_UNIT_TEST(domainUpdater_swap) {
        TDomain master({
            .Id = "101",
            .Master = "0",
            .Name = "name1",
            .Ena = true,
            .DefaultUid = "70500",
        });
        TDomain slave({
            .Id = "102",
            .Master = "101",
            .Name = "name2",
            .DefaultUid = "70500",
        });
        master.AddSlave(slave.Id());

        TDomainCache cache;
        cache.List.AddOrUpdateDomain(TDomain(master));
        cache.List.AddOrUpdateDomain(TDomain(slave));
        TDomainListUpdater b(cache);

        NDbPool::TRow row = CreateRow("100016", "101", "0", "name2", "0", "0", ADMIN_UID, "70500", BORN_DATE, "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        row = CreateRow("100017", "102", "101", "name1", "0", "0", ADMIN_UID, "70500", BORN_DATE, "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        b.Finish();
        UNIT_ASSERT_VALUES_EQUAL("100017", cache.LastEventId);
        UNIT_ASSERT_VALUES_EQUAL(2, cache.List.size());

        const TDomain* m = cache.List.FindById("101");
        UNIT_ASSERT(m);
        UNIT_ASSERT_VALUES_EQUAL("name2", m->AsciiName());
        UNIT_ASSERT_VALUES_EQUAL("name2", m->UtfName());
        UNIT_ASSERT(!m->Ena());
        UNIT_ASSERT(!m->Mx());
        UNIT_ASSERT_VALUES_EQUAL(ADMIN_UID, m->AdminUid());
        UNIT_ASSERT_VALUES_EQUAL("70500", m->DefaultUid());
        UNIT_ASSERT(m->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves({"102"}), m->Slaves());

        m = cache.List.FindByName("name2");
        UNIT_ASSERT(m);
        UNIT_ASSERT_VALUES_EQUAL("101", m->Id());

        const TDomain* s = cache.List.FindById("102");
        UNIT_ASSERT(s);
        UNIT_ASSERT_VALUES_EQUAL("name1", s->AsciiName());
        UNIT_ASSERT_VALUES_EQUAL("name1", s->UtfName());
        UNIT_ASSERT(!s->Ena());
        UNIT_ASSERT(!s->Mx());
        UNIT_ASSERT_VALUES_EQUAL(ADMIN_UID, m->AdminUid());
        UNIT_ASSERT_VALUES_EQUAL("70500", s->DefaultUid());
        UNIT_ASSERT(s->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves(), s->Slaves());

        s = cache.List.FindByName("name1");
        UNIT_ASSERT(s);
        UNIT_ASSERT_VALUES_EQUAL("102", s->Id());
    }

    Y_UNIT_TEST(domainUpdater_swap3_ver1) {
        TDomain master({
            .Id = "101",
            .Master = "0",
            .Name = "name1",
            .Ena = true,
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "org",
                .OrganizationId = "1",
            },
        });
        TDomain slave({
            .Id = "102",
            .Master = "101",
            .Name = "name2",
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "unused",
            },
        });
        TDomain slave2({
            .Id = "103",
            .Master = "101",
            .Name = "name3",
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "ignored",
                .OrganizationId = "too",
            },
        });
        master.AddSlave(slave.Id());

        TDomainCache cache;
        cache.List.AddOrUpdateDomain(TDomain(master));
        cache.List.AddOrUpdateDomain(TDomain(slave));
        cache.List.AddOrUpdateDomain(TDomain(slave2));
        TDomainListUpdater b(cache);

        NDbPool::TRow row = CreateRow("100016", "101", "0", "name2", "0", "0", ADMIN_UID, "70500", BORN_DATE, "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        row = CreateRow("100017", "102", "101", "name3", "0", "0", ADMIN_UID, "70500", BORN_DATE, "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        row = CreateRow("100018", "103", "101", "name1", "0", "0", ADMIN_UID, "70500", BORN_DATE, "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        b.Finish();
        UNIT_ASSERT_VALUES_EQUAL("100018", cache.LastEventId);
        UNIT_ASSERT_VALUES_EQUAL(3, cache.List.size());

        const TDomain* m = cache.List.FindById("101");
        UNIT_ASSERT(m);
        UNIT_ASSERT_VALUES_EQUAL("name2", m->AsciiName());
        UNIT_ASSERT_VALUES_EQUAL("name2", m->UtfName());
        UNIT_ASSERT(!m->Ena());
        UNIT_ASSERT(!m->Mx());
        UNIT_ASSERT_VALUES_EQUAL(ADMIN_UID, m->AdminUid());
        UNIT_ASSERT_VALUES_EQUAL("70500", m->DefaultUid());
        UNIT_ASSERT(m->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves({"102", "103"}), m->Slaves());

        m = cache.List.FindByName("name2");
        UNIT_ASSERT(m);
        UNIT_ASSERT_VALUES_EQUAL("101", m->Id());

        const TDomain* s = cache.List.FindById("102");
        UNIT_ASSERT(s);
        UNIT_ASSERT_VALUES_EQUAL("name3", s->AsciiName());
        UNIT_ASSERT_VALUES_EQUAL("name3", s->UtfName());
        UNIT_ASSERT(!s->Ena());
        UNIT_ASSERT(!s->Mx());
        UNIT_ASSERT_VALUES_EQUAL(ADMIN_UID, m->AdminUid());
        UNIT_ASSERT_VALUES_EQUAL("70500", s->DefaultUid());
        UNIT_ASSERT(s->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves(), s->Slaves());

        s = cache.List.FindByName("name3");
        UNIT_ASSERT(s);
        UNIT_ASSERT_VALUES_EQUAL("102", s->Id());

        const TDomain* s2 = cache.List.FindById("103");
        UNIT_ASSERT(s2);
        UNIT_ASSERT_VALUES_EQUAL("name1", s2->AsciiName());
        UNIT_ASSERT_VALUES_EQUAL("name1", s2->UtfName());
        UNIT_ASSERT(!s2->Ena());
        UNIT_ASSERT(!s2->Mx());
        UNIT_ASSERT_VALUES_EQUAL("70500", s2->DefaultUid());
        UNIT_ASSERT(s2->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves(), s2->Slaves());

        s2 = cache.List.FindByName("name1");
        UNIT_ASSERT(s2);
        UNIT_ASSERT_VALUES_EQUAL("103", s2->Id());
    }

    Y_UNIT_TEST(domainUpdater_swap3_ver2) {
        TDomain master({
            .Id = "101",
            .Master = "0",
            .Name = "name1",
            .Ena = true,
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "org",
                .OrganizationId = "1",
            },
        });
        TDomain slave({
            .Id = "102",
            .Master = "101",
            .Name = "name2",
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "unused",
            },
        });
        TDomain slave2({
            .Id = "103",
            .Master = "101",
            .Name = "name3",
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "ignored",
                .OrganizationId = "too",
            },
        });
        master.AddSlave(slave.Id());

        TDomainCache cache;
        cache.List.AddOrUpdateDomain(TDomain(master));
        cache.List.AddOrUpdateDomain(TDomain(slave));
        cache.List.AddOrUpdateDomain(TDomain(slave2));
        TDomainListUpdater b(cache);

        NDbPool::TRow row = CreateRow("100016", "101", "0", "name3", "0", "0", ADMIN_UID, "70500", BORN_DATE, "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        row = CreateRow("100017", "102", "101", "name1", "0", "0", ADMIN_UID, "70500", BORN_DATE, "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        row = CreateRow("100018", "103", "101", "name2", "0", "0", ADMIN_UID, "70500", BORN_DATE, "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        b.Finish();
        UNIT_ASSERT_VALUES_EQUAL("100018", cache.LastEventId);
        UNIT_ASSERT_VALUES_EQUAL(3, cache.List.size());

        const TDomain* m = cache.List.FindById("101");
        UNIT_ASSERT(m);
        UNIT_ASSERT_VALUES_EQUAL("name3", m->AsciiName());
        UNIT_ASSERT_VALUES_EQUAL("name3", m->UtfName());
        UNIT_ASSERT(!m->Ena());
        UNIT_ASSERT(!m->Mx());
        UNIT_ASSERT_VALUES_EQUAL("70500", m->DefaultUid());
        UNIT_ASSERT(m->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves({"102", "103"}), m->Slaves());

        m = cache.List.FindByName("name3");
        UNIT_ASSERT(m);
        UNIT_ASSERT_VALUES_EQUAL("101", m->Id());

        const TDomain* s = cache.List.FindById("102");
        UNIT_ASSERT(s);
        UNIT_ASSERT_VALUES_EQUAL("name1", s->AsciiName());
        UNIT_ASSERT_VALUES_EQUAL("name1", s->UtfName());
        UNIT_ASSERT(!s->Ena());
        UNIT_ASSERT(!s->Mx());
        UNIT_ASSERT_VALUES_EQUAL("70500", s->DefaultUid());
        UNIT_ASSERT(s->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves(), s->Slaves());

        s = cache.List.FindByName("name1");
        UNIT_ASSERT(s);
        UNIT_ASSERT_VALUES_EQUAL("102", s->Id());

        const TDomain* s2 = cache.List.FindById("103");
        UNIT_ASSERT(s2);
        UNIT_ASSERT_VALUES_EQUAL("name2", s2->AsciiName());
        UNIT_ASSERT_VALUES_EQUAL("name2", s2->UtfName());
        UNIT_ASSERT(!s2->Ena());
        UNIT_ASSERT(!s2->Mx());
        UNIT_ASSERT_VALUES_EQUAL("70500", s2->DefaultUid());
        UNIT_ASSERT(s2->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves(), s2->Slaves());

        s2 = cache.List.FindByName("name2");
        UNIT_ASSERT(s2);
        UNIT_ASSERT_VALUES_EQUAL("103", s2->Id());
    }

    Y_UNIT_TEST(domainUpdater_swap3_withDeletion_ver1) {
        TDomain master({
            .Id = "101",
            .Master = "0",
            .Name = "name1",
            .Ena = true,
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "org",
                .OrganizationId = "1",
            },
        });
        TDomain slave({
            .Id = "102",
            .Master = "101",
            .Name = "name2",
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "unused",
            },
        });
        TDomain slave2({
            .Id = "103",
            .Master = "101",
            .Name = "name3",
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "ignored",
                .OrganizationId = "too",
            },
        });
        master.AddSlave(slave.Id());

        TDomainCache cache;
        cache.List.AddOrUpdateDomain(TDomain(master));
        cache.List.AddOrUpdateDomain(TDomain(slave));
        cache.List.AddOrUpdateDomain(TDomain(slave2));
        TDomainListUpdater b(cache);

        NDbPool::TRow row = CreateRow("100016", "101", "0", "name2", "0", "0", ADMIN_UID, "70500", BORN_DATE, "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        row = CreateRow("100017", "102", "101", "name3", "0", "0", ADMIN_UID, "70500", BORN_DATE, "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        row = CreateRow("100018", "103", "", NDbPool::TValue("", true), "", "", "", "", "", "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        b.Finish();
        UNIT_ASSERT_VALUES_EQUAL("100018", cache.LastEventId);
        UNIT_ASSERT_VALUES_EQUAL(2, cache.List.size());

        const TDomain* m = cache.List.FindById("101");
        UNIT_ASSERT(m);
        UNIT_ASSERT_VALUES_EQUAL("name2", m->AsciiName());
        UNIT_ASSERT_VALUES_EQUAL("name2", m->UtfName());
        UNIT_ASSERT(!m->Ena());
        UNIT_ASSERT(!m->Mx());
        UNIT_ASSERT_VALUES_EQUAL("70500", m->DefaultUid());
        UNIT_ASSERT(m->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves({"102"}), m->Slaves());

        m = cache.List.FindByName("name2");
        UNIT_ASSERT(m);
        UNIT_ASSERT_VALUES_EQUAL("101", m->Id());

        const TDomain* s = cache.List.FindById("102");
        UNIT_ASSERT(s);
        UNIT_ASSERT_VALUES_EQUAL("name3", s->AsciiName());
        UNIT_ASSERT_VALUES_EQUAL("name3", s->UtfName());
        UNIT_ASSERT(!s->Ena());
        UNIT_ASSERT(!s->Mx());
        UNIT_ASSERT_VALUES_EQUAL("70500", s->DefaultUid());
        UNIT_ASSERT(s->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves(), s->Slaves());

        s = cache.List.FindByName("name3");
        UNIT_ASSERT(s);
        UNIT_ASSERT_VALUES_EQUAL("102", s->Id());

        UNIT_ASSERT_EQUAL(nullptr, cache.List.FindById("103"));
        UNIT_ASSERT_EQUAL(nullptr, cache.List.FindByName("name1"));
    }

    Y_UNIT_TEST(domainUpdater_swap3_withDeletion_ver2) {
        TDomain master({
            .Id = "101",
            .Master = "0",
            .Name = "name1",
            .Ena = true,
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "org",
                .OrganizationId = "1",
            },
        });
        TDomain slave({
            .Id = "102",
            .Master = "101",
            .Name = "name2",
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "unused",
            },
        });
        TDomain slave2({
            .Id = "103",
            .Master = "101",
            .Name = "name3",
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "ignored",
                .OrganizationId = "too",
            },
        });
        master.AddSlave(slave.Id());

        TDomainCache cache;
        cache.List.AddOrUpdateDomain(TDomain(master));
        cache.List.AddOrUpdateDomain(TDomain(slave));
        cache.List.AddOrUpdateDomain(TDomain(slave2));
        TDomainListUpdater b(cache);

        NDbPool::TRow row = CreateRow("100016", "101", "0", "name3", "0", "0", ADMIN_UID, "70500", BORN_DATE, "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        row = CreateRow("100017", "102", "101", "name1", "0", "0", ADMIN_UID, "70500", BORN_DATE, "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        row = CreateRow("100018", "103", "", NDbPool::TValue("", true), "", "", "", "", "", "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        b.Finish();
        UNIT_ASSERT_VALUES_EQUAL("100018", cache.LastEventId);
        UNIT_ASSERT_VALUES_EQUAL(2, cache.List.size());

        const TDomain* m = cache.List.FindById("101");
        UNIT_ASSERT(m);
        UNIT_ASSERT_VALUES_EQUAL("name3", m->AsciiName());
        UNIT_ASSERT_VALUES_EQUAL("name3", m->UtfName());
        UNIT_ASSERT(!m->Ena());
        UNIT_ASSERT(!m->Mx());
        UNIT_ASSERT_VALUES_EQUAL("70500", m->DefaultUid());
        UNIT_ASSERT(m->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves({"102"}), m->Slaves());

        m = cache.List.FindByName("name3");
        UNIT_ASSERT(m);
        UNIT_ASSERT_VALUES_EQUAL("101", m->Id());

        const TDomain* s = cache.List.FindById("102");
        UNIT_ASSERT(s);
        UNIT_ASSERT_VALUES_EQUAL("name1", s->AsciiName());
        UNIT_ASSERT_VALUES_EQUAL("name1", s->UtfName());
        UNIT_ASSERT(!s->Ena());
        UNIT_ASSERT(!s->Mx());
        UNIT_ASSERT_VALUES_EQUAL("70500", s->DefaultUid());
        UNIT_ASSERT(s->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves(), s->Slaves());

        s = cache.List.FindByName("name1");
        UNIT_ASSERT(s);
        UNIT_ASSERT_VALUES_EQUAL("102", s->Id());

        UNIT_ASSERT_EQUAL(nullptr, cache.List.FindById("103"));
        UNIT_ASSERT_EQUAL(nullptr, cache.List.FindByName("name2"));
    }

    Y_UNIT_TEST(domainUpdater_swap3_withDeletion_ver3) {
        TDomain master({
            .Id = "101",
            .Master = "0",
            .Name = "name1",
            .Ena = true,
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "org",
                .OrganizationId = "1",
            },
        });
        TDomain slave({
            .Id = "102",
            .Master = "101",
            .Name = "name2",
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "unused",
                .OrganizationId = "",
            },
        });
        TDomain slave2({
            .Id = "103",
            .Master = "101",
            .Name = "name3",
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "ignored",
                .OrganizationId = "too",
            },
        });
        master.AddSlave(slave.Id());

        TDomainCache cache;
        cache.List.AddOrUpdateDomain(TDomain(master));
        cache.List.AddOrUpdateDomain(TDomain(slave));
        cache.List.AddOrUpdateDomain(TDomain(slave2));
        TDomainListUpdater b(cache);

        NDbPool::TRow row = CreateRow("100016", "101", "0", "name3", "0", "0", ADMIN_UID, "70500", BORN_DATE, "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        row = CreateRow("100017", "102", "", NDbPool::TValue("", true), "", "", "", "", "", "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        row = CreateRow("100018", "103", "101", "name2", "0", "0", ADMIN_UID, "70500", BORN_DATE, "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        b.Finish();
        UNIT_ASSERT_VALUES_EQUAL("100018", cache.LastEventId);
        UNIT_ASSERT_VALUES_EQUAL(2, cache.List.size());

        const TDomain* m = cache.List.FindById("101");
        UNIT_ASSERT(m);
        UNIT_ASSERT_VALUES_EQUAL("name3", m->AsciiName());
        UNIT_ASSERT_VALUES_EQUAL("name3", m->UtfName());
        UNIT_ASSERT(!m->Ena());
        UNIT_ASSERT(!m->Mx());
        UNIT_ASSERT_VALUES_EQUAL("70500", m->DefaultUid());
        UNIT_ASSERT(m->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves({"103"}), m->Slaves());

        m = cache.List.FindByName("name3");
        UNIT_ASSERT(m);
        UNIT_ASSERT_VALUES_EQUAL("101", m->Id());

        const TDomain* s = cache.List.FindById("103");
        UNIT_ASSERT(s);
        UNIT_ASSERT_VALUES_EQUAL("name2", s->AsciiName());
        UNIT_ASSERT_VALUES_EQUAL("name2", s->UtfName());
        UNIT_ASSERT(!s->Ena());
        UNIT_ASSERT(!s->Mx());
        UNIT_ASSERT_VALUES_EQUAL("70500", s->DefaultUid());
        UNIT_ASSERT(s->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves(), s->Slaves());

        s = cache.List.FindByName("name2");
        UNIT_ASSERT(s);
        UNIT_ASSERT_VALUES_EQUAL("103", s->Id());

        UNIT_ASSERT_EQUAL(nullptr, cache.List.FindById("102"));
        UNIT_ASSERT_EQUAL(nullptr, cache.List.FindByName("name1"));
    }

    Y_UNIT_TEST(domainUpdater_swap3_withDeletion_ver4) {
        TDomain master({
            .Id = "101",
            .Master = "0",
            .Name = "name1",
            .Ena = true,
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "org",
                .OrganizationId = "1",
            },
        });
        TDomain slave({
            .Id = "102",
            .Master = "101",
            .Name = "name2",
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "unused",
            },
        });
        TDomain slave2({
            .Id = "103",
            .Master = "101",
            .Name = "name3",
            .DefaultUid = "70500",
            .Options = {
                .OrganizationName = "ignored",
                .OrganizationId = "too",
            },
        });
        master.AddSlave(slave.Id());

        TDomainCache cache;
        cache.List.AddOrUpdateDomain(TDomain(master));
        cache.List.AddOrUpdateDomain(TDomain(slave));
        cache.List.AddOrUpdateDomain(TDomain(slave2));
        TDomainListUpdater b(cache);

        NDbPool::TRow row = CreateRow("100016", "101", "0", "name1", "0", "0", ADMIN_UID, "70500", BORN_DATE, "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        row = CreateRow("100017", "102", "", NDbPool::TValue("", true), "", "", "", "", "", "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        row = CreateRow("100018", "103", "101", "name2", "0", "0", ADMIN_UID, "70500", BORN_DATE, "");
        UNIT_ASSERT_EQUAL(TDomainListUpdater::Event::GoNext, b.AddRow(row));

        b.Finish();
        UNIT_ASSERT_VALUES_EQUAL("100018", cache.LastEventId);
        UNIT_ASSERT_VALUES_EQUAL(2, cache.List.size());

        const TDomain* m = cache.List.FindById("101");
        UNIT_ASSERT(m);
        UNIT_ASSERT_VALUES_EQUAL("name1", m->AsciiName());
        UNIT_ASSERT_VALUES_EQUAL("name1", m->UtfName());
        UNIT_ASSERT(!m->Ena());
        UNIT_ASSERT(!m->Mx());
        UNIT_ASSERT_VALUES_EQUAL("70500", m->DefaultUid());
        UNIT_ASSERT(m->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves({"103"}), m->Slaves());

        m = cache.List.FindByName("name1");
        UNIT_ASSERT(m);
        UNIT_ASSERT_VALUES_EQUAL("101", m->Id());

        const TDomain* s = cache.List.FindById("103");
        UNIT_ASSERT(s);
        UNIT_ASSERT_VALUES_EQUAL("name2", s->AsciiName());
        UNIT_ASSERT_VALUES_EQUAL("name2", s->UtfName());
        UNIT_ASSERT(!s->Ena());
        UNIT_ASSERT(!s->Mx());
        UNIT_ASSERT_VALUES_EQUAL("70500", s->DefaultUid());
        UNIT_ASSERT(s->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves(), s->Slaves());

        s = cache.List.FindByName("name2");
        UNIT_ASSERT(s);
        UNIT_ASSERT_VALUES_EQUAL("103", s->Id());

        UNIT_ASSERT_EQUAL(nullptr, cache.List.FindById("102"));
        UNIT_ASSERT_EQUAL(nullptr, cache.List.FindByName("name3"));
    }

    Y_UNIT_TEST(domainUpdater_reload) {
        TDomainCache cache;
        TDomainListUpdater b(cache);
        NDbPool::TRow row = CreateRow("100016", "0", "129", "asdasd", "0", "0", ADMIN_UID, "70501", BORN_DATE, "");
        UNIT_ASSERT_VALUES_EQUAL((int)TDomainListUpdater::Event::Reload,
                                 (int)b.AddRow(row));
    }

    Y_UNIT_TEST(domainUpdater_size) {
        TDomainCache cache;
        TDomainListUpdater b(cache);
        NDbPool::TRow row;
        NDbPool::TRow row2;
        row.resize(3);
        UNIT_ASSERT_EXCEPTION_CONTAINS(b.AddRow(row), yexception, "DomainListUpdater: response from db has invalid number of fields: 3");
        row2.resize(9);
        UNIT_ASSERT_EXCEPTION_CONTAINS(b.AddRow(row2), yexception, "DomainListUpdater: response from db has invalid number of fields: 9");

        NDbPool::TRow row3 = CreateRow("100016", "0", "129", "asdas", "0", "0", ADMIN_UID, "70501", BORN_DATE, "");
        UNIT_ASSERT_NO_EXCEPTION(b.AddRow(row3));
    }

    Y_UNIT_TEST(domaFetcher_findSolo) {
        const NDbPool::TDestinationPtr dsn = NDbPool::TDestination::CreateSqlite(TestsDir() + "db/safeguarddb.sqlite3.sql");
        std::unique_ptr<NDbPool::TDbPool> pool = std::make_unique<NDbPool::TDbPool>(NDbPool::TDbPoolSettings{
            .Dsn = dsn,
        });

        TDomainFetcher f(*pool, {});

        TDomainFetcher::TResult res = f.Find("ololo.ru", TDomainFetcher::EPolicy::MasterOnly);
        UNIT_ASSERT(!res);

        TDomain d({
            .Id = "1",
            .Master = "0",
            .Name = "okna.ru",
            .Ena = true,
            .AdminUid = "1130000000000383",
            .DefaultUid = "1130000000000658",
            .BornDate = BORN_DATE,
        });

        res = f.Find("okna.ru", TDomainFetcher::EPolicy::MasterOnly);
        UNIT_ASSERT(res);
        UNIT_ASSERT_EQUAL(d, *res.Domain);
        res = f.Find("okna.ru", TDomainFetcher::EPolicy::ReplaceAliasWithMaster);
        UNIT_ASSERT(res);
        UNIT_ASSERT_EQUAL(d, *res.Domain);

        res = f.FindById("1", TDomainFetcher::EPolicy::MasterOnly);
        UNIT_ASSERT(res);
        UNIT_ASSERT_EQUAL(d, *res.Domain);
        res = f.FindById("1", TDomainFetcher::EPolicy::ReplaceAliasWithMaster);
        UNIT_ASSERT(res);
        UNIT_ASSERT_EQUAL(d, *res.Domain);
    }

    Y_UNIT_TEST(domaFetcher_findMasterAndSlave) {
        const NDbPool::TDestinationPtr dsn = NDbPool::TDestination::CreateSqlite(TestsDir() + "db/safeguarddb.sqlite3.sql");
        std::unique_ptr<NDbPool::TDbPool> pool = std::make_unique<NDbPool::TDbPool>(NDbPool::TDbPoolSettings{
            .Dsn = dsn,
        });

        TDomainFetcher f(*pool, {});

        TDomain m({
            .Id = "15",
            .Master = "0",
            .Name = "dveri.ru",
            .Ena = true,
            .Mx = true,
            .AdminUid = "0",
            .DefaultUid = "0",
            .BornDate = "2010-03-05 11:20:49",
        });
        TDomain s({
            .Id = "22",
            .Master = "15",
            .Name = "zveri.ru",
            .Ena = true,
            .Mx = true,
            .AdminUid = "0",
            .DefaultUid = "0",
            .BornDate = "2010-03-05 11:20:48",
        });
        m.AddSlave(s.Id());

        TDomainFetcher::TResult res;
        for (auto policy : {
                 TDomainFetcher::EPolicy::MasterOnly,
                 TDomainFetcher::EPolicy::MasterOrAlias,
                 TDomainFetcher::EPolicy::ReplaceAliasWithMaster,
             }) {
            res = f.Find("dveri.ru", policy);
            UNIT_ASSERT_C(res, policy);
            UNIT_ASSERT_EQUAL_C(m, *res.Domain, policy);
        }

        res = f.Find("zveri.ru", TDomainFetcher::EPolicy::MasterOnly);
        UNIT_ASSERT(!res);
        res = f.Find("zveri.ru", TDomainFetcher::EPolicy::MasterOrAlias);
        UNIT_ASSERT(res);
        UNIT_ASSERT_VALUES_EQUAL(s, *res.Domain);
        res = f.Find("zveri.ru", TDomainFetcher::EPolicy::ReplaceAliasWithMaster);
        UNIT_ASSERT(res);
        UNIT_ASSERT_EQUAL(m, *res.Domain);

        for (auto policy : {
                 TDomainFetcher::EPolicy::MasterOnly,
                 TDomainFetcher::EPolicy::MasterOrAlias,
                 TDomainFetcher::EPolicy::ReplaceAliasWithMaster,
             }) {
            res = f.FindById("15", policy);
            UNIT_ASSERT_C(res, policy);
            UNIT_ASSERT_EQUAL_C(m, *res.Domain, policy);
        }

        res = f.FindById("22", TDomainFetcher::EPolicy::MasterOnly);
        UNIT_ASSERT(!res);
        res = f.FindById("22", TDomainFetcher::EPolicy::MasterOrAlias);
        UNIT_ASSERT(res);
        UNIT_ASSERT_EQUAL(s, *res.Domain);
        res = f.FindById("22", TDomainFetcher::EPolicy::ReplaceAliasWithMaster);
        UNIT_ASSERT(res);
        UNIT_ASSERT_EQUAL(m, *res.Domain);
    }

    Y_UNIT_TEST(updateWithRows) {
        const NDbPool::TDestinationPtr dsn = NDbPool::TDestination::CreateSqlite(TestsDir() + "db/safeguarddb.sqlite3.sql");
        std::unique_ptr<NDbPool::TDbPool> pool = std::make_unique<NDbPool::TDbPool>(NDbPool::TDbPoolSettings{
            .Dsn = dsn,
        });

        class TFetTest: public TDomainFetcher {
        public:
            using TDomainFetcher::Cache_;
            using TDomainFetcher::SecondaryCache_;
            using TDomainFetcher::TDomainFetcher;
            using TDomainFetcher::UpdateWithRows;
        };

        TFetTest f(*pool, {});

        UNIT_ASSERT_VALUES_EQUAL(4, f.Cache_.Get()->List.size());
        UNIT_ASSERT_VALUES_EQUAL(4, f.SecondaryCache_->List.size());
        UNIT_ASSERT(f.Cache_.Get() != f.SecondaryCache_);

        NDbPool::TTable table;
        table.push_back(CreateRow("100016", "101", "0", "name1", "0", "0", ADMIN_UID, "70500", BORN_DATE, ""));
        table.push_back(CreateRow("100017", "102", "", NDbPool::TValue("", true), "", "", "", "", "", ""));
        table.push_back(CreateRow("100018", "103", "101", "name2", "0", "0", ADMIN_UID, "70500", BORN_DATE, ""));

        UNIT_ASSERT(f.UpdateWithRows(table));

        UNIT_ASSERT_VALUES_EQUAL(6, f.Cache_.Get()->List.size());
        UNIT_ASSERT_VALUES_EQUAL(6, f.SecondaryCache_->List.size());
        UNIT_ASSERT(f.Cache_.Get() != f.SecondaryCache_);

        const TDomain* m = f.SecondaryCache_->List.FindById("101");
        UNIT_ASSERT(m);
        UNIT_ASSERT_VALUES_EQUAL("name1", m->AsciiName());
        UNIT_ASSERT_VALUES_EQUAL("name1", m->UtfName());
        UNIT_ASSERT(!m->Ena());
        UNIT_ASSERT(!m->Mx());
        UNIT_ASSERT_VALUES_EQUAL("70500", m->DefaultUid());
        UNIT_ASSERT(m->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves({"103"}), m->Slaves());

        m = f.SecondaryCache_->List.FindByName("name1");
        UNIT_ASSERT(m);
        UNIT_ASSERT_VALUES_EQUAL("101", m->Id());
        UNIT_ASSERT(*f.Cache_.Get()->List.FindById("101") == *m);

        const TDomain* s = f.SecondaryCache_->List.FindById("103");
        UNIT_ASSERT(s);
        UNIT_ASSERT_VALUES_EQUAL("name2", s->AsciiName());
        UNIT_ASSERT_VALUES_EQUAL("name2", s->UtfName());
        UNIT_ASSERT(!s->Ena());
        UNIT_ASSERT(!s->Mx());
        UNIT_ASSERT_VALUES_EQUAL("70500", s->DefaultUid());
        UNIT_ASSERT(s->OrganizationName().empty());
        UNIT_ASSERT_VALUES_EQUAL(TDomain::TSlaves(), s->Slaves());

        s = f.SecondaryCache_->List.FindByName("name2");
        UNIT_ASSERT(s);
        UNIT_ASSERT_VALUES_EQUAL("103", s->Id());
        UNIT_ASSERT(*f.Cache_.Get()->List.FindById("103") == *s);
    }

    Y_UNIT_TEST(options) {
        TDomain::TOptions opt;

        opt = TDomainListBuilderBase::ReadDomainOptions(R"({"2":"Рога и копыта"})");
        UNIT_ASSERT_VALUES_EQUAL("Рога и копыта", opt.OrganizationName);
        UNIT_ASSERT_VALUES_EQUAL("", opt.OrganizationId);

        opt = TDomainListBuilderBase::ReadDomainOptions(R"({"organization_name":"Рога и копыта"})");
        UNIT_ASSERT_VALUES_EQUAL("Рога и копыта", opt.OrganizationName);
        UNIT_ASSERT_VALUES_EQUAL("", opt.OrganizationId);

        opt = TDomainListBuilderBase::ReadDomainOptions(R"({"2":"org_id:Рога и копыта"})");
        UNIT_ASSERT_VALUES_EQUAL("org_id:Рога и копыта", opt.OrganizationName);
        UNIT_ASSERT_VALUES_EQUAL("", opt.OrganizationId);

        opt = TDomainListBuilderBase::ReadDomainOptions(R"({"organization_name":"org_id:Рога и копыта"})");
        UNIT_ASSERT_VALUES_EQUAL("org_id:Рога и копыта", opt.OrganizationName);
        UNIT_ASSERT_VALUES_EQUAL("", opt.OrganizationId);

        opt = TDomainListBuilderBase::ReadDomainOptions(R"({"2":"org_id:123 и копыта"})");
        UNIT_ASSERT_VALUES_EQUAL("org_id:123 и копыта", opt.OrganizationName);
        UNIT_ASSERT_VALUES_EQUAL("", opt.OrganizationId);

        opt = TDomainListBuilderBase::ReadDomainOptions(R"({"organization_name":"org_id:123 и копыта"})");
        UNIT_ASSERT_VALUES_EQUAL("org_id:123 и копыта", opt.OrganizationName);
        UNIT_ASSERT_VALUES_EQUAL("", opt.OrganizationId);

        opt = TDomainListBuilderBase::ReadDomainOptions(R"({"2":"org_id:1234567891234567890"})");
        UNIT_ASSERT_VALUES_EQUAL("org_id:1234567891234567890", opt.OrganizationName);
        UNIT_ASSERT_VALUES_EQUAL("1234567891234567890", opt.OrganizationId);

        opt = TDomainListBuilderBase::ReadDomainOptions(R"({"organization_name":"org_id:1234567891234567890"})");
        UNIT_ASSERT_VALUES_EQUAL("org_id:1234567891234567890", opt.OrganizationName);
        UNIT_ASSERT_VALUES_EQUAL("1234567891234567890", opt.OrganizationId);

        opt = TDomainListBuilderBase::ReadDomainOptions(R"({"2":222})");
        UNIT_ASSERT_VALUES_EQUAL("", opt.OrganizationName);
        UNIT_ASSERT_VALUES_EQUAL("", opt.OrganizationId);

        opt = TDomainListBuilderBase::ReadDomainOptions(R"({"organization_name":[222,555]})");
        UNIT_ASSERT_VALUES_EQUAL("", opt.OrganizationName);
        UNIT_ASSERT_VALUES_EQUAL("", opt.OrganizationId);

        opt = TDomainListBuilderBase::ReadDomainOptions(R"({"organization_name":{"id":222, "name":"name"}})");
        UNIT_ASSERT_VALUES_EQUAL("", opt.OrganizationName);
        UNIT_ASSERT_VALUES_EQUAL("", opt.OrganizationId);

        opt = TDomainListBuilderBase::ReadDomainOptions(R"({"2""Рога и копыта"})");
        UNIT_ASSERT_VALUES_EQUAL("", opt.OrganizationName);
        UNIT_ASSERT_VALUES_EQUAL("", opt.OrganizationId);

        opt = TDomainListBuilderBase::ReadDomainOptions(R"({"2:"Рога и копыта"})");
        UNIT_ASSERT_VALUES_EQUAL("", opt.OrganizationName);
        UNIT_ASSERT_VALUES_EQUAL("", opt.OrganizationId);

        opt = TDomainListBuilderBase::ReadDomainOptions(R"({})");
        UNIT_ASSERT_VALUES_EQUAL("", opt.OrganizationName);
        UNIT_ASSERT_VALUES_EQUAL("", opt.OrganizationId);

        opt = TDomainListBuilderBase::ReadDomainOptions(R"()");
        UNIT_ASSERT_VALUES_EQUAL("", opt.OrganizationName);
        UNIT_ASSERT_VALUES_EQUAL("", opt.OrganizationId);

        opt = TDomainListBuilderBase::ReadDomainOptions(R"({"4":123456789})");
        UNIT_ASSERT_VALUES_EQUAL(TInstant::Seconds(123456789), opt.Glogout);

        opt = TDomainListBuilderBase::ReadDomainOptions(R"({"4":"123456789"})");
        UNIT_ASSERT_VALUES_EQUAL(TInstant(), opt.Glogout);
    }

    Y_UNIT_TEST(cacheOnDisk) {
        class TDomList: public TDomainFetcher {
        public:
            using TDomainFetcher::LoadFromProto;
            using TDomainFetcher::SerializeToProto;
        };

        TDomainCache cache;

        {
            TDomainListBuilder b(cache);
            b.SetLastEventId("100500");

            NDbPool::TRow row = CreateRow("300", "1129", "xn--__-6kcilbdeuclcee7aehgfmf1ay6wqa.xn--p1ai", "0", "0", ADMIN_UID, "70500", BORN_DATE, R"({})");
            UNIT_ASSERT_VALUES_EQUAL("300", b.AddRowAndGetDomId(std::move(row)));

            row = CreateRow("1129", "0", "okna.ru", "1", "0", ADMIN_UID, "70500", BORN_DATE, R"({"2" : "org_id:13"})");
            UNIT_ASSERT_VALUES_EQUAL("1129", b.AddRowAndGetDomId(std::move(row)));

            b.Finish();
        }

        UNIT_ASSERT_VALUES_EQUAL(cache, TDomList::LoadFromProto(TDomList::SerializeToProto(cache)));
    }
}

template <>
void Out<TDomainCache>(IOutputStream& o, const TDomainCache& value) {
    o << "last event id: " << value.LastEventId << Endl;

    domainlists_proto::Cache proto;
    value.List.Serialize(proto);
    o << "domain list: " << proto.DebugString() << Endl;
}

template <>
void Out<TDomain>(IOutputStream& o, const TDomain& value) {
    o << "id: " << value.Id() << Endl;
    o << "master: " << value.Master() << Endl;
    o << "asciiName: " << value.AsciiName() << Endl;
    o << "ena: " << value.Ena() << Endl;
    o << "mx: " << value.Mx() << Endl;
    o << "adminUid: " << value.AdminUid() << Endl;
    o << "defaultUid: " << value.DefaultUid() << Endl;
    o << "bornDate: " << value.BornDate() << Endl;
    o << "organizationName: " << value.OrganizationName() << Endl;
    o << "organizationId: " << value.OrganizationId() << Endl;
    o << "rawOptions: " << value.RawOptions() << Endl;
    for (const TString& s : value.Slaves()) {
        o << "slave: " << s << Endl;
    }
}
