#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/misc/attributes.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/dbfields_settings.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/login_status.h>
#include <passport/infra/daemons/blackbox/src/misc/stopwords_list.h>

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/destination.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NBb;
using namespace NPassport;

const TString tmpFileName("tmp_log_file");

Y_UNIT_TEST_SUITE(TPasspMiscUtilsTest) {
    Y_UNIT_TEST(AttributesSettings) {
        TAttributesSettings settings("req1,,req2  ", "b1, b1, b1,", "p1", "del1 del2,del3");

        UNIT_ASSERT(settings.RequiresGrant("req1"));
        UNIT_ASSERT(settings.RequiresGrant("req2"));
        UNIT_ASSERT(!settings.RequiresGrant(""));
        UNIT_ASSERT(!settings.RequiresGrant(" "));
        UNIT_ASSERT(!settings.RequiresGrant("req"));
        UNIT_ASSERT(!settings.RequiresGrant("b1"));
        UNIT_ASSERT(!settings.RequiresGrant("del1"));

        UNIT_ASSERT(!settings.IsBinary("1"));
        UNIT_ASSERT(settings.IsBinary("b1"));
        UNIT_ASSERT(!settings.IsBinary("b2"));
        UNIT_ASSERT(!settings.IsBinary("req1"));
        UNIT_ASSERT(!settings.IsBinary("del2"));

        UNIT_ASSERT(settings.RequiresGrantPhone("p1"));
        UNIT_ASSERT(!settings.RequiresGrantPhone("p2"));
        UNIT_ASSERT(!settings.RequiresGrantPhone(""));
        UNIT_ASSERT(!settings.RequiresGrantPhone("req1"));
        UNIT_ASSERT(!settings.RequiresGrantPhone("b1"));
        UNIT_ASSERT(!settings.RequiresGrantPhone("del1"));

        UNIT_ASSERT(settings.IsDeleted("del1"));
        UNIT_ASSERT(settings.IsDeleted("del2"));
        UNIT_ASSERT(settings.IsDeleted("del3"));
        UNIT_ASSERT(!settings.IsDeleted(""));
        UNIT_ASSERT(!settings.IsDeleted(","));
        UNIT_ASSERT(!settings.IsDeleted("req1"));
        UNIT_ASSERT(!settings.IsDeleted("b1"));
        UNIT_ASSERT(!settings.IsDeleted("p1"));
    }

    Y_UNIT_TEST(DbTypes) {
        UNIT_ASSERT(!TAttr::IsSynthetic(""));
        UNIT_ASSERT(!TAttr::IsSynthetic(","));
        UNIT_ASSERT(TAttr::IsSynthetic(TAttr::PHONE_NUMBER));
        UNIT_ASSERT(TAttr::IsSynthetic(TAttr::PHONE_CONFIRMATION_DATETIME));
        UNIT_ASSERT(!TAttr::IsSynthetic("5"));
        UNIT_ASSERT(!TAttr::IsSynthetic("111"));
        UNIT_ASSERT(TAttr::IsSynthetic("1111"));
        UNIT_ASSERT(TAttr::IsSynthetic("111111"));

        UNIT_ASSERT(!TPhoneAttr::IsSynthetic(""));
        UNIT_ASSERT(!TPhoneAttr::IsSynthetic("3"));
        UNIT_ASSERT(!TPhoneAttr::IsSynthetic("30"));
        UNIT_ASSERT(TPhoneAttr::IsSynthetic("101"));

        UNIT_ASSERT(!TEmailAttr::IsSynthetic("-"));
        UNIT_ASSERT(!TEmailAttr::IsSynthetic("2"));
        UNIT_ASSERT(!TEmailAttr::IsSynthetic("22"));
        UNIT_ASSERT(TEmailAttr::IsSynthetic("222"));

        UNIT_ASSERT(!TWebauthnAttr::IsSynthetic("-"));
        UNIT_ASSERT(!TWebauthnAttr::IsSynthetic("3"));
        UNIT_ASSERT(!TWebauthnAttr::IsSynthetic("33"));
        UNIT_ASSERT(TWebauthnAttr::IsSynthetic("333"));

        UNIT_ASSERT(!TOAuthTokenAttr::IsSynthetic("5"));
        UNIT_ASSERT(!TOAuthTokenAttr::IsSynthetic("55"));
        UNIT_ASSERT(!TOAuthTokenAttr::IsSynthetic("555"));
        UNIT_ASSERT(TOAuthTokenAttr::IsSynthetic("5555"));

        UNIT_ASSERT(!TOAuthClientAttr::IsSynthetic("7"));
        UNIT_ASSERT(!TOAuthClientAttr::IsSynthetic("77"));
        UNIT_ASSERT(!TOAuthClientAttr::IsSynthetic("777"));
        UNIT_ASSERT(TOAuthClientAttr::IsSynthetic("7777"));
    }

    Y_UNIT_TEST(DbfieldsSettings) {
        TDbFieldsSettings settings("pwd, phone,pwd");
        UNIT_ASSERT(settings.RequiresGrant("pwd"));
        UNIT_ASSERT(settings.RequiresGrant(" phone"));
        UNIT_ASSERT(settings.RequiresGrant("pwd"));
        UNIT_ASSERT(!settings.RequiresGrant("phone,pwd"));
        UNIT_ASSERT(!settings.RequiresGrant("name"));
        UNIT_ASSERT(!settings.RequiresGrant(" "));
        UNIT_ASSERT(!settings.RequiresGrant(","));
    }

    Y_UNIT_TEST(LoginStatusStatic) {
        UNIT_ASSERT_VALUES_EQUAL("VALID", TLoginStatus::LegacyStatusName(TLoginStatus::ELegacyStatus::VALID));
        UNIT_ASSERT_VALUES_EQUAL("DISABLED", TLoginStatus::LegacyStatusName(TLoginStatus::ELegacyStatus::DISABLED));
        UNIT_ASSERT_VALUES_EQUAL("INVALID", TLoginStatus::LegacyStatusName(TLoginStatus::ELegacyStatus::INVALID));
        UNIT_ASSERT_VALUES_EQUAL("SHOW_CAPTCHA", TLoginStatus::LegacyStatusName(TLoginStatus::ELegacyStatus::SHOW_CAPTCHA));
        UNIT_ASSERT_VALUES_EQUAL("EXPIRED", TLoginStatus::LegacyStatusName(TLoginStatus::ELegacyStatus::EXPIRED));
        UNIT_ASSERT_VALUES_EQUAL("SECOND_STEP_REQUIRED", TLoginStatus::LegacyStatusName(TLoginStatus::ELegacyStatus::SECOND_STEP_REQUIRED));

        UNIT_ASSERT_VALUES_EQUAL("UNKNOWN", TLoginStatus::AccountStatusName(TLoginStatus::acc_Unknown));
        UNIT_ASSERT_VALUES_EQUAL("NOT_FOUND", TLoginStatus::AccountStatusName(TLoginStatus::acc_Not_Found));
        UNIT_ASSERT_VALUES_EQUAL("DISABLED", TLoginStatus::AccountStatusName(TLoginStatus::acc_Disabled));
        UNIT_ASSERT_VALUES_EQUAL("VALID", TLoginStatus::AccountStatusName(TLoginStatus::acc_Valid));

        UNIT_ASSERT_VALUES_EQUAL("UNKNOWN", TLoginStatus::PasswordStatusName(TLoginStatus::pwd_Unknown));
        UNIT_ASSERT_VALUES_EQUAL("BAD", TLoginStatus::PasswordStatusName(TLoginStatus::pwd_Bad));
        UNIT_ASSERT_VALUES_EQUAL("VALID", TLoginStatus::PasswordStatusName(TLoginStatus::pwd_Valid));
        UNIT_ASSERT_VALUES_EQUAL("SECOND_STEP_REQUIRED", TLoginStatus::PasswordStatusName(TLoginStatus::pwd_Second_Step));
    }

    void LoginStatusAssignChecks(TLoginStatus& status) {
        status.Assign(TLoginStatus::TStatusPair(TLoginStatus::acc_Unknown, TLoginStatus::pwd_Valid), true, true, true, true);

        status.FormatRestrictComment("str1", "str2");
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::acc_Unknown, (int)status.AccountStatus());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::pwd_Valid, (int)status.PasswordStatus());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::ELegacyStatus::EXPIRED, status.GetLegacyStatus());
        UNIT_ASSERT_VALUES_EQUAL("Password expired", status.Comment());
        UNIT_ASSERT_VALUES_EQUAL((int)TAuthLog::BLOCKED, (int)status.AuthFlag());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::resist_Expired, (int)status.Policy());
        UNIT_ASSERT_VALUES_EQUAL(0, status.Delay());

        status.Assign(TLoginStatus::PAIR_NOT_FOUND, true, true, false, false);

        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::acc_Not_Found, (int)status.AccountStatus());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::pwd_Unknown, (int)status.PasswordStatus());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::ELegacyStatus::INVALID, status.GetLegacyStatus());
        UNIT_ASSERT_VALUES_EQUAL("Login not found; Too many login failures: str1; str2", status.Comment());
        UNIT_ASSERT_VALUES_EQUAL((int)TAuthLog::BAD, (int)status.AuthFlag());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::resist_Delay, (int)status.Policy());
        UNIT_ASSERT_VALUES_EQUAL(1000, status.Delay());

        status.FormatRestrictComment("str3", "str4");
        status.Assign(TLoginStatus::TStatusPair(TLoginStatus::acc_Disabled, TLoginStatus::pwd_Valid), true, true, false, false);

        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::acc_Disabled, (int)status.AccountStatus());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::pwd_Valid, (int)status.PasswordStatus());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::ELegacyStatus::DISABLED, status.GetLegacyStatus());
        UNIT_ASSERT_VALUES_EQUAL("Account disabled; Too many login failures: str3; str4", status.Comment());
        UNIT_ASSERT_VALUES_EQUAL((int)TAuthLog::DISABLED, (int)status.AuthFlag());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::resist_Delay, (int)status.Policy());
        UNIT_ASSERT_VALUES_EQUAL(1000, status.Delay());

        status.Assign(TLoginStatus::TStatusPair(TLoginStatus::acc_Disabled, TLoginStatus::pwd_Bad), true, true, true, false);

        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::acc_Disabled, (int)status.AccountStatus());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::pwd_Bad, (int)status.PasswordStatus());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::ELegacyStatus::SHOW_CAPTCHA, status.GetLegacyStatus());
        UNIT_ASSERT_VALUES_EQUAL("Bad password; account disabled; Too many login failures: str3; str4", status.Comment());
        UNIT_ASSERT_VALUES_EQUAL((int)TAuthLog::BAD, (int)status.AuthFlag());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::resist_Captcha, (int)status.Policy());
        UNIT_ASSERT_VALUES_EQUAL(1000, status.Delay());

        status.Assign(TLoginStatus::TStatusPair(TLoginStatus::acc_Disabled, TLoginStatus::pwd_Second_Step), true, true, false, false);

        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::acc_Disabled, (int)status.AccountStatus());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::pwd_Second_Step, (int)status.PasswordStatus());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::ELegacyStatus::INVALID, status.GetLegacyStatus());
        UNIT_ASSERT_VALUES_EQUAL("Bad password; account disabled; Too many login failures: str3; str4", status.Comment());
        UNIT_ASSERT_VALUES_EQUAL((int)TAuthLog::BAD, (int)status.AuthFlag());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::resist_Delay, (int)status.Policy());
        UNIT_ASSERT_VALUES_EQUAL(1000, status.Delay());

        status.SetScopeComment("no scopes, bro!");
        status.Assign(TLoginStatus::TStatusPair(TLoginStatus::acc_Valid, TLoginStatus::pwd_Second_Step), true, true, false, false);

        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::acc_Valid, (int)status.AccountStatus());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::pwd_Second_Step, (int)status.PasswordStatus());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::ELegacyStatus::SECOND_STEP_REQUIRED, status.GetLegacyStatus());
        UNIT_ASSERT_VALUES_EQUAL("Second step required; Too many login failures: str3; str4", status.Comment());
        UNIT_ASSERT_VALUES_EQUAL((int)TAuthLog::SECONDSTEP, (int)status.AuthFlag());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::resist_Delay, (int)status.Policy());
        UNIT_ASSERT_VALUES_EQUAL(1000, status.Delay());

        status.Assign(TLoginStatus::PAIR_BAD_PASSWORD, true, true, false, true);

        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::acc_Valid, (int)status.AccountStatus());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::pwd_Bad, (int)status.PasswordStatus());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::ELegacyStatus::INVALID, status.GetLegacyStatus());
        UNIT_ASSERT_VALUES_EQUAL("Bad password: no scopes, bro!; Too many login failures: str3; str4", status.Comment());
        UNIT_ASSERT_VALUES_EQUAL((int)TAuthLog::BAD, (int)status.AuthFlag());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::resist_Delay, (int)status.Policy());
        UNIT_ASSERT_VALUES_EQUAL(1000, status.Delay());
    }

    Y_UNIT_TEST(LoginStatus) {
        TLoginStatus status(5);

        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::acc_Unknown, (int)status.AccountStatus());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::pwd_Unknown, (int)status.PasswordStatus());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::ELegacyStatus::INVALID, status.GetLegacyStatus());
        UNIT_ASSERT_VALUES_EQUAL("unknown (internal error?)", status.Comment());
        UNIT_ASSERT_VALUES_EQUAL((int)TAuthLog::BAD, (int)status.AuthFlag());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::resist_None, (int)status.Policy());
        UNIT_ASSERT_VALUES_EQUAL(0, status.Delay());
        UNIT_ASSERT_VALUES_EQUAL(5, status.ApiVersion());
        UNIT_ASSERT_VALUES_EQUAL("INVALID", status.LegacyStatusAsString());

        UNIT_ASSERT(status == TLoginStatus::PAIR_UNKNOWN);
        UNIT_ASSERT(!(status == TLoginStatus::PAIR_VALID));

        LoginStatusAssignChecks(status);

        TLoginStatus status2(TLoginStatus::PAIR_VALID, "Custom login comment", 3);

        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::acc_Valid, (int)status2.AccountStatus());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::pwd_Valid, (int)status2.PasswordStatus());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::ELegacyStatus::VALID, status2.GetLegacyStatus());
        UNIT_ASSERT_VALUES_EQUAL("Custom login comment", status2.Comment());
        UNIT_ASSERT_VALUES_EQUAL((int)TAuthLog::OK, (int)status2.AuthFlag());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::resist_None, (int)status2.Policy());
        UNIT_ASSERT_VALUES_EQUAL(0, status2.Delay());
        UNIT_ASSERT_VALUES_EQUAL(3, status2.ApiVersion());
        UNIT_ASSERT_VALUES_EQUAL("VALID", status2.LegacyStatusAsString());

        UNIT_ASSERT(status2 == TLoginStatus::PAIR_VALID);
        UNIT_ASSERT(!(status2 == TLoginStatus::PAIR_NOT_FOUND));

        LoginStatusAssignChecks(status2);

        TLoginStatus status3(TLoginStatus::PAIR_BAD_PASSWORD, "Another login comment", 1);

        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::acc_Valid, (int)status3.AccountStatus());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::pwd_Bad, (int)status3.PasswordStatus());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::ELegacyStatus::INVALID, status3.GetLegacyStatus());
        UNIT_ASSERT_VALUES_EQUAL("Another login comment", status3.Comment());
        UNIT_ASSERT_VALUES_EQUAL((int)TAuthLog::BAD, (int)status3.AuthFlag());
        UNIT_ASSERT_VALUES_EQUAL((int)TLoginStatus::resist_None, (int)status3.Policy());
        UNIT_ASSERT_VALUES_EQUAL(0, status3.Delay());
        UNIT_ASSERT_VALUES_EQUAL(1, status3.ApiVersion());
        UNIT_ASSERT_VALUES_EQUAL("INVALID", status3.LegacyStatusAsString());

        UNIT_ASSERT(status3 == TLoginStatus::PAIR_BAD_PASSWORD);

        LoginStatusAssignChecks(status3);
    }

    Y_UNIT_TEST(StopWordList) {
        const NPassport::NDbPool::TDestinationPtr dsn = NPassport::NDbPool::TDestination::CreateSqlite(
            TestsDir() + "db/safeguarddb.sqlite3.sql");
        NPassport::NDbPool::TDbPool pool(NPassport::NDbPool::TDbPoolSettings{
            .Dsn = dsn,
        });

        TStopWordsList stoplist(pool, "support,admin,support,");

        UNIT_ASSERT(stoplist.IsStopWord("support"));
        UNIT_ASSERT(stoplist.IsStopWord("admin"));
        UNIT_ASSERT(stoplist.IsStopWord("volozh"));
        UNIT_ASSERT(stoplist.IsStopWord("porno"));
        UNIT_ASSERT(stoplist.IsStopWord("sex"));
        UNIT_ASSERT(!stoplist.IsStopWord(""));
        UNIT_ASSERT(!stoplist.IsStopWord(","));
        UNIT_ASSERT(!stoplist.IsStopWord("sporno"));
        UNIT_ASSERT(!stoplist.IsStopWord("essex"));
        UNIT_ASSERT(!stoplist.IsStopWord("porno,sex"));
        UNIT_ASSERT(!stoplist.IsStopWord("administrator"));

        UNIT_ASSERT(stoplist.IsStopSubstr("admin"));
        UNIT_ASSERT(stoplist.IsStopSubstr("administrator"));
        UNIT_ASSERT(stoplist.IsStopSubstr("superadministrator"));
        UNIT_ASSERT(stoplist.IsStopSubstr("brainfuck"));
        UNIT_ASSERT(stoplist.IsStopSubstr("uid-nahui"));
        UNIT_ASSERT(stoplist.IsStopSubstr("root"));
        UNIT_ASSERT(stoplist.IsStopSubstr("chrooter"));
        UNIT_ASSERT(!stoplist.IsStopSubstr(""));
        UNIT_ASSERT(!stoplist.IsStopSubstr(","));
        UNIT_ASSERT(!stoplist.IsStopSubstr("support"));
        UNIT_ASSERT(!stoplist.IsStopSubstr("volozh"));
    }

    Y_UNIT_TEST(Strings) {
        TString buf;
        TQueries::AppendPhoneOperationsMethod(buf, "1515151515");
        UNIT_ASSERT_VALUES_EQUAL(
            "SELECT CONCAT_WS(',',id,uid,phone_id,security_identity,type,UNIX_TIMESTAMP(started),"
            "UNIX_TIMESTAMP(finished),code_value,code_checks_count,code_send_count,UNIX_TIMESTAMP(code_last_sent),"
            "UNIX_TIMESTAMP(code_confirmed),UNIX_TIMESTAMP(password_verified),flags,phone_id2) AS value "
            "FROM phone_operations where finished<from_unixtime(1515151515)",
            buf);

        TQueries::AppendPhoneOperationsField(buf, "70500,70505,113000000000000");
        UNIT_ASSERT_VALUES_EQUAL(
            "SELECT CONCAT_WS(',',id,uid,phone_id,security_identity,type,UNIX_TIMESTAMP(started),"
            "UNIX_TIMESTAMP(finished),code_value,code_checks_count,code_send_count,UNIX_TIMESTAMP(code_last_sent),"
            "UNIX_TIMESTAMP(code_confirmed),UNIX_TIMESTAMP(password_verified),flags,phone_id2) AS value "
            "FROM phone_operations where finished<from_unixtime(1515151515)"
            "(SELECT uid,0 AS type,CONCAT_WS(',',id,uid,phone_id,security_identity,type,UNIX_TIMESTAMP(started),"
            "UNIX_TIMESTAMP(finished),code_value,code_checks_count,code_send_count,UNIX_TIMESTAMP(code_last_sent),"
            "UNIX_TIMESTAMP(code_confirmed),UNIX_TIMESTAMP(password_verified),flags,phone_id2) AS value,101 AS entity_type,id AS entity_id "
            "FROM phone_operations WHERE uid IN (70500,70505,113000000000000))",
            buf);
    }
}
