#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/password_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NBb;

Y_UNIT_TEST_SUITE(PasspBbPasswordChecker) {
    Y_UNIT_TEST(passwdMisc) {
        TPasswordChecker pc;

        TString pwd("The Magic Words are Squeamish Ossifrage");
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "", "", true));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "1", "", true));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "ab0d007f02ff0020ff09210000", "", true));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "ab0d007f02ff0020ff09210000", "1234", true));

        UNIT_ASSERT_EXCEPTION_CONTAINS(pc.PasswordMatches(pwd, "6:sfsgsfkjnfkgjdnfgkdnfkgjdnfgkjdnfgkdjfngdkfjgndfkjgndfksdg", "", true), TBlackboxError, "Argon is not configured");
        UNIT_ASSERT_EXCEPTION_CONTAINS(pc.PasswordMatches(pwd, "7:sfsgsfkjnfkgjdnfgkdnfkgjdnfgkjdnfgkdjfngdkfjgndfkjgndfksdg", "", true), TBlackboxError, "Argon is not configured");
        UNIT_ASSERT_EXCEPTION_CONTAINS(pc.MakeHash("qwerty", "1234", 6), TBlackboxError, "Argon is not configured");
        UNIT_ASSERT_EXCEPTION_CONTAINS(pc.MakeHash("qwerty", "1234", 7), TBlackboxError, "Argon is not configured");

        UNIT_ASSERT_EXCEPTION_CONTAINS(pc.MakeHash("qwerty", "", 0), TBlackboxError, "not supported hash version");
        UNIT_ASSERT_EXCEPTION_CONTAINS(pc.MakeHash("qwerty", "", 8), TBlackboxError, "not supported hash version");
    }

    Y_UNIT_TEST(passwdV1) {
        TPasswordChecker pc;

        TString pwd("The Magic Words are Squeamish Ossifrage");
        // supportOldHashes = true
        UNIT_ASSERT(pc.PasswordMatches(pwd, "1:$1$VaIoGQzx$NdU.PhQC81C.BgJX67.GS1", "", true));
        UNIT_ASSERT(pc.PasswordMatches(pwd, "1:$1$VaIoGQzx$NdU.PhQC81C.BgJX67.GS1", "1234", true));
        UNIT_ASSERT(pc.PasswordMatches(pwd, "1:$1$VaIoGQzx$NdU.PhQC81C.BgJX67.GS1", "yes", true));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "1:$2a$05$8ys8yeu2B0gAlBuIF/tMbu0qgBM1kpKklbselp9u7EN9kOj33wkJ6", "", true));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "1:$2a$05$8ys8yeu2B0gAlBuIF/tMbu0qgBM1kpKklbselp9u7EN9kOj33wkJ6", "1234", true));

        // supportOldHashes = false
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "1:$1$VaIoGQzx$NdU.PhQC81C.BgJX67.GS1", "", false));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "1:$1$VaIoGQzx$NdU.PhQC81C.BgJX67.GS1", "1234", false));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "1:$1$VaIoGQzx$NdU.PhQC81C.BgJX67.GS1", "yes", false));

        // makeHash not supported
        UNIT_ASSERT_EXCEPTION_CONTAINS(pc.MakeHash("qwerty", "", 1), TBlackboxError, "not supported hash version");
        UNIT_ASSERT_EXCEPTION_CONTAINS(pc.MakeHash("qwerty", "123456", 1), TBlackboxError, "not supported hash version");
    }

    Y_UNIT_TEST(passwdV2) {
        TPasswordChecker pc;

        UNIT_ASSERT(!pc.PasswordMatches("qwerty", "2:$2a$VaIoGQzx$NdU.PhQC81C.BgJX67.GS1", ""));

        UNIT_ASSERT_EXCEPTION_CONTAINS(pc.MakeHash("qwerty", "", 2), TBlackboxError, "not supported hash version");
        UNIT_ASSERT_EXCEPTION_CONTAINS(pc.MakeHash("qwerty", "123456", 2), TBlackboxError, "not supported hash version");
    }

    Y_UNIT_TEST(passwdV3) {
        TPasswordChecker pc;

        TString pwd("The Magic Words are Squeamish Ossifrage");
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "3:4b51e6026b2c0a88de94973508ceb7dd", "", true));
    }

    Y_UNIT_TEST(passwdV4) {
        TPasswordChecker pc;

        UNIT_ASSERT(!pc.PasswordMatches("qwerty", "4:$2a$VaIoGQzx$NdU.PhQC81C.BgJX67.GS1", ""));
        // makeHash not supported

        UNIT_ASSERT_EXCEPTION_CONTAINS(pc.MakeHash("qwerty", "", 4), TBlackboxError, "not supported hash version");
        UNIT_ASSERT_EXCEPTION_CONTAINS(pc.MakeHash("qwerty", "123456", 4), TBlackboxError, "not supported hash version");
    }

    Y_UNIT_TEST(passwdV5) {
        TPasswordChecker pc;

        TString pwd("The Magic Words are Squeamish Ossifrage");
        // supportOldHashes = true
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "5:$1$o0.yAs4f$F/jN830ckrg.goBdaD8mo0", "", true));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "5:$1$o0.yAs4f$F/jN830ckrg.goBdaD8mo0", "1234", true));
        UNIT_ASSERT(pc.PasswordMatches(pwd, "5:$1$o0.yAs4f$F/jN830ckrg.goBdaD8mo0", "123456", true));
        UNIT_ASSERT(pc.PasswordMatches(pwd, "5:$1$5KO4cKei$M5qoAOuuVX2TmJJ7qF4Ly0", "1234", true));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "5:$1$5KO4cKei$M5qoAOuuVX2TmJJ7qF4Ly0", "123456", true));

        // supportOldHashes = false
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "5:$1$o0.yAs4f$F/jN830ckrg.goBdaD8mo0", "123456", false));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "5:$1$5KO4cKei$M5qoAOuuVX2TmJJ7qF4Ly0", "1234", false));

        // makeHash not supported
        UNIT_ASSERT_EXCEPTION_CONTAINS(pc.MakeHash("qwerty", "", 5), TBlackboxError, "not supported hash version");
        UNIT_ASSERT_EXCEPTION_CONTAINS(pc.MakeHash("qwerty", "123456", 5), TBlackboxError, "not supported hash version");
    }

    Y_UNIT_TEST(passwdV6) {
        TPasswordChecker pc(NAuth::TKeyMap::CreateFromRawMap({{13, "mega_secret"}}, 13), 32, 1, 4096);

        TString pwd("The Magic Words are Squeamish Ossifrage");
        // supportOldHashes = false, doesn't apply
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6:13:1:4096:5DZY8IKQ:CwO5GoNRaB7rSf+ilMlXrA:lV4oSMLQgjchr02+LlCP0/tzYSUy0ZuSfzTPr9C9xsQ", ""));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6:13:1:4096:5DZY8IKQ:CwO5GoNRaB7rSf+ilMlXrA:lV4oSMLQgjchr02+LlCP0/tzYSUy0ZuSfzTPr9C9xsQ", "1234"));
        UNIT_ASSERT(pc.PasswordMatches(pwd, "6:13:1:4096:5DZY8IKQ:CwO5GoNRaB7rSf+ilMlXrA:lV4oSMLQgjchr02+LlCP0/tzYSUy0ZuSfzTPr9C9xsQ", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6:13:1:2048:5DZY8IKQ:CwO5GoNRaB7rSf+ilMlXrA:lV4oSMLQgjchr02+LlCP0/tzYSUy0ZuSfzTPr9C9xsQ", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6:13:2:4096:5DZY8IKQ:CwO5GoNRaB7rSf+ilMlXrA:lV4oSMLQgjchr02+LlCP0/tzYSUy0ZuSfzTPr9C9xsQ", "123456"));

        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6:", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6::1:4096:5DZY8IKQ:CwO5GoNRaB7rSf+ilMlXrA:lV4oSMLQgjchr02+LlCP0/tzYSUy0ZuSfzTPr9C9xsQ", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6:15:1:4096:5DZY8IKQ:CwO5GoNRaB7rSf+ilMlXrA:lV4oSMLQgjchr02+LlCP0/tzYSUy0ZuSfzTPr9C9xsQ", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6:13::4096:5DZY8IKQ:CwO5GoNRaB7rSf+ilMlXrA:lV4oSMLQgjchr02+LlCP0/tzYSUy0ZuSfzTPr9C9xsQ", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6:13:a:4096:5DZY8IKQ:CwO5GoNRaB7rSf+ilMlXrA:lV4oSMLQgjchr02+LlCP0/tzYSUy0ZuSfzTPr9C9xsQ", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6:13:0:4096:5DZY8IKQ:CwO5GoNRaB7rSf+ilMlXrA:lV4oSMLQgjchr02+LlCP0/tzYSUy0ZuSfzTPr9C9xsQ", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6:13:-1:4096:5DZY8IKQ:CwO5GoNRaB7rSf+ilMlXrA:lV4oSMLQgjchr02+LlCP0/tzYSUy0ZuSfzTPr9C9xsQ", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6:13:1::5DZY8IKQ:CwO5GoNRaB7rSf+ilMlXrA:lV4oSMLQgjchr02+LlCP0/tzYSUy0ZuSfzTPr9C9xsQ", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6:13:1:-1:5DZY8IKQ:CwO5GoNRaB7rSf+ilMlXrA:lV4oSMLQgjchr02+LlCP0/tzYSUy0ZuSfzTPr9C9xsQ", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6:13:1:1:5DZY8IKQ:CwO5GoNRaB7rSf+ilMlXrA:lV4oSMLQgjchr02+LlCP0/tzYSUy0ZuSfzTPr9C9xsQ", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6:13:1:a:5DZY8IKQ:CwO5GoNRaB7rSf+ilMlXrA:lV4oSMLQgjchr02+LlCP0/tzYSUy0ZuSfzTPr9C9xsQ", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6:13:1:4096::CwO5GoNRaB7rSf+ilMlXrA:lV4oSMLQgjchr02+LlCP0/tzYSUy0ZuSfzTPr9C9xsQ", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6:13:1:4096:5DZY8IKQ::lV4oSMLQgjchr02+LlCP0/tzYSUy0ZuSfzTPr9C9xsQ", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6:13:1:4096:5DZY8IKQ:CwO5GoNRaB7rSf+ilMlXrA:", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6:13:1:4096:5DZY8IKQ:CwO5GoNRaB7rSf+ilMlXrA:lV4oSMLQgjchr02+LlCP0/tzYSUy0ZuSfzTPr9C9xsQ:end", "123456"));

        UNIT_ASSERT(pc.PasswordMatches(pwd, "6:13:1:4096:LYjIWKnQ:U7XgFzJp6wQh0Kv77SFl+A:cY6DKK357iblggePdNR1QY+gxpDBZSRJaObQk6AhjD0", "1234"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6:13:1:4096:LYjIWKnQ:U7XgFzJp6wQh0Kv77SFl+A:cY6DKK357iblggePdNR1QY+gxpDBZSRJaObQk6AhjD0", "123456"));
        UNIT_ASSERT(pc.PasswordMatches(pwd, "6:13:2:2048:0kmnZRW8:/oih5icayXywVL13582dCg:KvKHQcDPbl6507YFXdP82MGdheTwXsGgs8bBVaLsK1A", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "6:13:2:2048:0kmnZRW8:/oih5icayXywVL13582dCg:KvKHQcDPbl6507YFXdP82MGdheTwXsGgs8bBVaLsK1A", "1234"));

        UNIT_ASSERT_EXCEPTION_CONTAINS(pc.MakeHash("qwerty", "", 6), TBlackboxError, "no UID for version=6 hash");

        TString hash = pc.MakeHash(pwd, "123456", 6);
        UNIT_ASSERT_VALUES_EQUAL("6:13:1:4096:", hash.substr(0, 12));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, hash, ""));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, hash, "1234"));
        UNIT_ASSERT(pc.PasswordMatches(pwd, hash, "123456"));

        hash = pc.ConvertMD5ToArgon("$1$VaIoGQzx$NdU.PhQC81C.BgJX67.GS1", "1234");
        UNIT_ASSERT_VALUES_EQUAL("6:13:1:4096:", hash.substr(0, 12));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, hash, ""));
        UNIT_ASSERT(pc.PasswordMatches(pwd, hash, "1234"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, hash, "123456"));

        UNIT_ASSERT_EXCEPTION_CONTAINS(pc.ConvertMD5ToArgon("qwerty", ""), TBlackboxError, "no UID for version=6 hash");
    }

    Y_UNIT_TEST(passwdV7) {
        TPasswordChecker pc(NAuth::TKeyMap::CreateFromRawMap({{17, "another_mega_secret"}}, 17), 32, 2, 1024);

        TString pwd("The Magic Words are Squeamish Ossifrage");
        // supportOldHashes = false, doesn't apply
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "7:17:2:1024:UMFbnyYKp0hy51odS92ULQ:5lpj/WvBPW2U3I9N6BLpusb/h6LnuzQ0ZKkZV/SS6sc", ""));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "7:17:2:1024:UMFbnyYKp0hy51odS92ULQ:5lpj/WvBPW2U3I9N6BLpusb/h6LnuzQ0ZKkZV/SS6sc", "1234"));
        UNIT_ASSERT(pc.PasswordMatches(pwd, "7:17:2:1024:UMFbnyYKp0hy51odS92ULQ:5lpj/WvBPW2U3I9N6BLpusb/h6LnuzQ0ZKkZV/SS6sc", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "7:17:1:1024:UMFbnyYKp0hy51odS92ULQ:5lpj/WvBPW2U3I9N6BLpusb/h6LnuzQ0ZKkZV/SS6sc", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "7:17:2:2048:UMFbnyYKp0hy51odS92ULQ:5lpj/WvBPW2U3I9N6BLpusb/h6LnuzQ0ZKkZV/SS6sc", "123456"));

        UNIT_ASSERT(!pc.PasswordMatches(pwd, "7:", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "7::2:1024:UMFbnyYKp0hy51odS92ULQ:5lpj/WvBPW2U3I9N6BLpusb/h6LnuzQ0ZKkZV/SS6sc", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "7:13:2:1024:UMFbnyYKp0hy51odS92ULQ:5lpj/WvBPW2U3I9N6BLpusb/h6LnuzQ0ZKkZV/SS6sc", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "7:17::1024:UMFbnyYKp0hy51odS92ULQ:5lpj/WvBPW2U3I9N6BLpusb/h6LnuzQ0ZKkZV/SS6sc", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "7:17:O:1024:UMFbnyYKp0hy51odS92ULQ:5lpj/WvBPW2U3I9N6BLpusb/h6LnuzQ0ZKkZV/SS6sc", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "7:17:0:1024:UMFbnyYKp0hy51odS92ULQ:5lpj/WvBPW2U3I9N6BLpusb/h6LnuzQ0ZKkZV/SS6sc", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "7:17:-1:1024:UMFbnyYKp0hy51odS92ULQ:5lpj/WvBPW2U3I9N6BLpusb/h6LnuzQ0ZKkZV/SS6sc", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "7:17:2::UMFbnyYKp0hy51odS92ULQ:5lpj/WvBPW2U3I9N6BLpusb/h6LnuzQ0ZKkZV/SS6sc", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "7:17:2:2:UMFbnyYKp0hy51odS92ULQ:5lpj/WvBPW2U3I9N6BLpusb/h6LnuzQ0ZKkZV/SS6sc", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "7:17:2:-1:UMFbnyYKp0hy51odS92ULQ:5lpj/WvBPW2U3I9N6BLpusb/h6LnuzQ0ZKkZV/SS6sc", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "7:17:2:-:UMFbnyYKp0hy51odS92ULQ:5lpj/WvBPW2U3I9N6BLpusb/h6LnuzQ0ZKkZV/SS6sc", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "7:17:2:1024::5lpj/WvBPW2U3I9N6BLpusb/h6LnuzQ0ZKkZV/SS6sc", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "7:17:2:1024:UMFbnyYKp0hy51odS92ULQ:", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "7:17:2:1024:UMFbnyYKp0hy51odS92ULQ:5lpj/WvBPW2U3I9N6BLpusb/h6LnuzQ0ZKkZV/SS6sc:more", "123456"));

        UNIT_ASSERT(pc.PasswordMatches(pwd, "7:17:2:1024:igjOGK/7qmIjPckwz3NmbA:PCMlQQvLYv11w170qlL2rf/+E9lLCVtNf3x89kmU7dc", "1234"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "7:17:2:1024:igjOGK/7qmIjPckwz3NmbA:PCMlQQvLYv11w170qlL2rf/+E9lLCVtNf3x89kmU7dc", "123456"));
        UNIT_ASSERT(pc.PasswordMatches(pwd, "7:17:2:1024:7nYXpksexw+SoAKFtf1uqg:P5nbB1HMbzDPqerNMLpPwU4YOqV35ZXtjtK1N4YXwuo", "123456"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, "7:17:2:1024:7nYXpksexw+SoAKFtf1uqg:P5nbB1HMbzDPqerNMLpPwU4YOqV35ZXtjtK1N4YXwuo", "1234"));

        UNIT_ASSERT_EXCEPTION_CONTAINS(pc.MakeHash("qwerty", "", 7), TBlackboxError, "no UID for version=7 hash");

        TString hash = pc.MakeHash(pwd, "123456", 7);
        UNIT_ASSERT_VALUES_EQUAL("7:17:2:1024:", hash.substr(0, 12));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, hash, ""));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, hash, "1234"));
        UNIT_ASSERT(pc.PasswordMatches(pwd, hash, "123456"));

        hash = pc.ConvertRawMD5ToArgon("4b51e6026b2c0a88de94973508ceb7dd", "1234");
        UNIT_ASSERT_VALUES_EQUAL("7:17:2:1024:", hash.substr(0, 12));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, hash, ""));
        UNIT_ASSERT(pc.PasswordMatches(pwd, hash, "1234"));
        UNIT_ASSERT(!pc.PasswordMatches(pwd, hash, "123456"));

        UNIT_ASSERT_EXCEPTION_CONTAINS(pc.ConvertRawMD5ToArgon("qwerty", ""), TBlackboxError, "no UID for version=7 hash");
    }
}
