#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/perimeter.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NBb;

Y_UNIT_TEST_SUITE(Perimeter) {
    class TTestPerimeter: public TPerimeterWrapper {
    public:
        using TPerimeterWrapper::ParseResponse;
    };

    Y_UNIT_TEST(badResponse) {
        TString reqId;
        TString msg;
        std::optional<TString> allowedSecondSteps;
        bool passwordChangeRequired;

        // response is not object
        UNIT_ASSERT_EXCEPTION_CONTAINS(TTestPerimeter::ParseResponse(R"(["status": "ok", "description": "MOTP user not found; LDAP+TOTP password correct", "request_id": "7d69a3c457b0fbf1f3747e44b78e79b0"])",
                                                                     reqId,
                                                                     msg,
                                                                     allowedSecondSteps,
                                                                     passwordChangeRequired),
                                       TBlackboxError,
                                       "Internal error: Perimeter response is not JSON object");

        // bad status
        UNIT_ASSERT_EXCEPTION_CONTAINS(TTestPerimeter::ParseResponse(R"({"status": 5500, "description": "MOTP user not found; LDAP+TOTP password correct", "request_id": "7d69a3c457b0fbf1f3747e44b78e79b0"})",
                                                                     reqId,
                                                                     msg,
                                                                     allowedSecondSteps,
                                                                     passwordChangeRequired),
                                       TBlackboxError,
                                       "Internal error: invalid Perimeter response contents");

        // bad description
        UNIT_ASSERT_EXCEPTION_CONTAINS(TTestPerimeter::ParseResponse(R"({"status": "ok", "description": 1050, "request_id": "7d69a3c457b0fbf1f3747e44b78e79b0"})",
                                                                     reqId,
                                                                     msg,
                                                                     allowedSecondSteps,
                                                                     passwordChangeRequired),
                                       TBlackboxError,
                                       "Internal error: invalid Perimeter response contents");

        // bad request_id
        UNIT_ASSERT_EXCEPTION_CONTAINS(TTestPerimeter::ParseResponse(R"({"status": "ok", "description": "MOTP user not found; LDAP+TOTP password correct", "request_id": 100500})",
                                                                     reqId,
                                                                     msg,
                                                                     allowedSecondSteps,
                                                                     passwordChangeRequired),
                                       TBlackboxError,
                                       "Internal error: invalid Perimeter response contents");

        // unknown status
        UNIT_ASSERT_EXCEPTION_CONTAINS(TTestPerimeter::ParseResponse(R"({"status": "kek", "second_steps": ["totp", "kek"], "description": "MOTP user not found; LDAP+TOTP password correct", "request_id": "7d69a3c457b0fbf1f3747e44b78e79b0"})",
                                                                     reqId,
                                                                     msg,
                                                                     allowedSecondSteps,
                                                                     passwordChangeRequired),
                                       TBlackboxError,
                                       "Internal error: perimeter request failed with status kek");
    }

    Y_UNIT_TEST(ok) {
        TString response = R"({"status": "ok", "second_steps": ["totp", "kek"], "description": "MOTP user not found; LDAP+TOTP password correct", "request_id": "7d69a3c457b0fbf1f3747e44b78e79b0"})";

        TString reqId;
        TString msg;
        std::optional<TString> allowedSecondSteps;
        bool passwordChangeRequired;

        UNIT_ASSERT_VALUES_EQUAL(TPerimeterWrapper::LoginOk,
                                 TTestPerimeter::ParseResponse(response, reqId, msg, allowedSecondSteps, passwordChangeRequired));
        UNIT_ASSERT_VALUES_EQUAL(reqId, "7d69a3c457b0fbf1f3747e44b78e79b0");
        UNIT_ASSERT_VALUES_EQUAL(msg, "MOTP user not found; LDAP+TOTP password correct");
        UNIT_ASSERT(!allowedSecondSteps);
    }

    Y_UNIT_TEST(rejected) {
        TString response = R"({"status": "password_invalid", "second_steps": ["totp", "kek"], "description": "MOTP user not found; LDAP+TOTP password correct", "request_id": "7d69a3c457b0fbf1f3747e44b78e79b0"})";

        TString reqId;
        TString msg;
        std::optional<TString> allowedSecondSteps;
        bool passwordChangeRequired;

        UNIT_ASSERT_VALUES_EQUAL(TPerimeterWrapper::Rejected,
                                 TTestPerimeter::ParseResponse(response, reqId, msg, allowedSecondSteps, passwordChangeRequired));
        UNIT_ASSERT_VALUES_EQUAL(reqId, "7d69a3c457b0fbf1f3747e44b78e79b0");
        UNIT_ASSERT_VALUES_EQUAL(msg, "MOTP user not found; LDAP+TOTP password correct");
        UNIT_ASSERT(!allowedSecondSteps);
    }

    Y_UNIT_TEST(secondStep_twoValues) {
        TString response = R"({"status": "second_step_required", "second_steps": ["totp", "kek"], "description": "MOTP user not found; LDAP+TOTP password correct", "request_id": "7d69a3c457b0fbf1f3747e44b78e79b0"})";

        TString reqId;
        TString msg;
        std::optional<TString> allowedSecondSteps;
        bool passwordChangeRequired;

        UNIT_ASSERT_VALUES_EQUAL(TPerimeterWrapper::SecondStep,
                                 TTestPerimeter::ParseResponse(response, reqId, msg, allowedSecondSteps, passwordChangeRequired));
        UNIT_ASSERT_VALUES_EQUAL(reqId, "7d69a3c457b0fbf1f3747e44b78e79b0");
        UNIT_ASSERT_VALUES_EQUAL(msg, "MOTP user not found; LDAP+TOTP password correct");
        UNIT_ASSERT(allowedSecondSteps);
        UNIT_ASSERT_VALUES_EQUAL(*allowedSecondSteps, "totp,kek");
    }

    Y_UNIT_TEST(secondStep_oneValue) {
        TString response = R"({"status": "second_step_required", "second_steps": ["totp"], "description": "MOTP user not found; LDAP+TOTP password correct", "request_id": "7d69a3c457b0fbf1f3747e44b78e79b0"})";

        TString reqId;
        TString msg;
        std::optional<TString> allowedSecondSteps;
        bool passwordChangeRequired;

        UNIT_ASSERT_VALUES_EQUAL(TPerimeterWrapper::SecondStep,
                                 TTestPerimeter::ParseResponse(response, reqId, msg, allowedSecondSteps, passwordChangeRequired));
        UNIT_ASSERT_VALUES_EQUAL(reqId, "7d69a3c457b0fbf1f3747e44b78e79b0");
        UNIT_ASSERT_VALUES_EQUAL(msg, "MOTP user not found; LDAP+TOTP password correct");
        UNIT_ASSERT(allowedSecondSteps);
        UNIT_ASSERT_VALUES_EQUAL(*allowedSecondSteps, "totp");
    }

    Y_UNIT_TEST(secondStep_noValue) {
        TString response = R"({"status": "second_step_required", "second_steps": [], "description": "MOTP user not found; LDAP+TOTP password correct", "request_id": "7d69a3c457b0fbf1f3747e44b78e79b0"})";

        TString reqId;
        TString msg;
        std::optional<TString> allowedSecondSteps;
        bool passwordChangeRequired;

        UNIT_ASSERT_VALUES_EQUAL(TPerimeterWrapper::SecondStep,
                                 TTestPerimeter::ParseResponse(response, reqId, msg, allowedSecondSteps, passwordChangeRequired));
        UNIT_ASSERT_VALUES_EQUAL(reqId, "7d69a3c457b0fbf1f3747e44b78e79b0");
        UNIT_ASSERT_VALUES_EQUAL(msg, "MOTP user not found; LDAP+TOTP password correct");
        UNIT_ASSERT(allowedSecondSteps);
        UNIT_ASSERT_VALUES_EQUAL(*allowedSecondSteps, "");
    }

    Y_UNIT_TEST(secondStep_errs) {
        TString reqId;
        TString msg;
        std::optional<TString> allowedSecondSteps;
        bool passwordChangeRequired;

        // no second_steps
        UNIT_ASSERT_EXCEPTION_CONTAINS(TTestPerimeter::ParseResponse(R"({"status": "second_step_required", "description": "MOTP user not found; LDAP+TOTP password correct", "request_id": "7d69a3c457b0fbf1f3747e44b78e79b0"})",
                                                                     reqId,
                                                                     msg,
                                                                     allowedSecondSteps,
                                                                     passwordChangeRequired),
                                       TBlackboxError,
                                       "Internal error: invalid Perimeter response contents");

        // second_steps - array
        UNIT_ASSERT_EXCEPTION_CONTAINS(TTestPerimeter::ParseResponse(R"({"status": "second_step_required", "second_steps": "totp", "description": "MOTP user not found; LDAP+TOTP password correct", "request_id": "7d69a3c457b0fbf1f3747e44b78e79b0"})",
                                                                     reqId,
                                                                     msg,
                                                                     allowedSecondSteps,
                                                                     passwordChangeRequired),
                                       TBlackboxError,
                                       "Internal error: invalid Perimeter response contents");

        // string in array
        UNIT_ASSERT_EXCEPTION_CONTAINS(TTestPerimeter::ParseResponse(R"({"status": "second_step_required", "second_steps": ["totp", 100500], "description": "MOTP user not found; LDAP+TOTP password correct", "request_id": "7d69a3c457b0fbf1f3747e44b78e79b0"})",
                                                                     reqId,
                                                                     msg,
                                                                     allowedSecondSteps,
                                                                     passwordChangeRequired),
                                       TBlackboxError,
                                       "Internal error: invalid Perimeter response contents");
    }

    Y_UNIT_TEST(PasswordChangeRequired_true) {
        TString response = R"({"status": "second_step_required", "second_steps": ["totp", "kek"], "password_change_required": true, "description": "second_step_required", "request_id": "7d69a3c457b0fbf1f3747e44b78e79b0"})";

        TString reqId;
        TString msg;
        std::optional<TString> allowedSecondSteps;
        bool passwordChangeRequired;

        UNIT_ASSERT_VALUES_EQUAL(TPerimeterWrapper::SecondStep,
                                 TTestPerimeter::ParseResponse(response, reqId, msg, allowedSecondSteps, passwordChangeRequired));
        UNIT_ASSERT_VALUES_EQUAL(reqId, "7d69a3c457b0fbf1f3747e44b78e79b0");
        UNIT_ASSERT_VALUES_EQUAL(msg, "second_step_required");
        UNIT_ASSERT(allowedSecondSteps);
        UNIT_ASSERT_VALUES_EQUAL(*allowedSecondSteps, "totp,kek");
        UNIT_ASSERT(passwordChangeRequired);
    }

    Y_UNIT_TEST(PasswordChangeRequired_false) {
        TString response = R"({"status": "ok", "password_change_required": true, "description": "Password expired", "request_id": "7d69a3c457b0fbf1f3747e44b78e79b0"})";

        TString reqId;
        TString msg;
        std::optional<TString> allowedSecondSteps;
        bool passwordChangeRequired;

        UNIT_ASSERT_VALUES_EQUAL(TPerimeterWrapper::LoginOk,
                                 TTestPerimeter::ParseResponse(response, reqId, msg, allowedSecondSteps, passwordChangeRequired));
        UNIT_ASSERT_VALUES_EQUAL(reqId, "7d69a3c457b0fbf1f3747e44b78e79b0");
        UNIT_ASSERT_VALUES_EQUAL(msg, "Password expired");
        UNIT_ASSERT(passwordChangeRequired);
    }

    Y_UNIT_TEST(PasswordChangeRequired_no_value) {
        TString response = R"({"status": "ok", "description": "ok", "request_id": "7d69a3c457b0fbf1f3747e44b78e79b0"})";

        TString reqId;
        TString msg;
        std::optional<TString> allowedSecondSteps;
        bool passwordChangeRequired;

        UNIT_ASSERT_VALUES_EQUAL(TPerimeterWrapper::LoginOk,
                                 TTestPerimeter::ParseResponse(response, reqId, msg, allowedSecondSteps, passwordChangeRequired));
        UNIT_ASSERT_VALUES_EQUAL(reqId, "7d69a3c457b0fbf1f3747e44b78e79b0");
        UNIT_ASSERT_VALUES_EQUAL(msg, "ok");
        UNIT_ASSERT(!passwordChangeRequired);
    }

    Y_UNIT_TEST(PasswordChangeRequired_invalid) {
        TString response = R"({"status": "password_invalid", "password_change_required": true, "description": "password_invalid", "request_id": "7d69a3c457b0fbf1f3747e44b78e79b0"})";

        TString reqId;
        TString msg;
        std::optional<TString> allowedSecondSteps;
        bool passwordChangeRequired;

        UNIT_ASSERT_VALUES_EQUAL(TPerimeterWrapper::Rejected,
                                 TTestPerimeter::ParseResponse(response, reqId, msg, allowedSecondSteps, passwordChangeRequired));
        UNIT_ASSERT_VALUES_EQUAL(reqId, "7d69a3c457b0fbf1f3747e44b78e79b0");
        UNIT_ASSERT_VALUES_EQUAL(msg, "password_invalid");
        UNIT_ASSERT(!passwordChangeRequired);
    }
}
