#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/misc/session_utils.h>

#include <passport/infra/libs/cpp/auth_core/session.h>
#include <passport/infra/libs/cpp/auth_core/sessionparser.h>
#include <passport/infra/libs/cpp/auth_core/sessionsigner.h>
#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/destination.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NBb;
using namespace NPassport::NAuth;

Y_UNIT_TEST_SUITE(PasspMiscSessionUtils) {
    Y_UNIT_TEST(session2UserStatus) {
        UNIT_ASSERT_VALUES_EQUAL(TSessionidProcessor::ESessionStatus::VALID,
                                 TSessionUtils::Session2UserStatus(TSession::VALID));
        UNIT_ASSERT_VALUES_EQUAL(TSessionidProcessor::ESessionStatus::NEED_RESET,
                                 TSessionUtils::Session2UserStatus(TSession::NEED_RESET));
        UNIT_ASSERT_VALUES_EQUAL(TSessionidProcessor::ESessionStatus::EXPIRED,
                                 TSessionUtils::Session2UserStatus(TSession::EXPIRED));
        UNIT_ASSERT_VALUES_EQUAL(TSessionidProcessor::ESessionStatus::INVALID,
                                 TSessionUtils::Session2UserStatus(TSession::SIGN_BROKEN));
        UNIT_ASSERT_VALUES_EQUAL(TSessionidProcessor::ESessionStatus::INVALID,
                                 TSessionUtils::Session2UserStatus(TSession::NO_COOKIE));
        UNIT_ASSERT_VALUES_EQUAL(TSessionidProcessor::ESessionStatus::INVALID,
                                 TSessionUtils::Session2UserStatus(TSession::CANT_CHECK));
        UNIT_ASSERT_VALUES_EQUAL(TSessionidProcessor::ESessionStatus::NOAUTH,
                                 TSessionUtils::Session2UserStatus(TSession::NOAUTH));
        UNIT_ASSERT_VALUES_EQUAL(TSessionidProcessor::ESessionStatus::DISABLED,
                                 TSessionUtils::Session2UserStatus(TSession::DISABLED));

    }

    Y_UNIT_TEST(validCookieStatus) {
        UNIT_ASSERT(TSessionUtils::ValidCookieStatus(TSessionidProcessor::ESessionStatus::VALID));
        UNIT_ASSERT(TSessionUtils::ValidCookieStatus(TSessionidProcessor::ESessionStatus::NEED_RESET));
        UNIT_ASSERT(!TSessionUtils::ValidCookieStatus(TSessionidProcessor::ESessionStatus::EXPIRED));
        UNIT_ASSERT(!TSessionUtils::ValidCookieStatus(TSessionidProcessor::ESessionStatus::NOAUTH));
        UNIT_ASSERT(!TSessionUtils::ValidCookieStatus(TSessionidProcessor::ESessionStatus::DISABLED));
        UNIT_ASSERT(!TSessionUtils::ValidCookieStatus(TSessionidProcessor::ESessionStatus::INVALID));
        UNIT_ASSERT(!TSessionUtils::ValidCookieStatus((TSessionidProcessor::ESessionStatus)6));
    }

    Y_UNIT_TEST(statusName) {
        UNIT_ASSERT_VALUES_EQUAL("VALID", TSessionUtils::StatusName(TSessionidProcessor::ESessionStatus::VALID));
        UNIT_ASSERT_VALUES_EQUAL("NEED_RESET", TSessionUtils::StatusName(TSessionidProcessor::ESessionStatus::NEED_RESET));
        UNIT_ASSERT_VALUES_EQUAL("EXPIRED", TSessionUtils::StatusName(TSessionidProcessor::ESessionStatus::EXPIRED));
        UNIT_ASSERT_VALUES_EQUAL("NOAUTH", TSessionUtils::StatusName(TSessionidProcessor::ESessionStatus::NOAUTH));
        UNIT_ASSERT_VALUES_EQUAL("DISABLED", TSessionUtils::StatusName(TSessionidProcessor::ESessionStatus::DISABLED));
        UNIT_ASSERT_VALUES_EQUAL("INVALID", TSessionUtils::StatusName(TSessionidProcessor::ESessionStatus::INVALID));
        UNIT_ASSERT_VALUES_EQUAL("UNKNOWN", TSessionUtils::StatusName({}));
    }

    Y_UNIT_TEST(dropExternalAuth) {
        const NPassport::NDbPool::TDestinationPtr dsn = NPassport::NDbPool::TDestination::CreateSqlite(
            TestsDir() + "db/safeguarddb.sqlite3.sql");
        NPassport::NDbPool::TDbPool pool(NPassport::NDbPool::TDbPoolSettings{
            .Dsn = dsn,
        });
        TSessionSigner sessSigner(pool);
        UNIT_ASSERT_NO_EXCEPTION(sessSigner.AddKeyspace("yandex.ru"));
        TSessionParser sessParser(sessSigner, EEnvironmentType::Testing);

        // Session has 3 users, trying to drop external auth for each of them:
        // 1. 70500, user with no int/ext flags (i.e. no external auth to drop) but with ext_glogout
        // 2. 70501, user with only ext flag (i.e. should be deleted completely from both cookies)
        // 3. 70502, user with int+ext flags (i.e. only ext flag should be deleted)
        TSession sess = sessParser.Create("70500", "0", "yandex.ru");
        sess.SetExtGlogouted(true, 0);
        sess.AddUser("70501");
        sess.SetExtGlogouted(true, 1);
        sess.SetExternalAuth(true, 1);
        sess.AddUser("70502");
        sess.SetExtGlogouted(true, 2);
        sess.SetInternalAuth(true, 2);
        sess.SetExternalAuth(true, 2);

        UNIT_ASSERT(!TSessionUtils::DropExternalAuth(sess, -1));
        UNIT_ASSERT(!TSessionUtils::DropExternalAuth(sess, 10));

        UNIT_ASSERT_VALUES_EQUAL(3, sess.UserCount());
        UNIT_ASSERT_VALUES_EQUAL("70500", sess.Uid(0));
        UNIT_ASSERT_VALUES_EQUAL(true, sess.IsExtGlogouted(0));
        UNIT_ASSERT_VALUES_EQUAL(false, sess.IsExternalAuth(0));
        UNIT_ASSERT_VALUES_EQUAL(false, sess.IsInternalAuth(0));
        UNIT_ASSERT_VALUES_EQUAL("70501", sess.Uid(1));
        UNIT_ASSERT_VALUES_EQUAL(true, sess.IsExtGlogouted(1));
        UNIT_ASSERT_VALUES_EQUAL(true, sess.IsExternalAuth(1));
        UNIT_ASSERT_VALUES_EQUAL(false, sess.IsInternalAuth(1));
        UNIT_ASSERT_VALUES_EQUAL("70502", sess.Uid(2));
        UNIT_ASSERT_VALUES_EQUAL(true, sess.IsExtGlogouted(2));
        UNIT_ASSERT_VALUES_EQUAL(true, sess.IsExternalAuth(2));
        UNIT_ASSERT_VALUES_EQUAL(true, sess.IsInternalAuth(2));

        UNIT_ASSERT(!TSessionUtils::DropExternalAuth(sess, 0));

        UNIT_ASSERT_VALUES_EQUAL(3, sess.UserCount());
        UNIT_ASSERT_VALUES_EQUAL("70500", sess.Uid(0));
        UNIT_ASSERT_VALUES_EQUAL(false, sess.IsExtGlogouted(0));
        UNIT_ASSERT_VALUES_EQUAL(false, sess.IsExternalAuth(0));
        UNIT_ASSERT_VALUES_EQUAL(false, sess.IsInternalAuth(0));
        UNIT_ASSERT_VALUES_EQUAL("70501", sess.Uid(1));
        UNIT_ASSERT_VALUES_EQUAL(true, sess.IsExtGlogouted(1));
        UNIT_ASSERT_VALUES_EQUAL(true, sess.IsExternalAuth(1));
        UNIT_ASSERT_VALUES_EQUAL(false, sess.IsInternalAuth(1));
        UNIT_ASSERT_VALUES_EQUAL("70502", sess.Uid(2));
        UNIT_ASSERT_VALUES_EQUAL(true, sess.IsExtGlogouted(2));
        UNIT_ASSERT_VALUES_EQUAL(true, sess.IsExternalAuth(2));
        UNIT_ASSERT_VALUES_EQUAL(true, sess.IsInternalAuth(2));

        UNIT_ASSERT(TSessionUtils::DropExternalAuth(sess, 1));

        UNIT_ASSERT_VALUES_EQUAL(2, sess.UserCount());
        UNIT_ASSERT_VALUES_EQUAL("70500", sess.Uid(0));
        UNIT_ASSERT_VALUES_EQUAL(false, sess.IsExtGlogouted(0));
        UNIT_ASSERT_VALUES_EQUAL(false, sess.IsExternalAuth(0));
        UNIT_ASSERT_VALUES_EQUAL(false, sess.IsInternalAuth(0));
        UNIT_ASSERT_VALUES_EQUAL("70502", sess.Uid(1));
        UNIT_ASSERT_VALUES_EQUAL(true, sess.IsExtGlogouted(1));
        UNIT_ASSERT_VALUES_EQUAL(true, sess.IsExternalAuth(1));
        UNIT_ASSERT_VALUES_EQUAL(true, sess.IsInternalAuth(1));

        UNIT_ASSERT(!TSessionUtils::DropExternalAuth(sess, 1));

        UNIT_ASSERT_VALUES_EQUAL(2, sess.UserCount());
        UNIT_ASSERT_VALUES_EQUAL("70500", sess.Uid(0));
        UNIT_ASSERT_VALUES_EQUAL(false, sess.IsExtGlogouted(0));
        UNIT_ASSERT_VALUES_EQUAL(false, sess.IsExternalAuth(0));
        UNIT_ASSERT_VALUES_EQUAL(false, sess.IsInternalAuth(0));
        UNIT_ASSERT_VALUES_EQUAL("70502", sess.Uid(1));
        UNIT_ASSERT_VALUES_EQUAL(false, sess.IsExtGlogouted(1));
        UNIT_ASSERT_VALUES_EQUAL(false, sess.IsExternalAuth(1));
        UNIT_ASSERT_VALUES_EQUAL(true, sess.IsInternalAuth(1));
    }

    Y_UNIT_TEST(getLoginId) {
        const NPassport::NDbPool::TDestinationPtr dsn = NPassport::NDbPool::TDestination::CreateSqlite(
            TestsDir() + "db/safeguarddb.sqlite3.sql");
        NPassport::NDbPool::TDbPool pool(NPassport::NDbPool::TDbPoolSettings{
            .Dsn = dsn,
        });
        TSessionSigner sessSigner(pool);
        UNIT_ASSERT_NO_EXCEPTION(sessSigner.AddKeyspace("yandex.ru"));
        TSessionParser sessParser(sessSigner, EEnvironmentType::Testing);

        TSession sess = sessParser.Create("70500", "0", "yandex.ru");

        UNIT_ASSERT_VALUES_EQUAL("", TSessionUtils::GetLoginId(sess));

        sess.SetAuthId("12345:AQAAfw:100");
        UNIT_ASSERT_VALUES_EQUAL("s:12345:AQAAfw:100", TSessionUtils::GetLoginId(sess));

        sess.SetLoginId("I am login id");
        UNIT_ASSERT_VALUES_EQUAL("I am login id", TSessionUtils::GetLoginId(sess));
    }
}
