#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/blackbox.h>
#include <passport/infra/daemons/blackbox/src/misc/shards_map.h>

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/destination.h>
#include <passport/infra/libs/cpp/utils/log/logger.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/string.h>

using namespace NPassport;
using namespace NPassport::NBb;

Y_UNIT_TEST_SUITE(TPasspBbShardsTestSuite) {
    // NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
    std::unique_ptr<TTestDbHolder> DB;

    TTestDbHolder& Db() {
        if (!DB) {
            DB = std::make_unique<TTestDbHolder>();
        }
        return *DB;
    }

    Y_UNIT_TEST(Init) {
        TStringStream badShard;
        badShard << "0 5\na 10\n";
        TStringStream badShard2;
        badShard2 << "0 5\n5 -1\n";
        TStringStream badLine;
        badLine << "0 5\n5 1 2 3\n";
        TStringStream ordShard;
        ordShard << "0 5\n5 1\n2 3\n";
        TStringStream gapShard;
        gapShard << "1 2\n20 1\n";
        TStringStream emptyShard;
        emptyShard << "0 5\n0 2\n";
        TStringStream emptyShard2;
        emptyShard2 << "0 5\n5 5\n5 2\n";
        TStringStream bigShard;
        bigShard << "0 2\n 10 100\n";
        TStringStream emptyConf;
        emptyConf << "\n\n";
        TStringStream trailingShard;
        trailingShard << "0 2\n# comment \n3 1\n5 2";
        TStringStream goodShard;
        goodShard << "0 2\n   10\t\t3 \n20  4\n";

        TShardRanges tst;

        UNIT_ASSERT_EXCEPTION_CONTAINS(tst.Init(badShard, 10), yexception, "can't parse range line 'a 10' : INVALID_PARAMS");
        UNIT_ASSERT_EXCEPTION_CONTAINS(tst.Init(badShard2, 10), yexception, "can't parse range line '5 -1' : INVALID_PARAMS");
        UNIT_ASSERT_EXCEPTION_CONTAINS(tst.Init(badLine, 10), yexception, "wrong number of elements in line");
        UNIT_ASSERT_EXCEPTION_CONTAINS(tst.Init(ordShard, 10), yexception, "range not sorted");
        UNIT_ASSERT_EXCEPTION_CONTAINS(tst.Init(gapShard, 10), yexception, "uncovered range [0,1]");
        UNIT_ASSERT_EXCEPTION_CONTAINS(tst.Init(emptyShard, 10), yexception, "range not sorted");
        UNIT_ASSERT_EXCEPTION_CONTAINS(tst.Init(emptyShard2, 10), yexception, "range not sorted");
        UNIT_ASSERT_EXCEPTION_CONTAINS(tst.Init(bigShard, 10), yexception, "range mapped to unknown shard: 100");
        UNIT_ASSERT_EXCEPTION_CONTAINS(tst.Init(emptyConf, 10), yexception, "range config empty");

        UNIT_ASSERT_NO_EXCEPTION(tst.Init(trailingShard, 10));
        UNIT_ASSERT_NO_EXCEPTION(tst.Init(goodShard, 10));
    }

    Y_UNIT_TEST(GetShard) {
        // 0..9 -> 2
        // 10..19 -> 3
        // 20..inf -> 4
        TStringStream shard_conf;
        shard_conf << "0 2\n   10\t\t3 \n#comment\n\n\n20  4\n";

        TShardRanges r;
        UNIT_ASSERT_NO_EXCEPTION(r.Init(shard_conf, 4));

        UNIT_ASSERT_EXCEPTION_CONTAINS(r.GetShard("abc"), yexception, "Internal error: can't parse uid abc");

        UNIT_ASSERT_VALUES_EQUAL((TShardRanges::TShard)1, r.GetShard("0"));
        UNIT_ASSERT_VALUES_EQUAL((TShardRanges::TShard)1, r.GetShard("1"));
        UNIT_ASSERT_VALUES_EQUAL((TShardRanges::TShard)1, r.GetShard("9"));
        UNIT_ASSERT_VALUES_EQUAL((TShardRanges::TShard)2, r.GetShard("10"));
        UNIT_ASSERT_VALUES_EQUAL((TShardRanges::TShard)2, r.GetShard("11"));
        UNIT_ASSERT_VALUES_EQUAL((TShardRanges::TShard)2, r.GetShard("19"));
        UNIT_ASSERT_VALUES_EQUAL((TShardRanges::TShard)3, r.GetShard("20"));
        UNIT_ASSERT_VALUES_EQUAL((TShardRanges::TShard)3, r.GetShard("30"));
        UNIT_ASSERT_VALUES_EQUAL((TShardRanges::TShard)3, r.GetShard("100"));
        UNIT_ASSERT_VALUES_EQUAL((TShardRanges::TShard)3, r.GetShard("10000"));
    }

    Y_UNIT_TEST(ShardsMapBadConfig) {
        NXml::TConfig config = NXml::TConfig::ReadFromFile(TestsDir() + "data/grants.conf");
        TRangedShardsMap shards;
        UNIT_ASSERT_EXCEPTION_CONTAINS(shards.Init("/config/components/component[@name='blackbox']", config, nullptr),
                                       NXml::TConfig::TMissingException,
                                       "nonexistent config param: /config/components/component[@name='blackbox']/ranges_path");
    }

    Y_UNIT_TEST(ShardsMap) {
        const TRangedShardsMap& shards = Db().GetShards();

        UNIT_ASSERT_VALUES_EQUAL(2, shards.GetShardCount());

        TString statusbuf;
        NDbPool::TDbPool* p;
        UNIT_ASSERT_NO_EXCEPTION(p = &shards.GetPool(0));
        UNIT_ASSERT(p->IsOk(&statusbuf));
        UNIT_ASSERT_STRING_CONTAINS(statusbuf, "dr=sqlite;db=");
        UNIT_ASSERT_STRING_CONTAINS(statusbuf, "db/safeguarddb.sqlite3.sql is OK");
        UNIT_ASSERT_STRING_CONTAINS(p->GetDbInfo().Serialized, "dr=sqlite;db=");
        UNIT_ASSERT_STRING_CONTAINS(p->GetDbInfo().Serialized, "db/safeguarddb.sqlite3.sql");

        UNIT_ASSERT_NO_EXCEPTION(p = &shards.GetPool(1));
        UNIT_ASSERT(p->IsOk(&statusbuf));
        UNIT_ASSERT_STRING_CONTAINS(statusbuf, "dr=sqlite;db=");
        UNIT_ASSERT_STRING_CONTAINS(statusbuf, "db/safeguarddb.sqlite3.sql is OK");
        UNIT_ASSERT_STRING_CONTAINS(p->GetDbInfo().Serialized, "dr=sqlite;db=");
        UNIT_ASSERT_STRING_CONTAINS(p->GetDbInfo().Serialized, "db/safeguarddb.sqlite3.sql");

        UNIT_ASSERT(shards.ShardsOk(statusbuf));
        UNIT_ASSERT_STRING_CONTAINS(statusbuf, "dr=sqlite;db=");
        UNIT_ASSERT_STRING_CONTAINS(statusbuf, "db/safeguarddb.sqlite3.sql is OK: 1/1 [1/1], dr=sqlite;db=");
        UNIT_ASSERT_STRING_CONTAINS(statusbuf, "db/safeguarddb.sqlite3.sql is OK");

        UNIT_ASSERT_NO_EXCEPTION(p = &shards.GetPool("0"));
        UNIT_ASSERT(p->IsOk(&statusbuf));
        UNIT_ASSERT_STRING_CONTAINS(statusbuf, "dr=sqlite;db=");
        UNIT_ASSERT_STRING_CONTAINS(statusbuf, "db/safeguarddb.sqlite3.sql is OK");
        UNIT_ASSERT_STRING_CONTAINS(p->GetDbInfo().Serialized, "dr=sqlite;db=");
        UNIT_ASSERT_STRING_CONTAINS(p->GetDbInfo().Serialized, "db/safeguarddb.sqlite3.sql");

        UNIT_ASSERT_NO_EXCEPTION(p = &shards.GetPool("1"));
        UNIT_ASSERT(p->IsOk(&statusbuf));
        UNIT_ASSERT_STRING_CONTAINS(statusbuf, "dr=sqlite;db=");
        UNIT_ASSERT_STRING_CONTAINS(statusbuf, "db/safeguarddb.sqlite3.sql is OK");
        UNIT_ASSERT_STRING_CONTAINS(p->GetDbInfo().Serialized, "dr=sqlite;db=");
        UNIT_ASSERT_STRING_CONTAINS(p->GetDbInfo().Serialized, "db/safeguarddb.sqlite3.sql");

        UNIT_ASSERT_NO_EXCEPTION(p = &shards.GetPool("10000000000000"));
        UNIT_ASSERT(p->IsOk(&statusbuf));
        UNIT_ASSERT_STRING_CONTAINS(statusbuf, "dr=sqlite;db=");
        UNIT_ASSERT_STRING_CONTAINS(statusbuf, "db/safeguarddb.sqlite3.sql is OK");
        UNIT_ASSERT_STRING_CONTAINS(p->GetDbInfo().Serialized, "dr=sqlite;db=");
        UNIT_ASSERT_STRING_CONTAINS(p->GetDbInfo().Serialized, "db/safeguarddb.sqlite3.sql");

        UNIT_ASSERT_EXCEPTION_CONTAINS(shards.GetPool("ops!"), yexception, "Internal error: can't parse uid ops!");

        UNIT_ASSERT_EXCEPTION_CONTAINS(shards.SplitUids({"1", "2", "", "3"}), yexception, "Internal error: can't parse uid ");

        auto vec = shards.SplitUids({"1", "3", "1", "1"});
        UNIT_ASSERT_VALUES_EQUAL(2, vec.size());
        UNIT_ASSERT_VALUES_EQUAL("1,3,1,1", vec[0]);
        UNIT_ASSERT_VALUES_EQUAL("", vec[1]);

        vec = shards.SplitUids({"10000000000022", "10000000000000"});
        UNIT_ASSERT_VALUES_EQUAL(2, vec.size());
        UNIT_ASSERT_VALUES_EQUAL("", vec[0]);
        UNIT_ASSERT_VALUES_EQUAL("10000000000022,10000000000000", vec[1]);

        vec = shards.SplitUids({"10", "100000000000003", "1", "11000000000000"});
        UNIT_ASSERT_VALUES_EQUAL(2, vec.size());
        UNIT_ASSERT_VALUES_EQUAL("10,1", vec[0]);
        UNIT_ASSERT_VALUES_EQUAL("100000000000003,11000000000000", vec[1]);

        UNIT_ASSERT(!shards.GetPoolUnsafe(2));
        UNIT_ASSERT_EXCEPTION_CONTAINS(shards.GetPool(2), TBlackboxError, "There is no shard #2");
    }
}
