#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/signer.h>

#include <passport/infra/libs/cpp/auth_core/keyring.h>
#include <passport/infra/libs/cpp/auth_core/sessionsigner.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NBb;
using namespace NPassport::NAuth;

Y_UNIT_TEST_SUITE(PasspBbSigner) {
    Y_UNIT_TEST(signerStatic) {
        UNIT_ASSERT_VALUES_EQUAL("OK", TSigner::StatusStr(TSigner::EStatus::Ok));
        UNIT_ASSERT_VALUES_EQUAL("EXPIRED", TSigner::StatusStr(TSigner::EStatus::Expired));
        UNIT_ASSERT_VALUES_EQUAL("NO_KEY", TSigner::StatusStr(TSigner::EStatus::NoKey));
        UNIT_ASSERT_VALUES_EQUAL("INVALID", TSigner::StatusStr(TSigner::EStatus::Invalid));
    }

    Y_UNIT_TEST(signerTest) {
        const NPassport::NDbPool::TDestinationPtr dsn = NPassport::NDbPool::TDestination::CreateSqlite(
            TestsDir() + "db/safeguarddb.sqlite3.sql");
        NPassport::NDbPool::TDbPool pool(NPassport::NDbPool::TDbPoolSettings{
            .Dsn = dsn,
        });

        TSessionSigner sessSigner(pool);
        UNIT_ASSERT_NO_EXCEPTION(sessSigner.AddKeyspace("signsmth"));
        UNIT_ASSERT_NO_EXCEPTION(sessSigner.AddKeyspace("signsmth_long"));

        TSigner signer(sessSigner, {"very_long_space"});

        // getRing
        UNIT_ASSERT_VALUES_EQUAL("signsmth", signer.GetRing("")->KSpace());
        UNIT_ASSERT_VALUES_EQUAL("signsmth", signer.GetRing("signsmth")->KSpace());
        UNIT_ASSERT_VALUES_EQUAL("signsmth", signer.GetRing("everyone")->KSpace());
        UNIT_ASSERT_VALUES_EQUAL("signsmth_long", signer.GetRing("very_long_space")->KSpace());

        TString value = "0 some test value to sign {:,;}& ｡◕‿◕｡";
        value[0] = '\0';

        // checkSign
        UNIT_ASSERT_EXCEPTION_CONTAINS(signer.Sign(value, "space1", "abc"), std::invalid_argument, "toTime() failed");
        UNIT_ASSERT_VALUES_EQUAL(TSigner::TCheckResult(TSigner::EStatus::Invalid, "bla"),
                                 signer.CheckSign("bla-bla.bla", "space"));

        UNIT_ASSERT_VALUES_EQUAL(TSigner::TCheckResult(TSigner::EStatus::Invalid, "abc"),
                                 signer.CheckSign("abc.abc.AAA.BBBB.CCCCCC", "space"));

        UNIT_ASSERT_VALUES_EQUAL(TSigner::TCheckResult(TSigner::EStatus::Expired, "abc", 1234),
                                 signer.CheckSign("1234.abc.AAA.BBBB.CCCCCC", "space"));

        UNIT_ASSERT_VALUES_EQUAL(TSigner::TCheckResult(TSigner::EStatus::NoKey, "1234", 4444444444),
                                 signer.CheckSign("4444444444.1234.AAA.BBBB.CCCCCC", "space"));

        UNIT_ASSERT_VALUES_EQUAL(TSigner::TCheckResult(TSigner::EStatus::Invalid, "248102", 3733058537),
                                 signer.CheckSign("3733058537.248102.y_5-7d9O27tivHJd.ndYihgeCRz0dLuBlGA.RedXD2ScO_ui2aji4_p-2g", ""));

        UNIT_ASSERT_VALUES_EQUAL(TSigner::TCheckResult(TSigner::EStatus::Invalid, "248102", 3733058537),
                                 signer.CheckSign("3733058537.248102.y_5-7d9O27tivHJd.ndYihgeCRz0dLuBlGA.RedXD2ScO_ui2aji4_p-2g", "space"));

        UNIT_ASSERT_VALUES_EQUAL(TSigner::TCheckResult(TSigner::EStatus::Invalid, "248102", 3733058537),
                                 signer.CheckSign("3733058537.248102.y_5-7d9O27tivHJd.ndYihgeCRz0dLuBlGA.REdXD2ScO_ui2aji4_p-2g", "space1"));

        UNIT_ASSERT_VALUES_EQUAL(TSigner::TCheckResult(TSigner::EStatus::Ok, "248102", 3733058537, "Hello, world!"),
                                 signer.CheckSign("3733058537.248102.y_5-7d9O27tivHJd.ndYihgeCRz0dLuBlGA.RedXD2ScO_ui2aji4_p-2g", "space1"));

        UNIT_ASSERT_VALUES_EQUAL(TSigner::TCheckResult(TSigner::EStatus::NoKey, "248102", 3733058537),
                                 signer.CheckSign("3733058537.248102.y_5-7d9O27tivHJd.ndYihgeCRz0dLuBlGA.RedXD2ScO_ui2aji4_p-2g", "very_long_space"));

        UNIT_ASSERT_VALUES_EQUAL(TSigner::TCheckResult(TSigner::EStatus::NoKey, "527306", 2837268202),
                                 signer.CheckSign("2837268202.527306.VITUMDSKT9Zt7qOW.eQhEaVd6BdFTnD5UwA.PYWvsoELFCwoQJBOSdTXuw", "space1"));

        UNIT_ASSERT_VALUES_EQUAL(TSigner::TCheckResult(TSigner::EStatus::Ok, "527306", 2837268202, "Hello, world!"),
                                 signer.CheckSign("2837268202.527306.VITUMDSKT9Zt7qOW.eQhEaVd6BdFTnD5UwA.PYWvsoELFCwoQJBOSdTXuw", "very_long_space"));

        // sign / checkSign
        const time_t now = time(nullptr);
        const TString signedVal = signer.Sign(value, "space1", "2222222222", now);
        const TString signedValLong = signer.Sign(value, "very_long_space", "3333333333", now);
        UNIT_ASSERT_STRINGS_UNEQUAL("", signedVal);
        UNIT_ASSERT_STRINGS_UNEQUAL("", signedValLong);

        UNIT_ASSERT_VALUES_EQUAL(TSigner::TCheckResult(TSigner::EStatus::Invalid, "248102", now + 2222222222),
                                 signer.CheckSign(signedVal, ""));

        UNIT_ASSERT_VALUES_EQUAL(TSigner::TCheckResult(TSigner::EStatus::Invalid, "248102", now + 2222222222),
                                 signer.CheckSign(signedVal, "space"));

        UNIT_ASSERT_VALUES_EQUAL(TSigner::TCheckResult(TSigner::EStatus::Ok, "248102", now + 2222222222, TString(value)),
                                 signer.CheckSign(signedVal, "space1"));

        UNIT_ASSERT_VALUES_EQUAL(TSigner::TCheckResult(TSigner::EStatus::NoKey, "527306", now + 3333333333),
                                 signer.CheckSign(signedValLong, "space1"));

        UNIT_ASSERT_VALUES_EQUAL(TSigner::TCheckResult(TSigner::EStatus::NoKey, "248102", now + 2222222222),
                                 signer.CheckSign(signedVal, "very_long_space"));

        UNIT_ASSERT_VALUES_EQUAL(TSigner::TCheckResult(TSigner::EStatus::Ok, "527306", now + 3333333333, TString(value)),
                                 signer.CheckSign(signedValLong, "very_long_space"));
    }
}
