#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/staff/staff_info.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

#include <unordered_map>

using namespace NPassport::NBb;
using namespace NPassport;

Y_UNIT_TEST_SUITE(PasspBbStaffInfoTest) {
    class TTestStaffClient: public TStaffClient {
    public:
        using TStaffClient::MakeRobotsRequest;
        using TStaffClient::TStaffClient;
    };

    using TPageByLastId = std::unordered_map<ui64, TString>;

    class TFakeStaffClient: public IStaffClient {
    public:
        TFakeStaffClient(const TPageByLastId& pages)
            : Pages_(pages)
        {
        }

        TResponse GetRobotsPage(ui64 lastEntityId, NUnistat::TTimeStat& unistatResponseTime) override {
            unistatResponseTime.Insert(TDuration::Seconds(0));

            auto it = Pages_.find(lastEntityId);
            if (it != Pages_.end()) {
                return {
                    .Success = true,
                    .Body = it->second,
                };
            }

            return {
                .Success = false,
            };
        }

    private:
        const TPageByLastId& Pages_;
    };

    class TTestStaffFetcher: public TStaffFetcher {
    public:
        TTestStaffFetcher(TSettings&& settings)
            : TStaffFetcher(std::move(settings))
        {
        }

        using TStaffFetcher::LastUpdateTime_;
        using TStaffFetcher::Run;

        std::unique_ptr<IStaffClient> GetStaffClient() override {
            return std::make_unique<TFakeStaffClient>(Pages_);
        }

        void SetPages(TPageByLastId&& pages) {
            Pages_ = std::move(pages);
        }

    private:
        TPageByLastId Pages_;
    };

    Y_UNIT_TEST(makeRobotsRequest) {
        TTestStaffClient::TSettings settings{
            .Limit = 200,
        };

        TTestStaffClient client(settings);

        UNIT_ASSERT_VALUES_EQUAL(
            "/v3/persons?_sort=id&_query=id%3E100500&official.is_robot=true&_limit=200&_nopage=1&_fields=uid,id",
            client.MakeRobotsRequest(100500));
    }

    Y_UNIT_TEST(validDiskCache) {
        const TString cachePath = GetOutputPath() / "staff_cache.json";

        TString cache = R"(1643284719.[
                {"result": [
                    {"id": 1, "uid": "120001"},
                    {"id": 2, "uid": "120002"},
                    {"id": 100, "uid": "120100"}
                ]},
                {"result": [
                    {"id": 100500, "uid": "220500"}
                ]},
                {"result": []}
            ]
        )";

        TFileOutput(cachePath).Write(cache);

        TTestStaffFetcher fetcher({.CachePath = cachePath});

        UNIT_ASSERT_VALUES_EQUAL(TInstant::Seconds(1643284719), fetcher.LastUpdateTime_);
        UNIT_ASSERT_VALUES_EQUAL(4, fetcher.GetRobots()->size());
        UNIT_ASSERT(fetcher.GetRobots()->contains(120001));
        UNIT_ASSERT(fetcher.GetRobots()->contains(120002));
        UNIT_ASSERT(fetcher.GetRobots()->contains(120100));
        UNIT_ASSERT(fetcher.GetRobots()->contains(220500));

        UNIT_ASSERT_VALUES_EQUAL(cache, TFileInput(cachePath).ReadAll());
    }

    Y_UNIT_TEST(invalidDiskCache) {
        const TString cachePath = GetOutputPath() / "staff_cache.json";
        NFs::Remove(cachePath);

        {
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                TStaffFetcher({.CachePath = cachePath}),
                yexception,
                "StaffFetcher: no valid disk cache, failed to update from staff");
        }

        for (auto& cache : {"", "dgs.[]", "-1435.[]", "1643284719", "1643284719.[some trash]", R"(1643284719.[{"result":[{}]}])"}) {
            TFileOutput(cachePath).Write(cache);

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                TStaffFetcher({.CachePath = cachePath}),
                yexception,
                "StaffFetcher: no valid disk cache, failed to update from staff");
        }
    }

    Y_UNIT_TEST(updateCache) {
        const TString cachePath = GetOutputPath() / "staff_cache.json";

        TFileOutput(cachePath).Write("1643284719.[]");

        TTestStaffFetcher fetcher({
            .CachePath = cachePath,
            .UpdatePeriod = TInstant::Now() - TInstant::Seconds(1643284719),
        });

        UNIT_ASSERT_VALUES_EQUAL(TInstant::Seconds(1643284719), fetcher.LastUpdateTime_);
        UNIT_ASSERT_VALUES_EQUAL(0, fetcher.GetRobots()->size());

        fetcher.SetPages({
            {0, R"({"result":[{"id":5,"uid":"120005"},{"id":11,"uid":"120011"}]})"},
            {11, R"({"result":[{"id":29,"uid":"120029"}]})"},
            {29, R"({"result":[]})"},
        });

        TInstant start = TInstant::Now();
        fetcher.Run();
        TInstant stop = TInstant::Now();

        TInstant updateTime = fetcher.LastUpdateTime_;
        UNIT_ASSERT_LE(start, updateTime);
        UNIT_ASSERT_GE(stop, updateTime);

        UNIT_ASSERT_VALUES_EQUAL(3, fetcher.GetRobots()->size());
        UNIT_ASSERT(fetcher.GetRobots()->contains(120005));
        UNIT_ASSERT(fetcher.GetRobots()->contains(120011));
        UNIT_ASSERT(fetcher.GetRobots()->contains(120029));

        TString expectedCache = IntToString<10>(updateTime.Seconds()) + R"(.[{"result":[{"id":5,"uid":"120005"},{"id":11,"uid":"120011"}]},{"result":[{"id":29,"uid":"120029"}]}])";
        UNIT_ASSERT_VALUES_EQUAL(expectedCache, TFileInput(cachePath).ReadAll());

        fetcher.Run();
        UNIT_ASSERT_VALUES_EQUAL(updateTime, fetcher.LastUpdateTime_);

        fetcher.SetPages({
            {0, R"({"result":[{"id":11,"uid":"120011"}]})"},
        });

        fetcher.LastUpdateTime_ = TInstant::Seconds(1643284719);
        fetcher.Run();
        UNIT_ASSERT_VALUES_EQUAL(TInstant::Seconds(1643284719), fetcher.LastUpdateTime_);

        UNIT_ASSERT_VALUES_EQUAL(3, fetcher.GetRobots()->size());
        UNIT_ASSERT(fetcher.GetRobots()->contains(120005));
        UNIT_ASSERT(fetcher.GetRobots()->contains(120011));
        UNIT_ASSERT(fetcher.GetRobots()->contains(120029));

        UNIT_ASSERT_VALUES_EQUAL(expectedCache, TFileInput(cachePath).ReadAll());
    }

    Y_UNIT_TEST(isRobot) {
        const TString cachePath = GetOutputPath() / "staff_cache.json";

        TFileOutput(cachePath).Write(R"(1643284719.[
               {"result": [
                   {"id": 1, "uid": "120001"},
                   {"id": 100, "uid": "120100"}
               ]}
            ]
        )");

        TStaffInfo info({.CachePath = cachePath});

        UNIT_ASSERT(info.IsRobot(120001));
        UNIT_ASSERT(info.IsRobot(120100));
        UNIT_ASSERT(!info.IsRobot(0));
        UNIT_ASSERT(!info.IsRobot(120002));
        UNIT_ASSERT(!info.IsRobot(120099));
    }
}
