#include "synthetic_attrs_ut.h"

#include <passport/infra/daemons/blackbox/src/misc/db_profile.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/synthetic_attrs.h>

#include <passport/infra/libs/cpp/utils/log/logger.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NBb;
using namespace NPassport;

namespace NPassport::NBb {
    TTestDbHolder& Db() {
        return *Singleton<TTestDbHolder>();
    }

    TFixture::TFixture()
        : SyntheticAttrs(1609500000, {})
        , Fetcher(Db().CreateFetcher())
    {
    }

    TDbProfile::TAttrs& TFixture::Attrs() {
        return const_cast<TDbProfile::TAttrs&>(Fetcher->DefaultProfile().Attrs());
    }

    TDbProfile::TAliases& TFixture::Aliases() {
        return const_cast<TDbProfile::TAliases&>(Fetcher->DefaultProfile().Aliases());
    }

    TDbProfile::TExtendedEntities& TFixture::Emailattrs() {
        return const_cast<TDbProfile::TExtendedEntities&>(Fetcher->DefaultProfile().ExtendedEmailAttrs());
    }

    TDbProfile::TExtendedEntities& TFixture::Phoneattrs() {
        return const_cast<TDbProfile::TExtendedEntities&>(Fetcher->DefaultProfile().ExtendedPhoneAttrs());
    }
}

class TTotpEncryptorFiture: public NUnitTest::TBaseFixture {
public:
    TTotpEncryptorFiture()
        : Totp(CreateTotpEncryptor())
    {
    }

    const TTotpEncryptor Totp;
};

Y_UNIT_TEST_SUITE_F(PasspBbSyntheticTest, TTotpEncryptorFiture) {
    Y_UNIT_TEST(badAttrs) {
        TFixture fix;

        for (const auto& attr : std::vector<TString>({"1", "2", "3", "10", "20", "100", "101", "102", "1000", "1029", "10000", "10001"})) {
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, attr),
                TBlackboxError,
                "invalid synthetic attribute type: '" + attr + "'");
        }
    }

    Y_UNIT_TEST(phoneNumber) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::PHONE_NUMBER);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Attrs(), TDbProfile::TAttrs({{TAttr::PHONES_SECURE, TDbValue()}}));
        UNIT_ASSERT_VALUES_EQUAL(fix.Phoneattrs(),
                                 TDbProfile::TExtendedEntities({{"", TDbProfile::TAttrs({{TPhoneAttr::NUMBER, TDbValue()}})}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PHONES_SECURE, "12345");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.ExtendedPhones_["12345"].clear();
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddPhoneAttr("12345", TPhoneAttr::NUMBER, "");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddPhoneAttr("12345", TPhoneAttr::NUMBER, "+abcd");
        UNIT_ASSERT_VALUES_EQUAL("+abcd", f(&p));

        p.AddPhoneAttr("12345", TPhoneAttr::NUMBER, "123abc");
        UNIT_ASSERT_VALUES_EQUAL("+123abc", f(&p));
    }

    Y_UNIT_TEST(phoneConfirmationDatetime) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::PHONE_CONFIRMATION_DATETIME);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Attrs(), TDbProfile::TAttrs({{TAttr::PHONES_SECURE, TDbValue()}}));
        UNIT_ASSERT_VALUES_EQUAL(fix.Phoneattrs(),
                                 TDbProfile::TExtendedEntities({{"", TDbProfile::TAttrs({{TPhoneAttr::CONFIRMED, TDbValue()}})}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PHONES_SECURE, "12345");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.ExtendedPhones_["12345"].clear();
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddPhoneAttr("12345", TPhoneAttr::CONFIRMED, "");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddPhoneAttr("12345", TPhoneAttr::CONFIRMED, "attribute value");
        UNIT_ASSERT_VALUES_EQUAL("attribute value", f(&p));
    }

    Y_UNIT_TEST(personBirthday) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::PERSON_BIRTH_DAY);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Attrs(), TDbProfile::TAttrs({{TAttr::PERSON_BIRTHDATE, TDbValue()}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "0");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "my day");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "1995");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "0123456789");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "012345678901234");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "1234-56789");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "1234-56-78");
        UNIT_ASSERT_VALUES_EQUAL("56-78", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "1234-56-789");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "0000-11-22");
        UNIT_ASSERT_VALUES_EQUAL("11-22", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "year-mm-dd");
        UNIT_ASSERT_VALUES_EQUAL("mm-dd", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "9999-99-0x");
        UNIT_ASSERT_VALUES_EQUAL("99-0x", f(&p));
    }

    Y_UNIT_TEST(personBirthyear) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::PERSON_BIRTH_YEAR);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Attrs(), TDbProfile::TAttrs({{TAttr::PERSON_BIRTHDATE, TDbValue()}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "0");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "my day");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "1995");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "0123456789");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "012345678901234");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "1234-56789");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "1234-56-78");
        UNIT_ASSERT_VALUES_EQUAL("1234", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "1234-56-789");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "0000-11-22");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "year-mm-dd");
        UNIT_ASSERT_VALUES_EQUAL("year", f(&p));

        p.AddAttr(TAttr::PERSON_BIRTHDATE, "9999-99-0x");
        UNIT_ASSERT_VALUES_EQUAL("9999", f(&p));
    }

    Y_UNIT_TEST(account2faOn) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_2FA_ON);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Attrs(), TDbProfile::TAttrs({{TAttr::ACCOUNT_TOTP_SECRET, TDbValue()}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::ACCOUNT_TOTP_SECRET, "0");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p));

        p.AddAttr(TAttr::ACCOUNT_TOTP_SECRET, "1");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p));
    }

    Y_UNIT_TEST(account2faPinWithNullEncryptor) {
        TFixture fix;

        // null totp_encryptor
        TSyntheticAttributes::TAttrFuncType f_null = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_2FA_PIN);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Attrs(), TDbProfile::TAttrs({{TAttr::ACCOUNT_TOTP_SECRET, TDbValue()}}));

        TTestDbProfile pNull(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f_null(&pNull));

        pNull.AddAttr(TAttr::ACCOUNT_TOTP_SECRET, "0");
        UNIT_ASSERT_VALUES_EQUAL("", f_null(&pNull));

        pNull.Uid_ = "foo";
        UNIT_ASSERT_VALUES_EQUAL("", f_null(&pNull));
    }

    Y_UNIT_TEST(account2faPin) {
        TFixture fix;

        // valid totp_encryptor
        std::unique_ptr<TTestDbFetcher> totpFetcher = Db().CreateFetcher();
        totpFetcher->SetTotpEncryptor(Totp);

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*totpFetcher, TAttr::ACCOUNT_2FA_PIN);
        TTestDbProfile p(totpFetcher->DefaultProfile());
        UNIT_ASSERT(p.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(p.Attrs(), TDbProfile::TAttrs({{TAttr::ACCOUNT_TOTP_SECRET, TDbValue()}}));
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::ACCOUNT_TOTP_SECRET, "0");
        UNIT_ASSERT_EXCEPTION_CONTAINS(f(&p), TBlackboxError, "invalid uid value");

        p.Uid_ = "foo";
        UNIT_ASSERT_EXCEPTION_CONTAINS(f(&p), TBlackboxError, "invalid uid value");

        p.Uid_ = "12341234";
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::ACCOUNT_TOTP_SECRET, "2:2:5:LKCMrmJk+N3JBbN5j7/Ffg==:NOcLHSIyL7AOE9oI1q2ZEj1JWcBRuHn3gcyTohgasXs=:U15I+WujVHah1XYWS+6Duz4XI8UjsK6XIW1kaTQpMJc=");
        UNIT_ASSERT_VALUES_EQUAL("8443", f(&p));

        p.AddAttr(TAttr::ACCOUNT_TOTP_SECRET, "2:2:5:1MyzEned6Sn9AxZmgqZ/9w==:ybyWSYCmYFIemPuGShDTr4w4u58b083zf/DwCVW4SOg=:/Frcu80XbLXFOZ4qR9E3QWB0I+Z54ttcg3AgPO9cLpo=");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::ACCOUNT_TOTP_SECRET, "3:2:5:z0N3EgFF+Uqw/6QJRJIZAw==:Oyxku3r/Kh/utmDEmeVFOVGBKpUWpxFk98mFZbAD/fGMLpCo8Lv+cmVzFVoeJeolHphJIjUwxbBq7qhVm7Xg1TXCmQ1pQ2FZCu42F1wCQsBbWh9raFGBAK4hOMd77T2q:rNkiIPG8QBdz/DpEJ/i8LzCEQyef6ksDP36mnshz28M=");
        UNIT_ASSERT_VALUES_EQUAL("002455", f(&p));
    }

    Y_UNIT_TEST(accountHavePassword) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_HAVE_PASSWORD);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Attrs(), TDbProfile::TAttrs({{TAttr::PASSWORD_ENCRYPTED, TDbValue()}, {TAttr::ACCOUNT_TOTP_SECRET, TDbValue()}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PASSWORD_ENCRYPTED, "0");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p));

        p.AddAttr(TAttr::PASSWORD_ENCRYPTED, "");
        p.AddAttr(TAttr::ACCOUNT_TOTP_SECRET, "1");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p));

        p.AddAttr(TAttr::PASSWORD_ENCRYPTED, "www");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p));
    }

    Y_UNIT_TEST(accountHaveHnt) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_HAVE_HINT);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Attrs(), TDbProfile::TAttrs({{TAttr::HINT_QUESTION_SERIALIZED, TDbValue()}, {TAttr::HINT_ANSWER_ENCRYPTED, TDbValue()}}));
        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::HINT_QUESTION_SERIALIZED, "   ");
        p.AddAttr(TAttr::HINT_ANSWER_ENCRYPTED, "qwerty");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::HINT_QUESTION_SERIALIZED, " to be or not to be  ");
        p.AddAttr(TAttr::HINT_ANSWER_ENCRYPTED, " \t\r\n  ");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::HINT_ANSWER_ENCRYPTED, "qwerty");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::HINT_QUESTION_SERIALIZED, " :question  ");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::HINT_QUESTION_SERIALIZED, " q:question  ");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::HINT_QUESTION_SERIALIZED, " 0:question  ");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::HINT_QUESTION_SERIALIZED, " 99:  ");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::HINT_QUESTION_SERIALIZED, " 99:question  ");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p));

        p.AddAttr(TAttr::HINT_QUESTION_SERIALIZED, " 199:  ");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p));
    }

    Y_UNIT_TEST(personFio) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::PERSON_FIO);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Attrs(), TDbProfile::TAttrs({{TAttr::PERSON_FIRSTNAME, TDbValue()}, {TAttr::PERSON_LASTNAME, TDbValue()}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_FIRSTNAME, "i");
        UNIT_ASSERT_VALUES_EQUAL(" i", f(&p));

        p.AddAttr(TAttr::PERSON_LASTNAME, "  ");
        UNIT_ASSERT_VALUES_EQUAL("   i", f(&p));

        p.AddAttr(TAttr::PERSON_LASTNAME, " 1");
        UNIT_ASSERT_VALUES_EQUAL(" 1 i", f(&p));
    }

    Y_UNIT_TEST(accountNormalizedLogin) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_NORMALIZED_LOGIN);
        UNIT_ASSERT_VALUES_EQUAL(fix.Aliases(),
                                 TDbProfile::TAliases({
                                     {TAlias::PORTAL_LOGIN, TDbValue()},
                                     {TAlias::PDD_MASTER_LOGIN, TDbValue()},
                                     {TAlias::ALT_DOMAIN_LOGIN, TDbValue()},
                                     {TAlias::SOCIAL_LOGIN, TDbValue()},
                                     {TAlias::LITE_LOGIN, TDbValue()},
                                     {TAlias::PHONY_LOGIN, TDbValue()},
                                     {TAlias::MAILISH_LOGIN, TDbValue()},
                                     {TAlias::UBER_ID, TDbValue()},
                                     {TAlias::YAMBOT, TDbValue()},
                                     {TAlias::COLONKISH, TDbValue()},
                                     {TAlias::KINOPOISK_ID, TDbValue()},
                                     {TAlias::NEOPHONISH, TDbValue()},
                                     {TAlias::SCHOLAR, TDbValue()},
                                 }));
        UNIT_ASSERT(fix.Attrs().empty());

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAlias(TAlias::KINOPOISK_ID, "kinopoisk_id");
        UNIT_ASSERT_VALUES_EQUAL("kinopoisk_id", f(&p));

        p.AddAlias(TAlias::COLONKISH, "colonkish");
        UNIT_ASSERT_VALUES_EQUAL("colonkish", f(&p));

        p.AddAlias(TAlias::YAMBOT, "i'm_yambot");
        UNIT_ASSERT_VALUES_EQUAL("i'm_yambot", f(&p));

        p.AddAlias(TAlias::UBER_ID, "uber_id");
        UNIT_ASSERT_VALUES_EQUAL("uber_id", f(&p));

        p.AddAlias(TAlias::MAILISH_LOGIN, "mailish-login");
        UNIT_ASSERT_VALUES_EQUAL("mailish-login", f(&p));

        p.AddAlias(TAlias::NEOPHONISH, "nphne-login");
        UNIT_ASSERT_VALUES_EQUAL("nphne-login", f(&p));

        p.AddAlias(TAlias::PHONY_LOGIN, "phne-login");
        UNIT_ASSERT_VALUES_EQUAL("phne-login", f(&p));

        p.AddAlias(TAlias::SCHOLAR, "ШкольникВася");
        UNIT_ASSERT_VALUES_EQUAL("ШкольникВася", f(&p));

        p.AddAlias(TAlias::LITE_LOGIN, "lite-login");
        UNIT_ASSERT_VALUES_EQUAL("lite-login", f(&p));

        p.AddAlias(TAlias::SOCIAL_LOGIN, "social-login");
        UNIT_ASSERT_VALUES_EQUAL("social-login", f(&p));

        p.AddAlias(TAlias::ALT_DOMAIN_LOGIN, "altdomain-login");
        UNIT_ASSERT_VALUES_EQUAL("altdomain-login", f(&p));

        p.AddAlias(TAlias::PORTAL_LOGIN, "portal-login");
        UNIT_ASSERT_VALUES_EQUAL("portal-login", f(&p));

        p.DomCache_ = std::make_shared<TDomainCache>();
        p.DomCache_->LastEventId = "6666";
        p.DomItem_ = TDomain({
            .Id = "111",
            .Master = "4",
            .Name = "xn-----6kccl6ap0agkc7a2i.xn--p1ai",
            .Mx = 1,
            .DefaultUid = "400500600",
            .Options = {
                .OrganizationName = "Рога && Копыта LTD",
                .OrganizationId = "356",
            },
        });
        UNIT_ASSERT_VALUES_EQUAL("portal-login", f(&p));

        p.Aliases_[TAlias::PORTAL_LOGIN] = TDbValue();
        UNIT_ASSERT_VALUES_EQUAL("altdomain-login", f(&p));

        p.Aliases_[TAlias::ALT_DOMAIN_LOGIN] = TDbValue();
        UNIT_ASSERT_VALUES_EQUAL("social-login", f(&p));

        p.Aliases_[TAlias::SOCIAL_LOGIN] = TDbValue();
        UNIT_ASSERT_VALUES_EQUAL("lite-login", f(&p));

        p.Aliases_[TAlias::LITE_LOGIN] = TDbValue();
        UNIT_ASSERT_VALUES_EQUAL("ШкольникВася", f(&p));

        p.AddAlias(TAlias::PDD_MASTER_LOGIN, "pdd-master-login");
        UNIT_ASSERT_VALUES_EQUAL("pdd-master-login", f(&p));
    }

    Y_UNIT_TEST(accountIsAvailable) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_IS_AVAILABLE);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Attrs(), TDbProfile::TAttrs({{TAttr::ACCOUNT_DISABLED, TDbValue()}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p));

        p.AddAttr(TAttr::ACCOUNT_DISABLED, "1");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::ACCOUNT_DISABLED, "a");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::ACCOUNT_DISABLED, "0");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p));

        p.DomCache_ = std::make_shared<TDomainCache>();
        p.DomCache_->LastEventId = "6666";
        p.DomItem_ = TDomain({
            .Id = "111",
            .Master = "4",
            .Name = "xn-----6kccl6ap0agkc7a2i.xn--p1ai",
            .DefaultUid = "400500600",
            .Options = {
                .OrganizationName = "Рога && Копыта LTD",
                .OrganizationId = "356",
            },
        });
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p)); // not pdd account

        p.AddAlias(TAlias::PDD_MASTER_LOGIN, "pdd@login");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));
    }

    Y_UNIT_TEST(accountTotpSecretIdsWithNullEncruptor) {
        TFixture fix;

        // null totp_encryptor
        TSyntheticAttributes::TAttrFuncType fNull = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_SECRET_IDS);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Attrs(), TDbProfile::TAttrs({{TAttr::ACCOUNT_TOTP_SECRET, TDbValue()}}));

        TTestDbProfile p_null(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", fNull(&p_null));

        p_null.AddAttr(TAttr::ACCOUNT_TOTP_SECRET, "0");
        UNIT_ASSERT_VALUES_EQUAL("", fNull(&p_null));

        p_null.Uid_ = "foo";
        UNIT_ASSERT_VALUES_EQUAL("", fNull(&p_null));
    }

    Y_UNIT_TEST(accountTotpSecretIds) {
        TFixture fix;

        // valid totp_encryptor
        std::unique_ptr<TTestDbFetcher> totpFetcher = Db().CreateFetcher();
        totpFetcher->SetTotpEncryptor(Totp);

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*totpFetcher, TAttr::ACCOUNT_SECRET_IDS);
        TTestDbProfile p(totpFetcher->DefaultProfile());
        UNIT_ASSERT(p.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(p.Attrs(), TDbProfile::TAttrs({{TAttr::ACCOUNT_TOTP_SECRET, TDbValue()}}));
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::ACCOUNT_TOTP_SECRET, "0");
        UNIT_ASSERT_EXCEPTION_CONTAINS(f(&p), TBlackboxError, "invalid uid value");

        p.Uid_ = "foo";
        UNIT_ASSERT_EXCEPTION_CONTAINS(f(&p), TBlackboxError, "invalid uid value");

        p.Uid_ = "12341234";
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::ACCOUNT_TOTP_SECRET, "2:2:5:LKCMrmJk+N3JBbN5j7/Ffg==:NOcLHSIyL7AOE9oI1q2ZEj1JWcBRuHn3gcyTohgasXs=:U15I+WujVHah1XYWS+6Duz4XI8UjsK6XIW1kaTQpMJc=");
        UNIT_ASSERT_VALUES_EQUAL("0:0", f(&p));

        p.AddAttr(TAttr::ACCOUNT_TOTP_SECRET, "2:2:5:1MyzEned6Sn9AxZmgqZ/9w==:ybyWSYCmYFIemPuGShDTr4w4u58b083zf/DwCVW4SOg=:/Frcu80XbLXFOZ4qR9E3QWB0I+Z54ttcg3AgPO9cLpo=");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::ACCOUNT_TOTP_SECRET, "3:2:5:z0N3EgFF+Uqw/6QJRJIZAw==:Oyxku3r/Kh/utmDEmeVFOVGBKpUWpxFk98mFZbAD/fGMLpCo8Lv+cmVzFVoeJeolHphJIjUwxbBq7qhVm7Xg1TXCmQ1pQ2FZCu42F1wCQsBbWh9raFGBAK4hOMd77T2q:rNkiIPG8QBdz/DpEJ/i8LzCEQyef6ksDP36mnshz28M=");
        UNIT_ASSERT_VALUES_EQUAL("13:123123,3:141414", f(&p));
    }

    Y_UNIT_TEST(accountHaveOrganizationName) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_HAVE_ORGANIZATION_NAME);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT(fix.Attrs().empty());

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.DomCache_ = std::make_shared<TDomainCache>();
        p.DomCache_->LastEventId = "6666";
        p.DomItem_ = TDomain({
            .Id = "111",
            .Master = "4",
            .Name = "xn-----6kccl6ap0agkc7a2i.xn--p1ai",
            .DefaultUid = "400500600",
            .Options = {},
        });
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.DomItem_ = TDomain({
            .Id = "111",
            .Master = "4",
            .Name = "xn-----6kccl6ap0agkc7a2i.xn--p1ai",
            .DefaultUid = "400500600",
            .Options = {
                .OrganizationName = "Рога && Копыта LTD",
                .OrganizationId = "",
            },
        });
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p));
    }

    Y_UNIT_TEST(accountTotpPinLengthWithNullEncryptor) {
        TFixture fix;

        // null totp_encryptor
        TSyntheticAttributes::TAttrFuncType fNull = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_PIN_LENGTH);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Attrs(), TDbProfile::TAttrs({{TAttr::ACCOUNT_TOTP_SECRET, TDbValue()}}));

        TTestDbProfile p_null(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", fNull(&p_null));

        p_null.AddAttr(TAttr::ACCOUNT_TOTP_SECRET, "0");
        UNIT_ASSERT_VALUES_EQUAL("", fNull(&p_null));

        p_null.Uid_ = "foo";
        UNIT_ASSERT_VALUES_EQUAL("", fNull(&p_null));
    }

    Y_UNIT_TEST(accountTotpPinLength) {
        TFixture fix;

        // valid totp_encryptor
        std::unique_ptr<TTestDbFetcher> totpFetcher = Db().CreateFetcher();
        totpFetcher->SetTotpEncryptor(Totp);

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*totpFetcher, TAttr::ACCOUNT_PIN_LENGTH);
        TTestDbProfile p(totpFetcher->DefaultProfile());
        UNIT_ASSERT(p.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(p.Attrs(), TDbProfile::TAttrs({{TAttr::ACCOUNT_TOTP_SECRET, TDbValue()}}));
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::ACCOUNT_TOTP_SECRET, "0");
        UNIT_ASSERT_EXCEPTION_CONTAINS(f(&p), TBlackboxError, "invalid uid value");

        p.Uid_ = "foo";
        UNIT_ASSERT_EXCEPTION_CONTAINS(f(&p), TBlackboxError, "invalid uid value");

        p.Uid_ = "12341234";
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::ACCOUNT_TOTP_SECRET, "2:2:5:LKCMrmJk+N3JBbN5j7/Ffg==:NOcLHSIyL7AOE9oI1q2ZEj1JWcBRuHn3gcyTohgasXs=:U15I+WujVHah1XYWS+6Duz4XI8UjsK6XIW1kaTQpMJc=");
        UNIT_ASSERT_VALUES_EQUAL("4", f(&p));

        p.AddAttr(TAttr::ACCOUNT_TOTP_SECRET, "2:2:5:1MyzEned6Sn9AxZmgqZ/9w==:ybyWSYCmYFIemPuGShDTr4w4u58b083zf/DwCVW4SOg=:/Frcu80XbLXFOZ4qR9E3QWB0I+Z54ttcg3AgPO9cLpo=");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::ACCOUNT_TOTP_SECRET, "3:2:5:z0N3EgFF+Uqw/6QJRJIZAw==:Oyxku3r/Kh/utmDEmeVFOVGBKpUWpxFk98mFZbAD/fGMLpCo8Lv+cmVzFVoeJeolHphJIjUwxbBq7qhVm7Xg1TXCmQ1pQ2FZCu42F1wCQsBbWh9raFGBAK4hOMd77T2q:rNkiIPG8QBdz/DpEJ/i8LzCEQyef6ksDP36mnshz28M=");
        UNIT_ASSERT_VALUES_EQUAL("6", f(&p));
    }

    Y_UNIT_TEST(accountSecurityLevel) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_PROTECTION_LEVEL);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Attrs(),
                                 TDbProfile::TAttrs({
                                     {TAttr::HINT_QUESTION_SERIALIZED, TDbValue()},
                                     {TAttr::HINT_ANSWER_ENCRYPTED, TDbValue()},
                                     {TAttr::PASSWORD_QUALITY, TDbValue()},
                                     {TAttr::ACCOUNT_STRONG_PASSWORD_REQUIRED, TDbValue()},
                                     {TAttr::PASSWORD_ENCRYPTED, TDbValue()},
                                     {TAttr::ACCOUNT_TOTP_SECRET, TDbValue()},
                                 }));
        UNIT_ASSERT_VALUES_EQUAL(
            fix.Phoneattrs(),
            TDbProfile::TExtendedEntities({{"",
                                            TDbProfile::TAttrs({
                                                {TPhoneAttr::NUMBER, TDbValue()},
                                                {TPhoneAttr::SECURED, TDbValue()},
                                                {TPhoneAttr::CONFIRMED, TDbValue()},
                                            })}}));
        UNIT_ASSERT_VALUES_EQUAL(
            fix.Emailattrs(),
            TDbProfile::TExtendedEntities({{"",
                                            TDbProfile::TAttrs({
                                                {TEmailAttr::ADDRESS, TDbValue()},
                                                {TEmailAttr::CONFIRMED, TDbValue()},
                                                {TEmailAttr::IS_RPOP, TDbValue()},
                                                {TEmailAttr::IS_SILENT, TDbValue()},
                                                {TEmailAttr::IS_UNSAFE, TDbValue()},
                                            })}}));

        // with no password level is grey
        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("-1", f(&p));

        // no hint, strong password and restore email/phone - red
        p.AddAttr(TAttr::PASSWORD_ENCRYPTED, "qwertypasswd");
        UNIT_ASSERT_VALUES_EQUAL("4", f(&p));

        // hasHint = true
        p.AddAttr(TAttr::HINT_QUESTION_SERIALIZED, "9: ");
        p.AddAttr(TAttr::HINT_ANSWER_ENCRYPTED, "qwerty");
        UNIT_ASSERT_VALUES_EQUAL("4", f(&p));

        // hasHint and no hasStrongPwd (bad quality) - red
        p.AddAttr(TAttr::PASSWORD_QUALITY, "");
        UNIT_ASSERT_VALUES_EQUAL("4", f(&p));

        p.AddAttr(TAttr::PASSWORD_QUALITY, "100");
        UNIT_ASSERT_VALUES_EQUAL("4", f(&p));

        p.AddAttr(TAttr::PASSWORD_QUALITY, ":");
        UNIT_ASSERT_VALUES_EQUAL("4", f(&p));

        p.AddAttr(TAttr::PASSWORD_QUALITY, ":q");
        UNIT_ASSERT_VALUES_EQUAL("4", f(&p));

        p.AddAttr(TAttr::PASSWORD_QUALITY, ":1000000");
        UNIT_ASSERT_VALUES_EQUAL("4", f(&p));

        p.AddAttr(TAttr::PASSWORD_QUALITY, ":-5");
        UNIT_ASSERT_VALUES_EQUAL("4", f(&p));

        // hasHint and hasStrongPwd
        p.AddAttr(TAttr::PASSWORD_QUALITY, ":50");
        UNIT_ASSERT_VALUES_EQUAL("16", f(&p));

        p.AddAttr(TAttr::ACCOUNT_STRONG_PASSWORD_REQUIRED, "2");
        UNIT_ASSERT_VALUES_EQUAL("4", f(&p));

        p.AddAttr(TAttr::PASSWORD_QUALITY, ":75");
        UNIT_ASSERT_VALUES_EQUAL("16", f(&p));

        p.AddAttr(TAttr::PASSWORD_QUALITY, ":175");
        UNIT_ASSERT_VALUES_EQUAL("4", f(&p));

        // no hasStrongPwd and restore Email
        p.AddAttr(TAttr::ACCOUNT_STRONG_PASSWORD_REQUIRED, "1");
        p.AddEmailAttr("", TEmailAttr::CONFIRMED, "111");
        UNIT_ASSERT_VALUES_EQUAL("4", f(&p));

        p.AddEmailAttr("", TEmailAttr::ADDRESS, "mail.ru");
        UNIT_ASSERT_VALUES_EQUAL("16", f(&p));

        p.AddAttr(TAttr::PASSWORD_QUALITY, ":85");
        UNIT_ASSERT_VALUES_EQUAL("32", f(&p));
        p.AddAttr(TAttr::PASSWORD_QUALITY, ":10");

        p.AddEmailAttr("", TEmailAttr::IS_RPOP, "1");
        UNIT_ASSERT_VALUES_EQUAL("4", f(&p));

        p.AddEmailAttr("", TEmailAttr::IS_RPOP, "0");
        p.AddEmailAttr("", TEmailAttr::IS_SILENT, "2");
        UNIT_ASSERT_VALUES_EQUAL("4", f(&p));

        p.AddEmailAttr("", TEmailAttr::IS_SILENT, "0");
        p.AddEmailAttr("", TEmailAttr::IS_UNSAFE, "3");
        UNIT_ASSERT_VALUES_EQUAL("4", f(&p));

        p.AddEmailAttr("", TEmailAttr::IS_UNSAFE, "0");
        UNIT_ASSERT_VALUES_EQUAL("16", f(&p));

        TDbProfile::TAttrs tmpEmailAttrs;
        tmpEmailAttrs.swap(p.ExtendedEmailAttrs()[""]);

        // hasStrongPwd no hasHint, no email and bad restore phone
        p.AddAttr(TAttr::HINT_QUESTION_SERIALIZED, "99: ");
        p.AddAttr(TAttr::ACCOUNT_STRONG_PASSWORD_REQUIRED, "10");
        p.AddAttr(TAttr::PASSWORD_QUALITY, ":85");
        p.AddPhoneAttr("", TPhoneAttr::CONFIRMED, "abc");
        UNIT_ASSERT_VALUES_EQUAL("4", f(&p));

        p.AddPhoneAttr("", TPhoneAttr::NUMBER, "79169161616");
        UNIT_ASSERT_VALUES_EQUAL("4", f(&p));

        p.AddPhoneAttr("", TPhoneAttr::CONFIRMED, "1515151515");
        UNIT_ASSERT_VALUES_EQUAL("4", f(&p));

        // hasStrongPwd, no email and good restore phone
        p.AddPhoneAttr("", TPhoneAttr::CONFIRMED, "1313131313");
        UNIT_ASSERT_VALUES_EQUAL("32", f(&p));

        p.AddPhoneAttr("", TPhoneAttr::CONFIRMED, "1515151515");
        p.AddPhoneAttr("", TPhoneAttr::SECURED, "111");
        UNIT_ASSERT_VALUES_EQUAL("32", f(&p));

        tmpEmailAttrs.swap(p.ExtendedEmailAttrs()[""]);

        // no hasStrongPwd no hasHint, good email and bad restore phone
        p.AddAttr(TAttr::PASSWORD_QUALITY, ":69");
        p.AddPhoneAttr("", TPhoneAttr::CONFIRMED, "abc");
        UNIT_ASSERT_VALUES_EQUAL("16", f(&p));

        p.AddPhoneAttr("", TPhoneAttr::NUMBER, "79169161616");
        UNIT_ASSERT_VALUES_EQUAL("16", f(&p));

        p.AddPhoneAttr("", TPhoneAttr::CONFIRMED, "1515151515");
        UNIT_ASSERT_VALUES_EQUAL("16", f(&p));

        // hasStrongPwd, good email and good restore phone
        p.AddAttr(TAttr::PASSWORD_QUALITY, ":85");
        p.AddPhoneAttr("", TPhoneAttr::CONFIRMED, "abc");
        UNIT_ASSERT_VALUES_EQUAL("32", f(&p));

        p.AddPhoneAttr("", TPhoneAttr::NUMBER, "79169161616");
        UNIT_ASSERT_VALUES_EQUAL("32", f(&p));

        p.AddPhoneAttr("", TPhoneAttr::CONFIRMED, "1515151515");
        UNIT_ASSERT_VALUES_EQUAL("32", f(&p));

        p.AddPhoneAttr("", TPhoneAttr::CONFIRMED, "1313131313");
        UNIT_ASSERT_VALUES_EQUAL("32", f(&p));

        p.AddPhoneAttr("", TPhoneAttr::CONFIRMED, "1515151515");
        p.AddPhoneAttr("", TPhoneAttr::SECURED, "111");
        UNIT_ASSERT_VALUES_EQUAL("32", f(&p));

        // with 2FA - green no matter of other attributes
        p.AddAttr(TAttr::ACCOUNT_TOTP_SECRET, "0");
        UNIT_ASSERT_VALUES_EQUAL("32", f(&p));
    }

    Y_UNIT_TEST(accountRfc2faOn) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_RFC_2FA_ON);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Attrs(), TDbProfile::TAttrs({{TAttr::ACCOUNT_RFC_TOTP_SECRET, TDbValue()}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::ACCOUNT_RFC_TOTP_SECRET, "0");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p));

        p.AddAttr(TAttr::ACCOUNT_RFC_TOTP_SECRET, "1");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p));
    }

    Y_UNIT_TEST(accountHavePlus) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_HAVE_PLUS);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Attrs(), TDbProfile::TAttrs({{TAttr::ACCOUNT_PLUS_ENABLED, TDbValue()}}));
        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::ACCOUNT_PLUS_ENABLED, "01");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::ACCOUNT_PLUS_ENABLED, "10");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p));
    }

    Y_UNIT_TEST(accountPDDOrganizationId) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_PDD_ORGANIZATION_ID);
        UNIT_ASSERT(fix.Aliases().empty());
        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.DomCache_ = std::make_shared<TDomainCache>();
        p.DomCache_->LastEventId = "6666";
        p.DomItem_ = TDomain({
            .Id = "111",
            .Master = "4",
            .Name = "xn-----6kccl6ap0agkc7a2i.xn--p1ai",
            .DefaultUid = "400500600",
            .Options = {
                .OrganizationName = "Рога && Копыта LTD",
                .OrganizationId = "123",
            },
        });
        UNIT_ASSERT_VALUES_EQUAL("123", f(&p));

        p.DomItem_ = TDomain({
            .Id = "111",
            .Master = "4",
            .Name = "xn-----6kccl6ap0agkc7a2i.xn--p1ai",
            .DefaultUid = "400500600",
            .Options = {
                .OrganizationName = "Рога && Копыта LTD",
                .OrganizationId = "",
            },
        });
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.DomCache_ = std::make_shared<TDomainCache>();
        p.DomCache_->LastEventId = "6666";
        p.DomItem_ = TDomain({
            .Id = "111",
            .Master = "4",
            .Name = "xn-----6kccl6ap0agkc7a2i.xn--p1ai",
            .DefaultUid = "400500600",
        });
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.DomCache_ = std::make_shared<TDomainCache>();
        p.DomCache_->LastEventId = "6666";
        p.DomItem_ = TDomain({
            .Id = "111",
            .Master = "4",
            .Name = "xn-----6kccl6ap0agkc7a2i.xn--p1ai",
            .DefaultUid = "400500600",
            .Options = {
                .OrganizationName = "Рога && Копыта LTD",
                .OrganizationId = "asdf",
            },
        });
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));
    }

    Y_UNIT_TEST(accountConnectOrganizationIds) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_CONNECT_ORGANIZATION_IDS);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Attrs(), TDbProfile::TAttrs({{TAttr::ACCOUNT_EXTERNAL_ORGANIZATION_IDS, TDbValue()}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        // not pdd user
        p.AddAttr(TAttr::ACCOUNT_EXTERNAL_ORGANIZATION_IDS, "1,22,333,22,1");
        UNIT_ASSERT_VALUES_EQUAL("1,22,333", f(&p));

        // pdd user, no org_id
        p.DomCache_ = std::make_shared<TDomainCache>();
        p.DomCache_->LastEventId = "6666";
        p.DomItem_ = TDomain({
            .Id = "111",
            .Master = "4",
            .Name = "xn-----6kccl6ap0agkc7a2i.xn--p1ai",
            .DefaultUid = "400500600",
        });
        UNIT_ASSERT_VALUES_EQUAL("1,22,333", f(&p));

        // bad external_ids
        p.AddAttr(TAttr::ACCOUNT_EXTERNAL_ORGANIZATION_IDS, "1,a,b");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::ACCOUNT_EXTERNAL_ORGANIZATION_IDS, "1,,,9");
        UNIT_ASSERT_VALUES_EQUAL("1,9", f(&p));

        p.AddAttr(TAttr::ACCOUNT_EXTERNAL_ORGANIZATION_IDS, "");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        // pdd with no org_id
        p.DomItem_ = TDomain({
            .Id = "111",
            .Master = "4",
            .Name = "xn-----6kccl6ap0agkc7a2i.xn--p1ai",
            .DefaultUid = "400500600",
            .Options = {
                .OrganizationName = "Рога && Копыта LTD",
                .OrganizationId = "",
            },
        });
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        // pdd with bad org_id
        p.DomItem_ = TDomain({
            .Id = "111",
            .Master = "4",
            .Name = "xn-----6kccl6ap0agkc7a2i.xn--p1ai",
            .DefaultUid = "400500600",
            .Options = {
                .OrganizationName = "Рога && Копыта LTD",
                .OrganizationId = "sss",
            },
        });
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        // pdd with org_id
        p.DomItem_ = TDomain({
            .Id = "111",
            .Master = "4",
            .Name = "xn-----6kccl6ap0agkc7a2i.xn--p1ai",
            .DefaultUid = "400500600",
            .Options = {
                .OrganizationName = "Рога && Копыта LTD",
                .OrganizationId = "123",
            },
        });
        UNIT_ASSERT_VALUES_EQUAL("123", f(&p));

        // pdd with duplicate org_id
        p.AddAttr(TAttr::ACCOUNT_EXTERNAL_ORGANIZATION_IDS, "1,123,12");
        UNIT_ASSERT_VALUES_EQUAL("1,12,123", f(&p));

        p.AddAttr(TAttr::ACCOUNT_EXTERNAL_ORGANIZATION_IDS, "1,11,12");
        UNIT_ASSERT_VALUES_EQUAL("1,11,12,123", f(&p));
    }

    Y_UNIT_TEST(accountSuggestPublicName) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_SUGGEST_PUBLIC_NAME);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(
            fix.Attrs(),
            TDbProfile::TAttrs({
                {TAttr::PERSON_FIRSTNAME, TDbValue()},
                {TAttr::PERSON_LASTNAME, TDbValue()},
                {TAttr::ACCOUNT_DISPLAY_NAME, TDbValue()},
                {TAttr::PERSON_DONT_USE_DISPLAYNAME_AS_PUBLICNAME, TDbValue()},
                {TAttr::PERSON_SHOW_FIO_IN_PUBLIC_NAME, TDbValue()},
                {TAttr::ACCOUNT_REGISTRATION_DATETIME, TDbValue()},
            }));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        // has firstname/lastname, fio suggested
        p.AddAttr(TAttr::PERSON_FIRSTNAME, "Иван");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_LASTNAME, "Петров");
        UNIT_ASSERT_VALUES_EQUAL("Иван Петров", f(&p));

        p.AddAttr(TAttr::PERSON_FIRSTNAME, "");
        UNIT_ASSERT_VALUES_EQUAL("Петров", f(&p));

        // has fio but suggest off
        p.AddAttr(TAttr::PERSON_FIRSTNAME, "Джон");
        p.AddAttr(TAttr::PERSON_LASTNAME, "Смит");
        p.AddAttr(TAttr::PERSON_SHOW_FIO_IN_PUBLIC_NAME, "1");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_SHOW_FIO_IN_PUBLIC_NAME, "0");
        p.AddAttr(TAttr::ACCOUNT_DISPLAY_NAME, "ignore bad display_name");
        UNIT_ASSERT_VALUES_EQUAL("Джон Смит", f(&p));

        // has fio and regisgtered after/before fullFioStartTime
        p.AddAttr(TAttr::ACCOUNT_REGISTRATION_DATETIME, "1609500001");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::ACCOUNT_REGISTRATION_DATETIME, "1609500000");
        UNIT_ASSERT_VALUES_EQUAL("Джон Смит", f(&p));

        p.AddAttr(TAttr::ACCOUNT_REGISTRATION_DATETIME, "1500000000");
        UNIT_ASSERT_VALUES_EQUAL("Джон Смит", f(&p));

        // also has a valid display name, no suggest
        p.AddAttr(TAttr::ACCOUNT_DISPLAY_NAME, "p:has display_name");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::PERSON_DONT_USE_DISPLAYNAME_AS_PUBLICNAME, "1");
        UNIT_ASSERT_VALUES_EQUAL("Джон Смит", f(&p));
    }

    Y_UNIT_TEST(accountHasCustomPublicId) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_HAS_CUSTOM_PUBLIC_ID);
        UNIT_ASSERT(fix.Attrs().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Aliases(), TDbProfile::TAliases({{TAlias::PUBLICID, TDbValue()}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAlias(TAlias::PUBLICID, "");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAlias(TAlias::PUBLICID, "Username");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p));
    }

    Y_UNIT_TEST(accountMusicContestRatingKlass) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_MUSIC_CONTENT_RATING_KLASS);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(
            fix.Attrs(),
            TDbProfile::TAttrs({
                {TAttr::ACCOUNT_MUSIC_CONTENT_RATING_CLASS, TDbValue()},
                {TAttr::ACCOUNT_CONTENT_RATING_CLASS, TDbValue()},
            }));
        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::ACCOUNT_CONTENT_RATING_CLASS, "10+");
        UNIT_ASSERT_VALUES_EQUAL("10+", f(&p));

        p.AddAttr(TAttr::ACCOUNT_MUSIC_CONTENT_RATING_CLASS, "3+");
        UNIT_ASSERT_VALUES_EQUAL("3+", f(&p));

        p.AddAttr(TAttr::ACCOUNT_VIDEO_CONTENT_RATING_CLASS, "18+");
        UNIT_ASSERT_VALUES_EQUAL("3+", f(&p));

        p.AddAttr(TAttr::ACCOUNT_MUSIC_CONTENT_RATING_CLASS, "");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));
    }

    Y_UNIT_TEST(accountVideoContestRatingKlass) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_VIDEO_CONTENT_RATING_KLASS);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(
            fix.Attrs(),
            TDbProfile::TAttrs({
                {TAttr::ACCOUNT_VIDEO_CONTENT_RATING_CLASS, TDbValue()},
                {TAttr::ACCOUNT_CONTENT_RATING_CLASS, TDbValue()},
            }));
        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAttr(TAttr::ACCOUNT_CONTENT_RATING_CLASS, "10+");
        UNIT_ASSERT_VALUES_EQUAL("10+", f(&p));

        p.AddAttr(TAttr::ACCOUNT_VIDEO_CONTENT_RATING_CLASS, "18+");
        UNIT_ASSERT_VALUES_EQUAL("18+", f(&p));

        p.AddAttr(TAttr::ACCOUNT_MUSIC_CONTENT_RATING_CLASS, "3+");
        UNIT_ASSERT_VALUES_EQUAL("18+", f(&p));

        p.AddAttr(TAttr::ACCOUNT_VIDEO_CONTENT_RATING_CLASS, "");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));
    }

    Y_UNIT_TEST(accountPlusMappedSubscriptionLevel) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_PLUS_MAPPED_SUBSCRIPTION_LEVEL);
        UNIT_ASSERT(fix.Aliases().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Attrs(), TDbProfile::TAttrs({{TAttr::ACCOUNT_PLUS_SUBSCRIPTION_LEVEL, TDbValue()}}));
        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        for (const TString& emptyValue : std::vector<TString>{"0", "5", "9", "50", "51", "100", "", "asgd"}) {
            p.AddAttr(TAttr::ACCOUNT_PLUS_SUBSCRIPTION_LEVEL, emptyValue);
            UNIT_ASSERT_VALUES_EQUAL("", f(&p));
        }

        for (const auto& [level, mappedLevel] : std::unordered_map<TString, TString>{
                 {"10", "1"},
                 {"19", "1"},
                 {"20", "2"},
                 {"29", "2"},
                 {"30", "3"},
                 {"39", "3"},
                 {"40", "4"},
                 {"49", "4"},
             }) {
            p.AddAttr(TAttr::ACCOUNT_PLUS_SUBSCRIPTION_LEVEL, level);
            UNIT_ASSERT_VALUES_EQUAL(mappedLevel, f(&p));
        }
    }

    Y_UNIT_TEST(syntheticProxyAttrs) {
        TSyntheticAttributes::TProxyAttrsMap mapSyntToRealAttrs = {
            {TAttr::ACCOUNT_KINOPOISK_OTT_SUBSCRIPTION_NAME, TAttr::ACCOUNT_OTT_SUBSCRIPTION},
            {TAttr::ACCOUNT_PLUS_TRIAL_USED_TS_SYNT, TAttr::ACCOUNT_PLUS_TRIAL_USED_TS},
            {TAttr::ACCOUNT_PLUS_SUBSCRIPTION_STOPPED_TS_SYNT, TAttr::ACCOUNT_PLUS_SUBSCRIPTION_STOPPED_TS},
            {TAttr::ACCOUNT_PLUS_SUBSCRIPTION_EXPIRE_TS_SYNT, TAttr::ACCOUNT_PLUS_SUBSCRIPTION_EXPIRE_TS},
            {TAttr::ACCOUNT_PLUS_NEXT_CHARGE_TS_SYNT, TAttr::ACCOUNT_PLUS_NEXT_CHARGE_TS},
            {TAttr::ACCOUNT_PLUS_FAMILY_ROLE_SYNT, TAttr::ACCOUNT_PLUS_FAMILY_ROLE},
            {TAttr::ACCOUNT_HAS_PLUS_CASHBACK, TAttr::ACCOUNT_PLUS_CASHBACK_ENABLED},
        };

        TFixture fix;

        for (const auto& pair : mapSyntToRealAttrs) {
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, pair.first),
                TBlackboxError,
                "invalid synthetic attribute type: '" + pair.first + "'");
        }

        fix.SyntheticAttrs = TSyntheticAttributes(1609500000, TSyntheticAttributes::TProxyAttrsMap(mapSyntToRealAttrs));

        for (const auto& pair : mapSyntToRealAttrs) {
            TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, pair.first);
            UNIT_ASSERT_VALUES_EQUAL_C(fix.Attrs(), TDbProfile::TAttrs({{pair.second, TDbValue()}}), pair.first);

            TTestDbProfile p(fix.Fetcher->DefaultProfile());
            UNIT_ASSERT_VALUES_EQUAL_C("", f(&p), pair.first);

            p.AddAttr(pair.second, "");
            UNIT_ASSERT_VALUES_EQUAL_C("", f(&p), pair.first);

            p.AddAttr(pair.second, "1,22,333,22,1");
            UNIT_ASSERT_VALUES_EQUAL_C("1,22,333,22,1", f(&p), pair.first);

            p.AddAttr(pair.second, "very good vip value!");
            UNIT_ASSERT_VALUES_EQUAL_C("very good vip value!", f(&p), pair.first);

            fix.Attrs().clear();
        }
    }

    Y_UNIT_TEST(accountCompletionRecommended) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_COMPLETION_RECOMMENDED);

        UNIT_ASSERT_VALUES_EQUAL(fix.Attrs(), TDbProfile::TAttrs({{TAttr::PERSON_FIRSTNAME, TDbValue()}, {TAttr::PERSON_LASTNAME, TDbValue()}}));
        UNIT_ASSERT_VALUES_EQUAL(fix.Aliases(), TDbProfile::TAliases({{TAlias::NEOPHONISH, TDbValue()}, {TAlias::PORTAL_LOGIN, TDbValue()}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAlias(TAlias::NEOPHONISH, "nphne-login");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p));

        p.AddAlias(TAlias::PORTAL_LOGIN, "portal-login");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.Aliases_[TAlias::NEOPHONISH] = TDbValue();
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.AddAlias(TAlias::NEOPHONISH, "nphne-login");
        p.Aliases_[TAlias::PORTAL_LOGIN] = TDbValue();

        p.AddAttr(TAttr::PERSON_FIRSTNAME, "Nick");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        p.Attrs_[TAttr::PERSON_FIRSTNAME] = TDbValue();

        p.AddAttr(TAttr::PERSON_LASTNAME, "Smith");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));
    }

    Y_UNIT_TEST(accountFamilyChildrenManagement) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_FAMILY_CHILDREN_MANAGEMENT_SYNT);

        UNIT_ASSERT_VALUES_EQUAL(fix.Attrs(), TDbProfile::TAttrs({{TAttr::ACCOUNT_FAMILY_CHILDREN_MANAGEMENT, TDbValue()}}));
        UNIT_ASSERT_VALUES_EQUAL(fix.Aliases(), TDbProfile::TAliases({}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT(p.NeedFamilyInfo_);

        p.SetUid("129");

        p.FamilyInfo_ = {};
        p.Attrs_[TAttr::ACCOUNT_FAMILY_CHILDREN_MANAGEMENT] = TDbValue();
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));
        p.AddAttr(TAttr::ACCOUNT_FAMILY_CHILDREN_MANAGEMENT, "1");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p));

        p.FamilyInfo_ = TFamilyInfo{.AdminUid = "100500"};
        p.Attrs_[TAttr::ACCOUNT_FAMILY_CHILDREN_MANAGEMENT] = TDbValue();
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));
        p.AddAttr(TAttr::ACCOUNT_FAMILY_CHILDREN_MANAGEMENT, "1");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p));

        p.FamilyInfo_ = TFamilyInfo{.AdminUid = "129"};
        p.Attrs_[TAttr::ACCOUNT_FAMILY_CHILDREN_MANAGEMENT] = TDbValue();
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p));
        p.AddAttr(TAttr::ACCOUNT_FAMILY_CHILDREN_MANAGEMENT, "1");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p));
    }

    Y_UNIT_TEST(accountIsKid) {
        TFixture fix;

        TSyntheticAttributes::TAttrFuncType f = fix.SyntheticAttrs.GetSyntheticFunc(*fix.Fetcher, TAttr::ACCOUNT_IS_KID);

        UNIT_ASSERT_VALUES_EQUAL(fix.Aliases(), TDbProfile::TAliases({}));

        // no portal, no kiddish
        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        // no portal, kiddish
        p.AddAlias(TAlias::KIDDISH, "kid-alias");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p));

        // portal, no kiddish
        p.AddAlias(TAlias::PORTAL_LOGIN, "portal-login");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));

        // portal, no kiddish
        p.Aliases_[TAlias::KIDDISH] = TDbValue();
        UNIT_ASSERT_VALUES_EQUAL("", f(&p));
    }
}
