#include "synthetic_attrs_ut.h"

#include <passport/infra/daemons/blackbox/src/misc/db_profile.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/synthetic_attrs.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NBb;
using namespace NPassport;

Y_UNIT_TEST_SUITE(PasspBbSyntheticPhoneTest) {
    Y_UNIT_TEST(badAttrs) {
        TFixture fix;

        for (const auto& attr : std::vector<TString>({"1", "2", "10", "20", "100", "110", "111", "112", "1001", "10000", "10001"})) {
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                fix.SyntheticAttrs.GetSyntheticExtPhoneFunc(*fix.Fetcher, attr),
                TBlackboxError,
                "invalid synthetic extended phone attribute type: '" + attr + "'");
        }
    }

    Y_UNIT_TEST(phoneFormattedNumber) {
        TFixture fix;

        TSyntheticAttributes::TExtAttrFuncType f = fix.SyntheticAttrs.GetSyntheticExtPhoneFunc(*fix.Fetcher, TPhoneAttr::FORMATTED_NUMBER);
        UNIT_ASSERT(fix.Attrs().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Phoneattrs(),
                                 TDbProfile::TExtendedEntities({{"", TDbProfile::TAttrs({{TPhoneAttr::NUMBER, TDbValue()}})}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, ""));
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "1"));

        p.AddPhoneAttr("2", TPhoneAttr::NUMBER, "");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "2"));

        p.AddPhoneAttr("2", TPhoneAttr::NUMBER, "79169161616");
        UNIT_ASSERT_VALUES_EQUAL("+7 916 916\u201216\u201216", f(&p, "2"));
    }

    Y_UNIT_TEST(phoneE164Number) {
        TFixture fix;

        TSyntheticAttributes::TExtAttrFuncType f = fix.SyntheticAttrs.GetSyntheticExtPhoneFunc(*fix.Fetcher, TPhoneAttr::E164_NUMBER);
        UNIT_ASSERT(fix.Attrs().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Phoneattrs(),
                                 TDbProfile::TExtendedEntities({{"", TDbProfile::TAttrs({{TPhoneAttr::NUMBER, TDbValue()}})}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, ""));
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "1"));

        p.AddPhoneAttr("2", TPhoneAttr::NUMBER, "");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "2"));

        p.AddPhoneAttr("2", TPhoneAttr::NUMBER, "79169161616");
        UNIT_ASSERT_VALUES_EQUAL("+79169161616", f(&p, "2"));

        p.AddPhoneAttr("2", TPhoneAttr::NUMBER, "+79169161616");
        UNIT_ASSERT_VALUES_EQUAL("+79169161616", f(&p, "2"));
    }

    Y_UNIT_TEST(phoneMaskedFormattedNumber) {
        TFixture fix;

        TSyntheticAttributes::TExtAttrFuncType f = fix.SyntheticAttrs.GetSyntheticExtPhoneFunc(*fix.Fetcher, TPhoneAttr::MASKED_FORMATTED_NUMBER);
        UNIT_ASSERT(fix.Attrs().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Phoneattrs(),
                                 TDbProfile::TExtendedEntities({{"", TDbProfile::TAttrs({{TPhoneAttr::NUMBER, TDbValue()}})}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, ""));
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "1"));

        p.AddPhoneAttr("2", TPhoneAttr::NUMBER, "");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "2"));

        p.AddPhoneAttr("2", TPhoneAttr::NUMBER, "79169161616");
        UNIT_ASSERT_VALUES_EQUAL("+7 916 ***\u2012**\u201216", f(&p, "2"));
    }

    Y_UNIT_TEST(phoneMaskedE164Number) {
        TFixture fix;

        TSyntheticAttributes::TExtAttrFuncType f = fix.SyntheticAttrs.GetSyntheticExtPhoneFunc(*fix.Fetcher, TPhoneAttr::MASKED_E164_NUMBER);
        UNIT_ASSERT(fix.Attrs().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Phoneattrs(),
                                 TDbProfile::TExtendedEntities({{"", TDbProfile::TAttrs({{TPhoneAttr::NUMBER, TDbValue()}})}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, ""));
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "1"));

        p.AddPhoneAttr("2", TPhoneAttr::NUMBER, "");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "2"));

        p.AddPhoneAttr("2", TPhoneAttr::NUMBER, "7");
        UNIT_ASSERT_VALUES_EQUAL("+7", f(&p, "2"));

        p.AddPhoneAttr("2", TPhoneAttr::NUMBER, "79169");
        UNIT_ASSERT_VALUES_EQUAL("+***69", f(&p, "2"));

        p.AddPhoneAttr("2", TPhoneAttr::NUMBER, "79169161616333333");
        UNIT_ASSERT_VALUES_EQUAL("+7916916161*****33", f(&p, "2"));

        p.AddPhoneAttr("2", TPhoneAttr::NUMBER, "79169161616");
        UNIT_ASSERT_VALUES_EQUAL("+7916*****16", f(&p, "2"));

        p.AddPhoneAttr("2", TPhoneAttr::NUMBER, "+79169161616");
        UNIT_ASSERT_VALUES_EQUAL("+7916*****16", f(&p, "2"));
    }

    Y_UNIT_TEST(phoneIsConfirmed) {
        TFixture fix;

        TSyntheticAttributes::TExtAttrFuncType f = fix.SyntheticAttrs.GetSyntheticExtPhoneFunc(*fix.Fetcher, TPhoneAttr::IS_CONFIRMED);
        UNIT_ASSERT(fix.Attrs().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Phoneattrs(),
                                 TDbProfile::TExtendedEntities({{"", TDbProfile::TAttrs({{TPhoneAttr::CONFIRMED, TDbValue()}})}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, ""));
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "1"));

        p.AddPhoneAttr("2", TPhoneAttr::CONFIRMED, "");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "2"));

        p.AddPhoneAttr("2", TPhoneAttr::CONFIRMED, "0");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p, "2"));

        p.AddPhoneAttr("2", TPhoneAttr::CONFIRMED, "1");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p, "2"));
    }

    Y_UNIT_TEST(phoneIsBound) {
        TFixture fix;

        TSyntheticAttributes::TExtAttrFuncType f = fix.SyntheticAttrs.GetSyntheticExtPhoneFunc(*fix.Fetcher, TPhoneAttr::IS_BOUND);
        UNIT_ASSERT(fix.Attrs().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Phoneattrs(),
                                 TDbProfile::TExtendedEntities({{"", TDbProfile::TAttrs({{TPhoneAttr::BOUND, TDbValue()}})}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, ""));
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "1"));

        p.AddPhoneAttr("2", TPhoneAttr::BOUND, "");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "2"));

        p.AddPhoneAttr("2", TPhoneAttr::BOUND, "0");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p, "2"));

        p.AddPhoneAttr("2", TPhoneAttr::BOUND, "1");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p, "2"));
    }

    Y_UNIT_TEST(phoneIsDefault) {
        TFixture fix;

        TSyntheticAttributes::TExtAttrFuncType f = fix.SyntheticAttrs.GetSyntheticExtPhoneFunc(*fix.Fetcher, TPhoneAttr::IS_DEFAULT);
        UNIT_ASSERT_VALUES_EQUAL(fix.Attrs(), TDbProfile::TAttrs({{TAttr::PHONES_DEFAULT, TDbValue()}}));
        UNIT_ASSERT_VALUES_EQUAL(fix.Phoneattrs(),
                                 TDbProfile::TExtendedEntities({{"", TDbProfile::TAttrs({{TPhoneAttr::NUMBER, TDbValue()},
                                                                                         {TPhoneAttr::BOUND, TDbValue()}})}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        p.DefaultPhoneId_ = "3";
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, ""));
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "1"));

        p.AddPhoneAttr("2", TPhoneAttr::NUMBER, "");
        p.AddPhoneAttr("3", TPhoneAttr::NUMBER, "");

        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "2"));
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p, "3"));
    }

    Y_UNIT_TEST(phoneIsSecured) {
        TFixture fix;

        TSyntheticAttributes::TExtAttrFuncType f = fix.SyntheticAttrs.GetSyntheticExtPhoneFunc(*fix.Fetcher, TPhoneAttr::IS_SECURED);
        UNIT_ASSERT(fix.Attrs().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Phoneattrs(),
                                 TDbProfile::TExtendedEntities({{"", TDbProfile::TAttrs({{TPhoneAttr::SECURED, TDbValue()}})}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, ""));
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "1"));

        p.AddPhoneAttr("2", TPhoneAttr::SECURED, "");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "2"));

        p.AddPhoneAttr("2", TPhoneAttr::SECURED, "0");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p, "2"));

        p.AddPhoneAttr("2", TPhoneAttr::SECURED, "1");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p, "2"));

        p.ExtendedPhones_["2"].clear();
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "2"));
    }

    Y_UNIT_TEST(phoneIsBank) {
        TFixture fix;

        TSyntheticAttributes::TExtAttrFuncType f = fix.SyntheticAttrs.GetSyntheticExtPhoneFunc(*fix.Fetcher, TPhoneAttr::IS_BANK);
        UNIT_ASSERT(fix.Attrs().empty());
        UNIT_ASSERT_VALUES_EQUAL(fix.Aliases(), TDbProfile::TAliases({{TAlias::BANK_PHONE_NUMBER, TDbValue()}}));
        UNIT_ASSERT_VALUES_EQUAL(fix.Phoneattrs(),
                                 TDbProfile::TExtendedEntities({{"", TDbProfile::TAttrs({{TPhoneAttr::NUMBER, TDbValue()}})}}));

        TTestDbProfile p(fix.Fetcher->DefaultProfile());
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, ""));
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "1"));

        p.AddPhoneAttr("5", TPhoneAttr::NUMBER, "79991112233");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "5"));

        p.AddAlias(TAlias::BANK_PHONE_NUMBER, "71112223344");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "5"));

        p.AddAlias(TAlias::BANK_PHONE_NUMBER, "+79991112233");
        UNIT_ASSERT_VALUES_EQUAL("", f(&p, "5"));

        p.AddAlias(TAlias::BANK_PHONE_NUMBER, "79991112233");
        UNIT_ASSERT_VALUES_EQUAL("1", f(&p, "5"));
    }

}
