#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/blackbox.h>
#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/lcookie_parser.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/misc/ya_domains.h>
#include <passport/infra/daemons/blackbox/src/output/lcookie_result.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NBb;

Y_UNIT_TEST_SUITE(PasspBbUtils) {
    Y_UNIT_TEST(legacyPhoneHeuristics) {
        // generic
        UNIT_ASSERT_VALUES_EQUAL("123", TUtils::NormalizePhoneByLegacyWay("123", ""));
        UNIT_ASSERT_VALUES_EQUAL("123456789", TUtils::NormalizePhoneByLegacyWay("123456789", ""));

        // ru
        UNIT_ASSERT_VALUES_EQUAL("79231234567", TUtils::NormalizePhoneByLegacyWay("79231234567", ""));
        UNIT_ASSERT_VALUES_EQUAL("80231234567", TUtils::NormalizePhoneByLegacyWay("80231234567", ""));
        UNIT_ASSERT_VALUES_EQUAL("71231234567", TUtils::NormalizePhoneByLegacyWay("81231234567", ""));
        UNIT_ASSERT_VALUES_EQUAL("79231234567", TUtils::NormalizePhoneByLegacyWay("89231234567", ""));
        UNIT_ASSERT_VALUES_EQUAL("79231234567", TUtils::NormalizePhoneByLegacyWay("89231234567", "yandex.com"));
        UNIT_ASSERT_VALUES_EQUAL("80231234567", TUtils::NormalizePhoneByLegacyWay("80231234567", "yandex.com"));
        UNIT_ASSERT_VALUES_EQUAL("79231234567", TUtils::NormalizePhoneByLegacyWay("89231234567", "yandex.ua"));
        UNIT_ASSERT_VALUES_EQUAL("80231234567", TUtils::NormalizePhoneByLegacyWay("80231234567", "yandex.ua"));
        UNIT_ASSERT_VALUES_EQUAL("79231234567", TUtils::NormalizePhoneByLegacyWay("89231234567", "narod.ru"));
        UNIT_ASSERT_VALUES_EQUAL("80231234567", TUtils::NormalizePhoneByLegacyWay("80231234567", "narod.ru"));
        UNIT_ASSERT_VALUES_EQUAL("79231234567", TUtils::NormalizePhoneByLegacyWay("89231234567", "yandex.by"));

        // ua
        UNIT_ASSERT_VALUES_EQUAL("0012345678", TUtils::NormalizePhoneByLegacyWay("0012345678", ""));
        UNIT_ASSERT_VALUES_EQUAL("0012345678", TUtils::NormalizePhoneByLegacyWay("0012345678", "yandex.ua"));
        UNIT_ASSERT_VALUES_EQUAL("0123456789", TUtils::NormalizePhoneByLegacyWay("0123456789", ""));
        UNIT_ASSERT_VALUES_EQUAL("380123456789", TUtils::NormalizePhoneByLegacyWay("0123456789", "yandex.ua"));
        UNIT_ASSERT_VALUES_EQUAL("0123456789", TUtils::NormalizePhoneByLegacyWay("0123456789", "yandex.com"));

        // by
        UNIT_ASSERT_VALUES_EQUAL("012345678", TUtils::NormalizePhoneByLegacyWay("012345678", ""));
        UNIT_ASSERT_VALUES_EQUAL("012345678", TUtils::NormalizePhoneByLegacyWay("012345678", "yandex.by"));
        UNIT_ASSERT_VALUES_EQUAL("375512345678", TUtils::NormalizePhoneByLegacyWay("512345678", "yandex.by"));
        UNIT_ASSERT_VALUES_EQUAL("8012345678", TUtils::NormalizePhoneByLegacyWay("8012345678", "yandex.by"));
        UNIT_ASSERT_VALUES_EQUAL("8112345678", TUtils::NormalizePhoneByLegacyWay("8112345678", "yandex.by"));
        UNIT_ASSERT_VALUES_EQUAL("375112345678", TUtils::NormalizePhoneByLegacyWay("0112345678", "yandex.by"));
        UNIT_ASSERT_VALUES_EQUAL("375112345678", TUtils::NormalizePhoneByLegacyWay("80112345678", "yandex.by"));

        // kz
        UNIT_ASSERT_VALUES_EQUAL("77231234567", TUtils::NormalizePhoneByLegacyWay("77231234567", ""));
        UNIT_ASSERT_VALUES_EQUAL("77231234567", TUtils::NormalizePhoneByLegacyWay("7231234567", ""));
        UNIT_ASSERT_VALUES_EQUAL("77231234567", TUtils::NormalizePhoneByLegacyWay("7231234567", "yandex.ru"));
        UNIT_ASSERT_VALUES_EQUAL("77231234567", TUtils::NormalizePhoneByLegacyWay("7231234567", "yandex.com"));

        // tr
        UNIT_ASSERT_VALUES_EQUAL("05123456789", TUtils::NormalizePhoneByLegacyWay("05123456789", ""));
        UNIT_ASSERT_VALUES_EQUAL("05123456789", TUtils::NormalizePhoneByLegacyWay("05123456789", "yandex.ru"));
        UNIT_ASSERT_VALUES_EQUAL("905123456789", TUtils::NormalizePhoneByLegacyWay("05123456789", "yandex.com"));
        UNIT_ASSERT_VALUES_EQUAL("905123456789", TUtils::NormalizePhoneByLegacyWay("05123456789", "yandex.com.tr"));

        // etc
        UNIT_ASSERT_VALUES_EQUAL("79231234567", TUtils::NormalizePhoneByLegacyWay("89231234567", "narod.ru"));
        UNIT_ASSERT_VALUES_EQUAL("79231234567", TUtils::NormalizePhoneByLegacyWay("89231234567", "yandex.by"));
        UNIT_ASSERT_VALUES_EQUAL("380123456789", TUtils::NormalizePhoneByLegacyWay("0123456789", "yandex.ua"));
        UNIT_ASSERT_VALUES_EQUAL("375112345678", TUtils::NormalizePhoneByLegacyWay("0112345678", "yandex.by"));
        UNIT_ASSERT_VALUES_EQUAL("375112345678", TUtils::NormalizePhoneByLegacyWay("80112345678", "yandex.by"));
        UNIT_ASSERT_VALUES_EQUAL("77231234567", TUtils::NormalizePhoneByLegacyWay("7231234567", "yandex.com"));
        UNIT_ASSERT_VALUES_EQUAL("905123456789", TUtils::NormalizePhoneByLegacyWay("05123456789", "yandex.com"));
        UNIT_ASSERT_VALUES_EQUAL("905123456789", TUtils::NormalizePhoneByLegacyWay("05123456789", "yandex.com.tr"));
    }

    Y_UNIT_TEST(getDigitsOnly) {
        UNIT_ASSERT_VALUES_EQUAL("123", TUtils::GetDigitsOnly(" (1+2+3) "));
        UNIT_ASSERT_VALUES_EQUAL("123", TUtils::GetDigitsOnly("(1)  2 - 3      "));
        UNIT_ASSERT_VALUES_EQUAL("89231234567", TUtils::GetDigitsOnly("8-(923)-123-45-67"));
        UNIT_ASSERT_VALUES_EQUAL("89231234567", TUtils::GetDigitsOnly(" (8923) 123 45 67"));
        UNIT_ASSERT_VALUES_EQUAL("0123456789", TUtils::GetDigitsOnly("+ 0123 (456)789"));
        UNIT_ASSERT_VALUES_EQUAL("0112345678", TUtils::GetDigitsOnly("+(0112)345 678"));
        UNIT_ASSERT_VALUES_EQUAL("80112345678", TUtils::GetDigitsOnly("' 801_12_34_56_78'"));
        UNIT_ASSERT_VALUES_EQUAL("7231234567", TUtils::GetDigitsOnly("__7231234567__"));
        UNIT_ASSERT_VALUES_EQUAL("05123456789", TUtils::GetDigitsOnly("[05](123)456{789}"));
        UNIT_ASSERT_VALUES_EQUAL("05123456789", TUtils::GetDigitsOnly("*0\"5#1=2%3&4$5_6'7/8\\9"));
    }

    Y_UNIT_TEST(normalizeLogin) {

        UNIT_ASSERT_VALUES_EQUAL(ESanitizeStatus::Ok, TUtils::SanitizeLogin(""));
        UNIT_ASSERT_VALUES_EQUAL(ESanitizeStatus::Ok, TUtils::SanitizeLogin("login"));
        UNIT_ASSERT_VALUES_EQUAL(ESanitizeStatus::Ok, TUtils::SanitizeLogin("login@yandex.ru"));
        UNIT_ASSERT_VALUES_EQUAL(ESanitizeStatus::Ok, TUtils::SanitizeLogin("+7 916 123-45-67"));
        UNIT_ASSERT_VALUES_EQUAL(ESanitizeStatus::InvalidChars, TUtils::SanitizeLogin("ab\ncd"));
        UNIT_ASSERT_VALUES_EQUAL(ESanitizeStatus::InvalidChars, TUtils::SanitizeLogin("lo\x01gin"));
        UNIT_ASSERT_VALUES_EQUAL(ESanitizeStatus::Ok, TUtils::SanitizeLogin(TString(255, 'a')));
        UNIT_ASSERT_VALUES_EQUAL(ESanitizeStatus::TooLong, TUtils::SanitizeLogin(TString(256, 'a')));

        UNIT_ASSERT_VALUES_EQUAL("", TUtils::NormalizeLogin(""));
        UNIT_ASSERT_VALUES_EQUAL("user-vasya", TUtils::NormalizeLogin("User-vasya"));
        UNIT_ASSERT_VALUES_EQUAL("a-s-pushkin", TUtils::NormalizeLogin("A.S.Pushkin"));
    }

    Y_UNIT_TEST(normalizePhone) {
        TYandexDomains domains;
        UNIT_ASSERT_VALUES_EQUAL("123", TUtils::NormalizePhone(" (1+2+3) ", "", domains));
        UNIT_ASSERT_VALUES_EQUAL("123", TUtils::NormalizePhone("(1)  2 - 3      ", "", domains));
        UNIT_ASSERT_VALUES_EQUAL("79231234567", TUtils::NormalizePhone("8-(923)-123-45-67", "narod.ru", domains));
        UNIT_ASSERT_VALUES_EQUAL("79231234567", TUtils::NormalizePhone(" (8923) 123 45 67", "yandex.by", domains));
        UNIT_ASSERT_VALUES_EQUAL("380123456789", TUtils::NormalizePhone("+ 0123 (456)789", "yandex.ua", domains));
        UNIT_ASSERT_VALUES_EQUAL("375112345678", TUtils::NormalizePhone("+(0112)345 678", "yandex.by", domains));
        UNIT_ASSERT_VALUES_EQUAL("375112345678", TUtils::NormalizePhone("' 801_12_34_56_78'", "yandex.by", domains));
        UNIT_ASSERT_VALUES_EQUAL("77231234567", TUtils::NormalizePhone("__7231234567__", "yandex.com", domains));
        UNIT_ASSERT_VALUES_EQUAL("905123456789", TUtils::NormalizePhone("[05](123)456{789}", "yandex.com", domains));
        UNIT_ASSERT_VALUES_EQUAL("905123456789", TUtils::NormalizePhone("*0\"5#1=2%3&4$5_6'7/8\\9", "yandex.com.tr", domains));

        UNIT_ASSERT_VALUES_EQUAL("0554123456", TUtils::NormalizePhone("0554123456", "", domains));
        UNIT_ASSERT_VALUES_EQUAL("0554123456", TUtils::NormalizePhone("0554123456", "yandex.kg", domains));
        TYandexDomains::TDomainData d("yandex.kg");
        d.SetPhoneCountry("kg");
        domains.Insert(d);
        UNIT_ASSERT_VALUES_EQUAL("0554123456", TUtils::NormalizePhone("0554123456", "", domains));
        UNIT_ASSERT_VALUES_EQUAL("0554123456", TUtils::NormalizePhone("0554123456", "yandex.ru", domains));
        UNIT_ASSERT_VALUES_EQUAL("996554123456", TUtils::NormalizePhone("0554123456", "yandex.kg", domains));

        UNIT_ASSERT_VALUES_EQUAL("996554123456", TUtils::NormalizePhone("0554123456", "", domains, "kg"));
        UNIT_ASSERT_VALUES_EQUAL("0554123456", TUtils::NormalizePhone("0554123456", "ololo.com", domains, "kg"));
    }

    TYandexDomains GetDomains() {
        TYandexDomains domains;

        domains.Insert("galatasaray.net");
        domains.Insert("narod.ru");
        domains.Insert("yandex.com");

        {
            TYandexDomains::TDomainData d("yandex.us");
            d.SetPhoneCountry("us");
            domains.Insert(d);
        }
        {
            TYandexDomains::TDomainData d("ya.ru");
            d.SetPhoneCountry("ru");
            domains.Insert(d);
        }
        {
            TYandexDomains::TDomainData d("yandex.az");
            d.SetPhoneCountry("az");
            domains.Insert(d);
        }
        {
            TYandexDomains::TDomainData d("yandex.by");
            d.SetPhoneCountry("by");
            domains.Insert(d);
        }
        {
            TYandexDomains::TDomainData d("yandex.co.il");
            d.SetPhoneCountry("il");
            domains.Insert(d);
        }
        {
            TYandexDomains::TDomainData d("yandex.com.am");
            d.SetPhoneCountry("am");
            domains.Insert(d);
        }
        {
            TYandexDomains::TDomainData d("yandex.com.ge");
            d.SetPhoneCountry("ge");
            domains.Insert(d);
        }
        {
            TYandexDomains::TDomainData d("yandex.com.tr");
            d.SetPhoneCountry("tr");
            domains.Insert(d);
        }
        {
            TYandexDomains::TDomainData d("yandex.ee");
            d.SetPhoneCountry("ee");
            domains.Insert(d);
        }
        {
            TYandexDomains::TDomainData d("yandex.fr");
            d.SetPhoneCountry("fr");
            domains.Insert(d);
        }
        {
            TYandexDomains::TDomainData d("yandex.kg");
            d.SetPhoneCountry("kg");
            domains.Insert(d);
        }
        {
            TYandexDomains::TDomainData d("yandex.kz");
            d.SetPhoneCountry("kz");
            domains.Insert(d);
        }
        {
            TYandexDomains::TDomainData d("yandex.lt");
            d.SetPhoneCountry("lt");
            domains.Insert(d);
        }
        {
            TYandexDomains::TDomainData d("yandex.lv");
            d.SetPhoneCountry("lv");
            domains.Insert(d);
        }
        {
            TYandexDomains::TDomainData d("yandex.md");
            d.SetPhoneCountry("md");
            domains.Insert(d);
        }
        {
            TYandexDomains::TDomainData d("yandex.ru");
            d.SetPhoneCountry("ru");
            domains.Insert(d);
        }
        {
            TYandexDomains::TDomainData d("yandex.tj");
            d.SetPhoneCountry("tj");
            domains.Insert(d);
        }
        {
            TYandexDomains::TDomainData d("yandex.tm");
            d.SetPhoneCountry("tm");
            domains.Insert(d);
        }
        {
            TYandexDomains::TDomainData d("yandex.ua");
            d.SetPhoneCountry("ua");
            domains.Insert(d);
        }
        {
            TYandexDomains::TDomainData d("yandex.uz");
            d.SetPhoneCountry("uz");
            domains.Insert(d);
        }
        {
            TYandexDomains::TDomainData d("яндекс.рф");
            d.SetPhoneCountry("ru");
            domains.Insert(d);
        }

        return domains;
    }
    const std::set<TString> STRANGE_DOMAINS = {
        "ololo2.ua",
        "ololo.com",
        "ya.kz",
    };
    const std::set<TString> COUNTRIES = {
        "am",
        "az",
        "by",
        "ee",
        "fr",
        "ge",
        "il",
        "kg",
        "kz",
        "lt",
        "lv",
        "md",
        "ru",
        "tj",
        "tm",
        "tr",
        "ua",
        "us",
        "uz",
        "junk",
    };

    Y_UNIT_TEST(normalizePhone_Russia) {
        const TYandexDomains domains = GetDomains();

        for (const TString& c : COUNTRIES) {
            UNIT_ASSERT_VALUES_EQUAL("79026411724", TUtils::NormalizePhone("+7 902 64 117 24", "", domains, c));
            UNIT_ASSERT_VALUES_EQUAL("79026411724", TUtils::NormalizePhone("7 902 64 117 24", "", domains, c));
            UNIT_ASSERT_VALUES_EQUAL("79026411724", TUtils::NormalizePhone("8 902 64 117 24", "", domains, c));
        }

        for (const TString& c : COUNTRIES) {
            for (const TString& d : domains) {
                UNIT_ASSERT_VALUES_EQUAL("79026411724", TUtils::NormalizePhone("+7 902 64 117 24", d, domains, c));
                UNIT_ASSERT_VALUES_EQUAL("79026411724", TUtils::NormalizePhone("7 902 64 117 24", d, domains, c));
                UNIT_ASSERT_VALUES_EQUAL("79026411724", TUtils::NormalizePhone("8 902 64 117 24", d, domains, c));
            }
            for (const TString& d : STRANGE_DOMAINS) {
                UNIT_ASSERT_VALUES_EQUAL("79026411724", TUtils::NormalizePhone("+7 902 64 117 24", d, domains, c));
                UNIT_ASSERT_VALUES_EQUAL("79026411724", TUtils::NormalizePhone("7 902 64 117 24", d, domains, c));
                UNIT_ASSERT_VALUES_EQUAL("79026411724", TUtils::NormalizePhone("8 902 64 117 24", d, domains, c));
            }
        }

        for (const TString& d : domains) {
            UNIT_ASSERT_VALUES_EQUAL("79026411724", TUtils::NormalizePhone("+7 902 64 117 24", d, domains, ""));
            UNIT_ASSERT_VALUES_EQUAL("79026411724", TUtils::NormalizePhone("7 902 64 117 24", d, domains, ""));
            UNIT_ASSERT_VALUES_EQUAL("79026411724", TUtils::NormalizePhone("8 902 64 117 24", d, domains, ""));
        }
        for (const TString& d : STRANGE_DOMAINS) {
            UNIT_ASSERT_VALUES_EQUAL("79026411724", TUtils::NormalizePhone("+7 902 64 117 24", d, domains, ""));
            UNIT_ASSERT_VALUES_EQUAL("79026411724", TUtils::NormalizePhone("7 902 64 117 24", d, domains, ""));
            UNIT_ASSERT_VALUES_EQUAL("79026411724", TUtils::NormalizePhone("8 902 64 117 24", d, domains, ""));
        }
    }

    Y_UNIT_TEST(normalizePhone_Kazakhstan) {
        const TYandexDomains domains = GetDomains();

        for (const TString& c : COUNTRIES) {
            UNIT_ASSERT_VALUES_EQUAL("77011234567", TUtils::NormalizePhone("7011234567", "", domains, c));
        }

        for (const TString& c : COUNTRIES) {
            for (const TString& d : domains) {
                UNIT_ASSERT_VALUES_EQUAL("77011234567", TUtils::NormalizePhone("7011234567", d, domains, c));
            }
            for (const TString& d : STRANGE_DOMAINS) {
                UNIT_ASSERT_VALUES_EQUAL("77011234567", TUtils::NormalizePhone("7011234567", d, domains, c));
            }
        }

        for (const TString& d : domains) {
            UNIT_ASSERT_VALUES_EQUAL("77011234567", TUtils::NormalizePhone("7011234567", d, domains, ""));
        }
        for (const TString& d : STRANGE_DOMAINS) {
            UNIT_ASSERT_VALUES_EQUAL("77011234567", TUtils::NormalizePhone("7011234567", d, domains, ""));
        }
    }

    Y_UNIT_TEST(normalizePhone_Turkey) {
        const TYandexDomains domains = GetDomains();

        for (const TString& c : COUNTRIES) {
            UNIT_ASSERT_VALUES_EQUAL("905312345678", TUtils::NormalizePhone("05312345678", "yandex.com", domains, c));
            UNIT_ASSERT_VALUES_EQUAL("905312345678", TUtils::NormalizePhone("05312345678", "yandex.com.tr", domains, c));
        }
        UNIT_ASSERT_VALUES_EQUAL("905312345678", TUtils::NormalizePhone("05312345678", "yandex.com", domains, ""));
        UNIT_ASSERT_VALUES_EQUAL("905312345678", TUtils::NormalizePhone("05312345678", "yandex.com.tr", domains, ""));

        for (const TString& c : COUNTRIES) {
            for (const TString& d : domains) {
                if (d == "yandex.com" || d == "yandex.com.tr") {
                    UNIT_ASSERT_VALUES_EQUAL("905312345678", TUtils::NormalizePhone("05312345678", d, domains, c));
                } else {
                    UNIT_ASSERT_VALUES_EQUAL("05312345678", TUtils::NormalizePhone("05312345678", d, domains, c));
                }
            }
            for (const TString& d : STRANGE_DOMAINS) {
                UNIT_ASSERT_VALUES_EQUAL("05312345678", TUtils::NormalizePhone("05312345678", d, domains, c));
            }
        }
    }

    Y_UNIT_TEST(normalizePhone_Etc) {
        const TYandexDomains domains = GetDomains();

        for (const TString& c : COUNTRIES) {
            UNIT_ASSERT_VALUES_EQUAL("15139831100", TUtils::NormalizePhone("+1 513 983 11 00", "", domains, c));
            UNIT_ASSERT_VALUES_EQUAL("15139831100", TUtils::NormalizePhone("1 513 983 11 00", "", domains, c));

            if (c == "us") {
                UNIT_ASSERT_VALUES_EQUAL("15139831100", TUtils::NormalizePhone("513 983 11 00", "", domains, c));
            } else {
                UNIT_ASSERT_VALUES_EQUAL("5139831100", TUtils::NormalizePhone("513 983 11 00", "", domains, c));
            }
        }

        for (const TString& c : COUNTRIES) {
            for (const TString& d : domains) {
                UNIT_ASSERT_VALUES_EQUAL("15139831100", TUtils::NormalizePhone("+1 513 983 11 00", d, domains, c));
                UNIT_ASSERT_VALUES_EQUAL("15139831100", TUtils::NormalizePhone("1 513 983 11 00", d, domains, c));

                if (d == "yandex.us") {
                    UNIT_ASSERT_VALUES_EQUAL("15139831100", TUtils::NormalizePhone("513 983 11 00", d, domains, c));
                } else {
                    UNIT_ASSERT_VALUES_EQUAL("5139831100", TUtils::NormalizePhone("513 983 11 00", d, domains, c));
                }
            }
            for (const TString& d : STRANGE_DOMAINS) {
                UNIT_ASSERT_VALUES_EQUAL("15139831100", TUtils::NormalizePhone("+1 513 983 11 00", d, domains, c));
                UNIT_ASSERT_VALUES_EQUAL("15139831100", TUtils::NormalizePhone("1 513 983 11 00", d, domains, c));
            }
        }

        for (const TString& d : domains) {
            UNIT_ASSERT_VALUES_EQUAL("15139831100", TUtils::NormalizePhone("+1 513 983 11 00", d, domains, ""));
            UNIT_ASSERT_VALUES_EQUAL("15139831100", TUtils::NormalizePhone("1 513 983 11 00", d, domains, ""));

            if (d == "yandex.us") {
                UNIT_ASSERT_VALUES_EQUAL("15139831100", TUtils::NormalizePhone("513 983 11 00", d, domains, ""));
            } else {
                UNIT_ASSERT_VALUES_EQUAL("5139831100", TUtils::NormalizePhone("513 983 11 00", d, domains, ""));
            }
        }
        for (const TString& d : STRANGE_DOMAINS) {
            UNIT_ASSERT_VALUES_EQUAL("15139831100", TUtils::NormalizePhone("+1 513 983 11 00", d, domains, ""));
            UNIT_ASSERT_VALUES_EQUAL("15139831100", TUtils::NormalizePhone("1 513 983 11 00", d, domains, ""));
            UNIT_ASSERT_VALUES_EQUAL("5139831100", TUtils::NormalizePhone("513 983 11 00", d, domains, ""));
        }
    }

    Y_UNIT_TEST_F(lcookie, TBlackboxFixture) {
        const NPassport::NAuth::TSessionSigner& sessionSigner = Bb->Impl->SessionSigner();
        std::unique_ptr<TLCookieResult> res;

        res = TLCookieParser::Parse("UlgKAllTVFBQUFBQUFBQUFBQWVZZWFdYDgoNUwcLBBYXUg4HAlAbBQVYCgYYAA8FBBlMDQ4GCA8MDVQVEQkZGw==.1400761800.1.221453.39035f3f91da75c8a56b4c1d0165970c",
                                    sessionSigner);
        UNIT_ASSERT_VALUES_EQUAL("111111111", res->Uid);
        UNIT_ASSERT_VALUES_EQUAL("yandex-login", res->Login);
        UNIT_ASSERT_VALUES_EQUAL("yandex-login", res->DisplayLogin);
        UNIT_ASSERT_VALUES_EQUAL("1400761800", res->Timestamp);

        res = TLCookieParser::Parse("UVhRK1NTU1NTU1NTU1ZTUVRZWFdVWVRSEhQRBBNMERQRBBNMBBkVEwBMFwQTGEwXBBMYTA0ODwZMGAAPBQQZTA0OBggP.1400761964.1.391225.bec7457ed3a1f3ab51746c9fe8004dab",
                                    sessionSigner);
        UNIT_ASSERT_VALUES_EQUAL("222222222", res->Uid);
        UNIT_ASSERT_VALUES_EQUAL("super-puper-extra-very-very-long-yandex-login", res->Login);
        UNIT_ASSERT_VALUES_EQUAL("super-puper-extra-very-very-long-yandex-login", res->DisplayLogin);
        UNIT_ASSERT_VALUES_EQUAL("1400761964", res->Timestamp);

        res = TLCookieParser::Parse("UltSKFBQUFBQUFBQUFpRWltUW1ZXVVNbERcSBxBPEhcSBxBPBxoWEANPFAcQG08UBxAbTw4NDAVPGwMMBgcaTw4NBQsM.1400762446.2.338823.f2e243723a1e610e4f3291e44447bbe3",
                                    sessionSigner);
        UNIT_ASSERT_VALUES_EQUAL("222222222", res->Uid);
        UNIT_ASSERT_VALUES_EQUAL("super-puper-extra-very-very-long-yandex-login", res->Login);
        UNIT_ASSERT_VALUES_EQUAL("super-puper-extra-very-very-long-yandex-login", res->DisplayLogin);
        UNIT_ASSERT_VALUES_EQUAL("1400762446", res->Timestamp);

        res = TLCookieParser::Parse("UQZRA1BQUlFRUVFRUVFRUVFRUVBYVlRTEQUFIQ4KDwBPExQ=.1400763784.1.397116.6eef25a754eb752dd8de4c2cabdba1ef",
                                    sessionSigner);
        UNIT_ASSERT_VALUES_EQUAL("1130000000000001", res->Uid);
        UNIT_ASSERT_VALUES_EQUAL("pdd@okna.ru", res->Login);
        UNIT_ASSERT_VALUES_EQUAL("pdd@okna.ru", res->DisplayLogin);
        UNIT_ASSERT_VALUES_EQUAL("1400763784", res->Timestamp);

        res = TLCookieParser::Parse("UQZRCVBQUlFRUVFRUVFRUVFRUVNXUFdSEQUFIbHfsdux3LHRT7DhsOU=.1400763929.1.361033.3f9a2f2551a8da2a2c7ce7e0cf1d4b33",
                                    sessionSigner);
        UNIT_ASSERT_VALUES_EQUAL("1130000000000002", res->Uid);
        UNIT_ASSERT_VALUES_EQUAL("pdd@окна.рф", res->Login);
        UNIT_ASSERT_VALUES_EQUAL("pdd@окна.рф", res->DisplayLogin);
        UNIT_ASSERT_VALUES_EQUAL("1400763929", res->Timestamp);

        res = TLCookieParser::Parse("UVU9WnZ9f0hbeHlVWlh9YngGCGJ6WVl1XjNZXGFeNwZJHA0FDTszX1kBMRImKV4SUiRAFQ8YDRsrKmBjREJXYA==.1376939944.1002323.216145.e46392a735b0d198614ac9cb321d79b4",
                                    sessionSigner);
        UNIT_ASSERT_VALUES_EQUAL("1", res->Uid);
        UNIT_ASSERT_VALUES_EQUAL("username-124", res->Login);
        UNIT_ASSERT_VALUES_EQUAL("username-124", res->DisplayLogin);
        UNIT_ASSERT_VALUES_EQUAL("1376939944", res->Timestamp);

        res = TLCookieParser::Parse("AgN+Xn56eEFTe3hSXFp7YXkHAGV8VFdxBjVFSzMcKw9eDy9aAi8oGEASZFQ6J0AcFyJaXloMDxc2Nn9nEU9WZg==.1376939981.1002323.273612.eec41a4a89676bae03d50e35a27896b0",
                                    sessionSigner);
        UNIT_ASSERT_VALUES_EQUAL("1130000000000005", res->Uid);
        UNIT_ASSERT_VALUES_EQUAL("username@okna.ru", res->Login);
        UNIT_ASSERT_VALUES_EQUAL("username@okna.ru", res->DisplayLogin);
        UNIT_ASSERT_VALUES_EQUAL("1376939981", res->Timestamp);

        res = TLCookieParser::Parse("AAN/SHZ8ekBQe3hSXFp7YXkHAGB9VFhwTmNZ6Oi+/76Out91uMCZiOLWgNWT6+XKSZi1t+4TUhs1KnVgERgRJw==.1376939998.1002323.264175.e11678d6c6178f606d35324a78deea08",
                                    sessionSigner);
        UNIT_ASSERT_VALUES_EQUAL("1130000000000005", res->Uid);
        UNIT_ASSERT_VALUES_EQUAL("окна@собака.рф", res->Login);
        UNIT_ASSERT_VALUES_EQUAL("окна@собака.рф", res->DisplayLogin);
        UNIT_ASSERT_VALUES_EQUAL("1376939998", res->Timestamp);

        res = TLCookieParser::Parse("VSNkVH5GUgNSUWtncVZhakVVekJrU2dCECUdRCENOAs3BghC.1422888128.959.348353.5e27a0a2f225b0472809e65e8b3fb71a",
                                    sessionSigner);
        UNIT_ASSERT_VALUES_EQUAL("4000521239", res->Uid);
        UNIT_ASSERT_VALUES_EQUAL("uid-mkjaydbq", res->Login);
        UNIT_ASSERT_VALUES_EQUAL("", res->DisplayLogin);
        UNIT_ASSERT_VALUES_EQUAL("1422888128", res->Timestamp);

        res = TLCookieParser::Parse("VSNkVH5GUgNSUWtnd1pgbUBUfUdgVmJAFSQXDGEcPhAjA11LbQ==.1422888199.959.310493.3aa1ff91b626c7d4528bdfcb4fca0dec",
                                    sessionSigner);
        UNIT_ASSERT_VALUES_EQUAL("4000521255", res->Uid);
        UNIT_ASSERT_VALUES_EQUAL("phne-zlzma7x5", res->Login);
        UNIT_ASSERT_VALUES_EQUAL("", res->DisplayLogin);
        UNIT_ASSERT_VALUES_EQUAL("1422888199", res->Timestamp);

        UNIT_ASSERT_EXCEPTION_CONTAINS(TLCookieParser::Parse("", sessionSigner), TBlackboxError, "Cookie l body has broken format");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TLCookieParser::Parse("abc.def", sessionSigner), TBlackboxError, "Cookie l body has broken format");
        // UNIT_ASSERT_EXCEPTION_CONTAINS(LCookieParser::parse("UVAqDHt5eUJRfHpRWFp9aHgFAyN9UFF3RDBZSW8cdVhVXgVAGihxWl0AOQs2MFkMAC9bFxlYW0I/PTs3EhMKJg==.1376941096.1002323.76558.3233cfa606cec5b84083e0fac875a27f"), BlackboxError, "Cookie l body has broken format");;
        // UNIT_ASSERT_EXCEPTION_CONTAINS(LCookieParser::parse("A1ctDHd6eUJbeXFTX1x5YXkFA2N1VlZwWDBdSj4dcBoDUwNaDignUFAVOx0yIQYQX3sCHl1cEAB0filjQ0YGaFsDRwVD.1376941582.1002323.10274.6a482adeb63a6eab776b90603daee890"), BlackboxError, "Cookie l body has broken format");;
        UNIT_ASSERT_EXCEPTION_CONTAINS(TLCookieParser::Parse("AgN+Xn56eEFTe3hSXFp7YXkHAGV8VFdxBjVFSzMcKw9eDy9aAi8oGEASZFQ6J0AcFyJaXloMDxc2Nn9nEU9WZg==.1376939981.1002320.273612.eec41a4a89676bae03d50e35a27896b0",
                                                             sessionSigner),
                                       TBlackboxError, "Key not found in keyspace cookiel");
    }

    Y_UNIT_TEST(maskPhone) {
        UNIT_ASSERT_VALUES_EQUAL("012*****89", TUtils::MaskPhoneNumber("0123456789"));
        UNIT_ASSERT_VALUES_EQUAL("***89", TUtils::MaskPhoneNumber("56789"));
        UNIT_ASSERT_VALUES_EQUAL("0 1  2*  **  ** 8  9   ", TUtils::MaskPhoneNumber("0 1  23  45  67 8  9   "));
        UNIT_ASSERT_VALUES_EQUAL("* -* ) * () 8 - 9", TUtils::MaskPhoneNumber("5 -6 ) 7 () 8 - 9"));
        UNIT_ASSERT_VALUES_EQUAL("+7 903 ***-**-33", TUtils::MaskPhoneNumber("+7 903 222-11-33"));
    }

    template <class T>
    TString Tos(T num) {
        ui64 n = num;
        TString s;
        for (size_t idx = 0; idx < sizeof(T); ++idx) {
            s.push_back(n & 0xFF);
            n >>= 8;
        }
        std::reverse(s.begin(), s.vend());
        return s;
    }

    Y_UNIT_TEST(isSyntheticLogin) {
        UNIT_ASSERT(TUtils::IsSyntheticLogin("uid-mvlmxmbp"));
        UNIT_ASSERT(TUtils::IsSyntheticLogin("phne-6o5llczm"));
        UNIT_ASSERT(TUtils::IsSyntheticLogin("yambot-chmauyok"));
        UNIT_ASSERT(TUtils::IsSyntheticLogin("kolonkish-mwgg4uxm"));
        UNIT_ASSERT(TUtils::IsSyntheticLogin("uid-"));
        UNIT_ASSERT(TUtils::IsSyntheticLogin("phne-"));
        UNIT_ASSERT(TUtils::IsSyntheticLogin("yambot-"));
        UNIT_ASSERT(TUtils::IsSyntheticLogin("kolonkish-"));
        UNIT_ASSERT(!TUtils::IsSyntheticLogin("uid-suid@mail.ru"));
        UNIT_ASSERT(!TUtils::IsSyntheticLogin("phne-phne@gmail.com"));
        UNIT_ASSERT(!TUtils::IsSyntheticLogin("yambot-@"));
        UNIT_ASSERT(!TUtils::IsSyntheticLogin("kolonkish-@asdf"));
        UNIT_ASSERT(!TUtils::IsSyntheticLogin("uid"));
        UNIT_ASSERT(!TUtils::IsSyntheticLogin("phne"));
        UNIT_ASSERT(!TUtils::IsSyntheticLogin("yambot"));
        UNIT_ASSERT(!TUtils::IsSyntheticLogin("kolonkish"));
        UNIT_ASSERT(!TUtils::IsSyntheticLogin("login"));
    }

    Y_UNIT_TEST(hasLetter) {
        // ascii search
        UNIT_ASSERT(!TUtils::HasLetter(""));
        UNIT_ASSERT(!TUtils::HasLetter(" 00"));
        UNIT_ASSERT(!TUtils::HasLetter("8-(916)-123-45-67 (добавочный 100)"));
        UNIT_ASSERT(!TUtils::HasLetter("абвгд κλμνξοπρ"));
        UNIT_ASSERT(!TUtils::HasLetter("!@#$%^&*(){}[]<>,."));
        UNIT_ASSERT(!TUtils::HasLetter("11\xa0+22\xd0"));
        UNIT_ASSERT(TUtils::HasLetter("mob79261112233"));
        UNIT_ASSERT(TUtils::HasLetter("11\xa0-22\xd0"
                                      "x"));
        UNIT_ASSERT(TUtils::HasLetter("11\xf0x22\xd0"));
        UNIT_ASSERT(TUtils::HasLetter("L"));

        // utf search
        UNIT_ASSERT(!TUtils::HasLetter("", true));
        UNIT_ASSERT(!TUtils::HasLetter(" 00", true));
        UNIT_ASSERT(TUtils::HasLetter("8-(916)-123-45-67 (добавочный 100)", true));
        UNIT_ASSERT(!TUtils::HasLetter("κλμνξοπρ", true));
        UNIT_ASSERT(!TUtils::HasLetter("!@#$%^&*(){}[]<>,.", true));
        UNIT_ASSERT(!TUtils::HasLetter("11\xa0\xb0\0d1", true));
        UNIT_ASSERT(TUtils::HasLetter("тел79261112233", true));
        UNIT_ASSERT(TUtils::HasLetter("Ы", true));
        UNIT_ASSERT(TUtils::HasLetter("11\xa0\xb0ф\0d1", true));
        UNIT_ASSERT(!TUtils::HasLetter("11\xf0xxl\xb0\0f1", true));
    }

    Y_UNIT_TEST(tolowerUTF) {
        UNIT_ASSERT_VALUES_EQUAL("", TUtils::TolowerUtf(""));
        UNIT_ASSERT_VALUES_EQUAL("!@#$%^&*(){}[]<>,.", TUtils::TolowerUtf("!@#$%^&*(){}[]<>,."));
        UNIT_ASSERT_VALUES_EQUAL("abcdefghijklmnopqrstuvwxyz", TUtils::TolowerUtf("abcdefghijklmnopqrstuvwxyz"));
        UNIT_ASSERT_VALUES_EQUAL("abcdefghijklmnopqrstuvwxyz", TUtils::TolowerUtf("ABCDEFGHIJKLMNOPQRSTUVWXYZ"));
        UNIT_ASSERT_VALUES_EQUAL("абвгдеёжзийклмнопрстуфхцчшщъыьэюя", TUtils::TolowerUtf("абвгдеёжзийклмнопрстуфхцчшщъыьэюя"));
        UNIT_ASSERT_VALUES_EQUAL("абвгдеёжзийклмнопрстуфхцчшщъыьэюя", TUtils::TolowerUtf("АБВГДЕЁЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯ"));
        UNIT_ASSERT_VALUES_EQUAL("ΑΒΓΔΕΖΗΘΙΚΛΜΝΞΟΠΡΣΤΥΦΧΨΩ", TUtils::TolowerUtf("ΑΒΓΔΕΖΗΘΙΚΛΜΝΞΟΠΡΣΤΥΦΧΨΩ"));
    }
}
