#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/oauth/error.h>
#include <passport/infra/daemons/blackbox/src/oauth/token_info.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NBb;

Y_UNIT_TEST_SUITE(BbOAuthBaseFetcher) {
    class TTestFetcher: public TOAuthBaseFetcher {
    public:
        using TOAuthBaseFetcher::AddClientAttrs_;
        using TOAuthBaseFetcher::AllClientAttrs_;
        using TOAuthBaseFetcher::PostProcessClientAttributes;
        using TOAuthBaseFetcher::PostProcessTokenAttributes;
        using TOAuthBaseFetcher::ProcessClientAttribute;
        using TOAuthBaseFetcher::ProcessTokenAttribute;
        using TOAuthBaseFetcher::TOAuthBaseFetcher;

        void LogOAuth(const TOAuthTokenInfo&,
                      const TString&,
                      const TString& reason) const override {
            Reasons.push_back(reason);
        }

        static TTestFetcher Create() {
            return TTestFetcher(TTestDbHolder::GetSingleton().GetOAuthConfig());
        }

        mutable std::vector<TString> Reasons;
    };

    Y_UNIT_TEST(addClientAttr) {
        TTestFetcher fetcher = TTestFetcher::Create();

        for (TString attr : {"a", "0", "0000", "7", "15", "29", "42", "17"}) {
            fetcher.AddClientAttr(attr);
        }

        UNIT_ASSERT_VALUES_EQUAL(
            TTestFetcher::TAttrSet({"a", "42", "17"}),
            fetcher.AddClientAttrs_);
    }

    Y_UNIT_TEST(addAllClientAttrs) {
        TTestFetcher fetcher = TTestFetcher::Create();

        UNIT_ASSERT(!fetcher.AllClientAttrs_);

        fetcher.AddAllClientAttrs();
        UNIT_ASSERT(fetcher.AllClientAttrs_);
    }

    Y_UNIT_TEST(processTokenAttribute) {
        TTestFetcher fetcher = TTestFetcher::Create();
        TOAuthTokenInfo info(false);

        NDbPool::TRow row;
        row.reserve(3); // to keep refs valid
        NDbPool::TValue& id = row.emplace_back("");
        NDbPool::TValue& type = row.emplace_back("100");
        NDbPool::TValue& value = row.emplace_back("kek");

        auto test = [&](const TString& attr, const TString& val, auto func) {
            type = NDbPool::TValue(attr.c_str());
            value = NDbPool::TValue(val.c_str());
            func();
            UNIT_ASSERT_VALUES_EQUAL(val, info.TokenAttrs[attr]);
        };

        test("100", "kek", [&]() {
            UNIT_ASSERT_VALUES_EQUAL("", info.TokenId);
            fetcher.ProcessTokenAttribute(row, info);
            UNIT_ASSERT_VALUES_EQUAL("", info.TokenId);
            UNIT_ASSERT_VALUES_EQUAL("", info.TokenAttrs[TOAuthTokenAttr::TOKEN_ID]);
        });
        test("100", "kek", [&]() {
            id = NDbPool::TValue("10500");
            fetcher.ProcessTokenAttribute(row, info);
            UNIT_ASSERT_VALUES_EQUAL("10500", info.TokenId);
            UNIT_ASSERT_VALUES_EQUAL("10500", info.TokenAttrs[TOAuthTokenAttr::TOKEN_ID]);
        });
        test("100", "kek", [&]() {
            id = NDbPool::TValue("10499");
            fetcher.ProcessTokenAttribute(row, info);
            UNIT_ASSERT_VALUES_EQUAL("10500", info.TokenId);
            UNIT_ASSERT_VALUES_EQUAL("10500", info.TokenAttrs[TOAuthTokenAttr::TOKEN_ID]);
        });

        test(TOAuthTokenAttr::SCOPE_IDS, "|1|5|", [&]() {
            UNIT_ASSERT_VALUES_EQUAL("", info.GetScopesKeywordList());
            fetcher.ProcessTokenAttribute(row, info);
            UNIT_ASSERT_VALUES_EQUAL("auto:auction_api foo:bar", info.GetScopesKeywordList());
        });

        test(TOAuthTokenAttr::UID, "0", [&]() {
            UNIT_ASSERT_VALUES_EQUAL("", info.Uid);
            fetcher.ProcessTokenAttribute(row, info);
            UNIT_ASSERT_VALUES_EQUAL("", info.Uid);
        });
        test(TOAuthTokenAttr::UID, "7894561123", [&]() {
            UNIT_ASSERT_VALUES_EQUAL("", info.Uid);
            fetcher.ProcessTokenAttribute(row, info);
            UNIT_ASSERT_VALUES_EQUAL("7894561123", info.Uid);
        });

        test(TOAuthTokenAttr::DEVICE_ID, "-", [&]() {
            UNIT_ASSERT_VALUES_EQUAL("", info.DeviceId);
            fetcher.ProcessTokenAttribute(row, info);
            UNIT_ASSERT_VALUES_EQUAL("", info.DeviceId);
        });
        test(TOAuthTokenAttr::DEVICE_ID, "asdfdg", [&]() {
            UNIT_ASSERT_VALUES_EQUAL("", info.DeviceId);
            fetcher.ProcessTokenAttribute(row, info);
            UNIT_ASSERT_VALUES_EQUAL("asdfdg", info.DeviceId);
        });

        test(TOAuthTokenAttr::IS_REFRESHABLE, "false", [&]() {
            UNIT_ASSERT_VALUES_EQUAL(false, info.IsTtlRefreshable);
            fetcher.ProcessTokenAttribute(row, info);
            UNIT_ASSERT_VALUES_EQUAL(false, info.IsTtlRefreshable);
        });
        test(TOAuthTokenAttr::IS_REFRESHABLE, "true", [&]() {
            UNIT_ASSERT_VALUES_EQUAL(false, info.IsTtlRefreshable);
            fetcher.ProcessTokenAttribute(row, info);
            UNIT_ASSERT_VALUES_EQUAL(true, info.IsTtlRefreshable);
        });

        test(TOAuthTokenAttr::EXPIRES, "654123.789", [&]() {
            UNIT_ASSERT_VALUES_EQUAL(0, info.ExpireTimeTs);
            fetcher.ProcessTokenAttribute(row, info);
            UNIT_ASSERT_VALUES_EQUAL(654123, info.ExpireTimeTs);
        });

        test(TOAuthTokenAttr::CREATED, "1235435.", [&]() {
            UNIT_ASSERT_VALUES_EQUAL(0, info.CreateTimeTs);
            fetcher.ProcessTokenAttribute(row, info);
            UNIT_ASSERT_VALUES_EQUAL(1235435, info.CreateTimeTs);
        });

        test(TOAuthTokenAttr::ISSUED, "753159", [&]() {
            UNIT_ASSERT_VALUES_EQUAL(0, info.IssueTimeTs);
            fetcher.ProcessTokenAttribute(row, info);
            UNIT_ASSERT_VALUES_EQUAL(753159, info.IssueTimeTs);
        });
    }

    Y_UNIT_TEST(postProcessTokenAttributes) {
        TTestFetcher fetcher = TTestFetcher::Create();
        TOAuthTokenInfo info(false);

        UNIT_ASSERT_VALUES_EQUAL(false, info.NullExpire);

        fetcher.PostProcessTokenAttributes(info);
        UNIT_ASSERT_VALUES_EQUAL(true, info.NullExpire);

        info.ExpireTime = "kek";
        fetcher.PostProcessTokenAttributes(info);
        UNIT_ASSERT_VALUES_EQUAL(false, info.NullExpire);

        UNIT_ASSERT_VALUES_EQUAL(false, info.IsRefreshRequired);

        info.IsTtlRefreshable = true;
        fetcher.PostProcessTokenAttributes(info);
        UNIT_ASSERT_VALUES_EQUAL(true, info.IsRefreshRequired);
    }

    Y_UNIT_TEST(processClientAttribute) {
        TTestFetcher fetcher = TTestFetcher::Create();
        TOAuthTokenInfo info(false);
        TOAuthError err;

        NDbPool::TRow row;
        row.reserve(2); // to keep refs valid
        NDbPool::TValue& type = row.emplace_back("100");
        NDbPool::TValue& value = row.emplace_back("kek");

        auto test = [&](const TString& attr, const TString& val, auto func) {
            type = NDbPool::TValue(attr.c_str());
            value = NDbPool::TValue(val.c_str());
            func();
            UNIT_ASSERT_VALUES_EQUAL(val, info.ClientAttrs[attr]);
            fetcher.Reasons.clear();
        };

        test("100", "kek", [&]() {
            UNIT_ASSERT(fetcher.ProcessClientAttribute(row, info, err));
        });

        test(TOAuthClientAttr::CREATED, "789654.016", [&]() {
            UNIT_ASSERT_VALUES_EQUAL("", info.ClientCreatetime);
            UNIT_ASSERT(fetcher.ProcessClientAttribute(row, info, err));
            UNIT_ASSERT_VALUES_EQUAL("1970-01-10 06:20:54", info.ClientCreatetime);
        });

        test(TOAuthClientAttr::IS_YANDEX, "1", [&]() {
            UNIT_ASSERT_VALUES_EQUAL(false, info.ClientIsYandex);
            UNIT_ASSERT(fetcher.ProcessClientAttribute(row, info, err));
            UNIT_ASSERT_VALUES_EQUAL(true, info.ClientIsYandex);
        });

        test(TOAuthClientAttr::ICON_ID, "asdddd", [&]() {
            UNIT_ASSERT_VALUES_EQUAL("", info.ClientIcon);
            UNIT_ASSERT(fetcher.ProcessClientAttribute(row, info, err));
            UNIT_ASSERT_VALUES_EQUAL("https://avatars.mdst.yandex.net/get-oauth/asdddd/normal",
                                     info.ClientIcon);
        });

        test(TOAuthClientAttr::IS_BLOCKED, "0", [&]() {
            UNIT_ASSERT(fetcher.ProcessClientAttribute(row, info, err));
        });
        test(TOAuthClientAttr::IS_BLOCKED, "1", [&]() {
            UNIT_ASSERT(!fetcher.ProcessClientAttribute(row, info, err));
            UNIT_ASSERT_VALUES_EQUAL(TOAuthError::ClientBlocked, err.Error());
            UNIT_ASSERT_VALUES_EQUAL(1, fetcher.Reasons.size());
            UNIT_ASSERT_VALUES_EQUAL("client.blocked", fetcher.Reasons.front());
        });

        test(TOAuthClientAttr::GLOGOUTED, "100499", [&]() {
            info.IssueTimeTs = 100500;
            UNIT_ASSERT(fetcher.ProcessClientAttribute(row, info, err));
        });
        test(TOAuthClientAttr::GLOGOUTED, "100500", [&]() {
            info.IssueTimeTs = 100500;
            UNIT_ASSERT(!fetcher.ProcessClientAttribute(row, info, err));
            UNIT_ASSERT_VALUES_EQUAL(TOAuthError::TokenExpired, err.Error());
            UNIT_ASSERT_VALUES_EQUAL(1, fetcher.Reasons.size());
            UNIT_ASSERT_VALUES_EQUAL("client.glogout", fetcher.Reasons.front());
        });

        test(TOAuthClientAttr::DELETED, "", [&]() {
            UNIT_ASSERT(fetcher.ProcessClientAttribute(row, info, err));
        });
        test(TOAuthClientAttr::DELETED, "10000.789", [&]() {
            UNIT_ASSERT(fetcher.ProcessClientAttribute(row, info, err, 100));
        });
        test(TOAuthClientAttr::DELETED, "10000.789", [&]() {
            UNIT_ASSERT(!fetcher.ProcessClientAttribute(row, info, err, 1000000));
            UNIT_ASSERT_VALUES_EQUAL(TOAuthError::ClientNotFound, err.Error());
            UNIT_ASSERT_VALUES_EQUAL(1, fetcher.Reasons.size());
            UNIT_ASSERT_VALUES_EQUAL("client.not_found", fetcher.Reasons.front());
        });
    }

    Y_UNIT_TEST(postProcessClientAttributes) {
        TTestFetcher fetcher = TTestFetcher::Create();
        TOAuthTokenInfo info(false);

        UNIT_ASSERT_VALUES_EQUAL("", info.ClientAttrs[TOAuthClientAttr::CLIENT_ID]);
        TTestFetcher::PostProcessClientAttributes(info);
        UNIT_ASSERT_VALUES_EQUAL("", info.ClientAttrs[TOAuthClientAttr::CLIENT_ID]);

        info.TokenAttrs[TOAuthTokenAttr::CLIENT_ID] = "kek";
        TTestFetcher::PostProcessClientAttributes(info);
        UNIT_ASSERT_VALUES_EQUAL("kek", info.ClientAttrs[TOAuthClientAttr::CLIENT_ID]);
    }
}
