#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/oauth/multi_fetcher.h>
#include <passport/infra/daemons/blackbox/src/oauth/token_info.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NBb;

using TClientIdIndex = TOAuthMultiFetcher::TClientIdIndex;

Y_UNIT_TEST_SUITE(BbOAuthMultiFetcher) {
    Y_UNIT_TEST(buildShardQuery) {
        NDbPool::TNonBlockingHandle h(TTestDbHolder::GetSingleton().GetShards().GetPool(0));

        UNIT_ASSERT_VALUES_EQUAL(
            "SELECT a.id, a.type, a.value "
            "FROM token_attributes AS a, token_by_params AS p "
            "WHERE a.id=p.id AND p.uid=asdasd",
            TOAuthMultiFetcher::BuildShardQuery("asdasd", {}, "", h));
        UNIT_ASSERT_VALUES_EQUAL(
            "SELECT a.id, a.type, a.value "
            "FROM token_attributes AS a, token_by_params AS p "
            "WHERE a.id=p.id AND p.uid='",
            TOAuthMultiFetcher::BuildShardQuery("'", {}, "", h));

        UNIT_ASSERT_VALUES_EQUAL(
            "SELECT a.id, a.type, a.value "
            "FROM token_attributes AS a, token_by_params AS p "
            "WHERE a.id=p.id AND p.uid=123 AND p.client_id=789",
            TOAuthMultiFetcher::BuildShardQuery("123", 789, "", h));
        UNIT_ASSERT_VALUES_EQUAL(
            "SELECT a.id, a.type, a.value "
            "FROM token_attributes AS a, token_by_params AS p "
            "WHERE a.id=p.id AND p.uid=123 AND p.device_id='qweasdzxc'",
            TOAuthMultiFetcher::BuildShardQuery("123", {}, "qweasdzxc", h));
        UNIT_ASSERT_VALUES_EQUAL(
            "SELECT a.id, a.type, a.value "
            "FROM token_attributes AS a, token_by_params AS p "
            "WHERE a.id=p.id AND p.uid=123 AND p.client_id=789 AND p.device_id='qweasdzxc'",
            TOAuthMultiFetcher::BuildShardQuery("123", 789, "qweasdzxc", h));
    }

    Y_UNIT_TEST(buildIndex) {
        TOAuthMultiFetcher::TTokens tokens;

        auto add = [&tokens](ui32 tokenId, TString clientId) {
            auto tokenInfo = std::make_unique<TOAuthTokenInfo>(false);
            tokenInfo->SetTokenAttr("4", std::move(clientId));
            tokens[tokenId] = TOAuthMultiFetcher::TToken{.Info = std::move(tokenInfo)};
        };

        UNIT_ASSERT_VALUES_EQUAL(
            TClientIdIndex({}),
            TOAuthMultiFetcher::BuildIndex(tokens));

        add(142, "");
        UNIT_ASSERT_VALUES_EQUAL(
            TClientIdIndex({}),
            TOAuthMultiFetcher::BuildIndex(tokens));

        add(789, "asdqwr");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TOAuthMultiFetcher::BuildIndex(tokens),
            yexception,
            "Invalid client_id in token_id=789: asdqwr");

        add(789, "456");
        UNIT_ASSERT_VALUES_EQUAL(
            TClientIdIndex({
                {456, {.Ids = {789}}},
            }),
            TOAuthMultiFetcher::BuildIndex(tokens));

        add(785, "456");
        UNIT_ASSERT_VALUES_EQUAL(
            TClientIdIndex({
                {456, {.Ids = {785, 789}}},
            }),
            TOAuthMultiFetcher::BuildIndex(tokens));

        add(100, "159");
        UNIT_ASSERT_VALUES_EQUAL(
            TClientIdIndex({
                {456, {.Ids = {785, 789}}},
                {159, {.Ids = {100}}},
            }),
            TOAuthMultiFetcher::BuildIndex(tokens));
    }

    Y_UNIT_TEST(buildCentralQuery) {
        TOAuthMultiFetcher::TTokens tokens;

        auto add = [&tokens](ui32 tokenId, ui32 clientId) {
            auto tokenInfo = std::make_unique<TOAuthTokenInfo>(false);
            tokenInfo->SetTokenAttr("4", ToString(clientId));
            tokens[tokenId] = TOAuthMultiFetcher::TToken{.Info = std::move(tokenInfo)};
        };
        add(789, 10);
        add(795, 10);
        add(709, 16);
        const TClientIdIndex idx = TOAuthMultiFetcher::BuildIndex(tokens);

        TOAuthMultiFetcher fetcher(TTestDbHolder::GetSingleton().GetOAuthConfig());

        UNIT_ASSERT_VALUES_EQUAL(
            "SELECT type,value,id FROM client_attributes WHERE id IN(10,16) AND type IN (12,13,15,20,26,27,29,7,8)",
            fetcher.BuildCentralQuery(idx));

        for (const TString& s : {"12", "5", "89", "1"}) {
            fetcher.AddClientAttr(s);
        }
        UNIT_ASSERT_VALUES_EQUAL(
            "SELECT type,value,id FROM client_attributes WHERE id IN(10,16) AND type IN (12,13,15,20,26,27,29,7,8,1,5,89)",
            fetcher.BuildCentralQuery(idx));

        fetcher.AddAllClientAttrs();
        UNIT_ASSERT_VALUES_EQUAL(
            "SELECT type,value,id FROM client_attributes WHERE id IN(10,16)",
            fetcher.BuildCentralQuery(idx));
    }

    Y_UNIT_TEST(readTokenAttrRow) {
        TOAuthMultiFetcher fetcher(TTestDbHolder::GetSingleton().GetOAuthConfig());

        NDbPool::TRow row;
        row.reserve(3); // to keep refs valid
        NDbPool::TValue& id = row.emplace_back("142");
        NDbPool::TValue& type = row.emplace_back("100");
        row.emplace_back("kek");

        TOAuthMultiFetcher::TTokens tokens;
        UNIT_ASSERT_VALUES_EQUAL(0, tokens.size());

        fetcher.ReadTokenAttrRow(row, tokens);
        UNIT_ASSERT_VALUES_EQUAL(1, tokens.size());

        type = NDbPool::TValue("101");
        fetcher.ReadTokenAttrRow(row, tokens);
        UNIT_ASSERT_VALUES_EQUAL(1, tokens.size());

        id = NDbPool::TValue("100500");
        fetcher.ReadTokenAttrRow(row, tokens);
        UNIT_ASSERT_VALUES_EQUAL(2, tokens.size());

        for (const auto& [id, token] : tokens) {
            UNIT_ASSERT(token.Info);
        }
        UNIT_ASSERT(tokens.contains(142));
        UNIT_ASSERT(tokens.contains(100500));

        UNIT_ASSERT_VALUES_EQUAL(3, tokens[142].Info->TokenAttrs.size());
        UNIT_ASSERT_VALUES_EQUAL("142", tokens[142].Info->GetTokenAttr("0"));
        UNIT_ASSERT_VALUES_EQUAL("kek", tokens[142].Info->GetTokenAttr("100"));
        UNIT_ASSERT_VALUES_EQUAL("kek", tokens[142].Info->GetTokenAttr("101"));
        UNIT_ASSERT_VALUES_EQUAL(2, tokens[100500].Info->TokenAttrs.size());
        UNIT_ASSERT_VALUES_EQUAL("100500", tokens[100500].Info->GetTokenAttr("0"));
        UNIT_ASSERT_VALUES_EQUAL("kek", tokens[100500].Info->GetTokenAttr("101"));
    }

    Y_UNIT_TEST(finishProcessingForTokenAttributes) {
        TOAuthMultiFetcher fetcher(TTestDbHolder::GetSingleton().GetOAuthConfig());

        TOAuthMultiFetcher::TTokens tokens;

        auto add = [&tokens](ui32 tokenId, const TString& type, TString value) {
            auto tokenInfo = std::make_unique<TOAuthTokenInfo>(false);
            tokenInfo->SetTokenAttr(type, std::move(value));
            tokens[tokenId] = TOAuthMultiFetcher::TToken{.Info = std::move(tokenInfo)};
        };
        add(142, "9999999999999", "789456");

        UNIT_ASSERT(!tokens[142].Info->NullExpire);
        UNIT_ASSERT_VALUES_EQUAL(TOAuthError::Ok, tokens[142].Error.Error());

        tokens[142].Info->IssueTimeTs = 789654123;

        tokens = fetcher.FinishProcessingForTokenAttributes(std::move(tokens));
        UNIT_ASSERT(tokens[142].Info->NullExpire);
        UNIT_ASSERT_VALUES_EQUAL(TOAuthError::TokenExpired, // no scopes
                                 tokens[142].Error.Error());
    }

    Y_UNIT_TEST(readClientAttrRow) {
        TOAuthMultiFetcher fetcher(TTestDbHolder::GetSingleton().GetOAuthConfig());
        TOAuthMultiFetcher::TTokens tokens;

        auto add = [&tokens](ui32 tokenId, TString clientId) {
            auto tokenInfo = std::make_unique<TOAuthTokenInfo>(false);
            tokenInfo->SetTokenAttr(TOAuthTokenAttr::CLIENT_ID, std::move(clientId));
            tokens[tokenId] = TOAuthMultiFetcher::TToken{.Info = std::move(tokenInfo)};
        };
        add(753, "123");
        add(199, "123");

        NDbPool::TRow row;
        row.emplace_back("100");        // type
        row.emplace_back("some_value"); // value
        row.emplace_back("123");        // id

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            fetcher.ReadClientAttrRow(row, {}, tokens),
            yexception,
            "got client_id=123 in db response, which was not in query");

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            fetcher.ReadClientAttrRow(row, TClientIdIndex{{123, {.Ids = {876461232}}}}, tokens),
            yexception,
            "idx is broken: token info was not found for tokid=876461232");

        UNIT_ASSERT_NO_EXCEPTION(
            fetcher.ReadClientAttrRow(row, TClientIdIndex{{123, {.Ids = {753, 199}}}}, tokens));
        UNIT_ASSERT_VALUES_EQUAL(2, tokens.size());
        UNIT_ASSERT_VALUES_EQUAL("some_value", tokens[753].Info->GetClientAttr("100"));
        UNIT_ASSERT_VALUES_EQUAL("some_value", tokens[199].Info->GetClientAttr("100"));
    }

    Y_UNIT_TEST(finishProcessingForClientAttributes) {
        TOAuthMultiFetcher fetcher(TTestDbHolder::GetSingleton().GetOAuthConfig());
        TOAuthMultiFetcher::TTokens tokens;

        auto add = [&tokens](ui32 tokenId, TString clientId) {
            auto tokenInfo = std::make_unique<TOAuthTokenInfo>(false);
            tokenInfo->SetTokenAttr(TOAuthTokenAttr::CLIENT_ID, std::move(clientId));
            tokens[tokenId] = TOAuthMultiFetcher::TToken{.Info = std::move(tokenInfo)};
        };
        add(753, "123");
        add(199, "123");
        add(555555, "66666");
        UNIT_ASSERT(tokens.contains(555555));

        UNIT_ASSERT_VALUES_EQUAL("", tokens[753].Info->GetClientAttr(TOAuthClientAttr::CLIENT_ID));
        UNIT_ASSERT_VALUES_EQUAL("", tokens[199].Info->GetClientAttr(TOAuthClientAttr::CLIENT_ID));

        tokens = TOAuthMultiFetcher::FinishProcessingForClientAttributes(
            std::move(tokens),
            TClientIdIndex{{123, {.Ids = {9999999, 753}, .WasFound = true}}});
        UNIT_ASSERT_VALUES_EQUAL("123", tokens[753].Info->GetClientAttr(TOAuthClientAttr::CLIENT_ID));
        UNIT_ASSERT_VALUES_EQUAL("123", tokens[199].Info->GetClientAttr(TOAuthClientAttr::CLIENT_ID));
        UNIT_ASSERT_VALUES_EQUAL(TOAuthError::Ok, tokens[753].Error.Error());
        UNIT_ASSERT_VALUES_EQUAL(TOAuthError::Ok, tokens[199].Error.Error());
        UNIT_ASSERT_VALUES_EQUAL(TOAuthError::Ok, tokens[555555].Error.Error());
        tokens.clear();

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            fetcher.FinishProcessingForClientAttributes(
                std::move(tokens),
                TClientIdIndex{{123, {.Ids = {9999999, 753}, .WasFound = false}}}),
            yexception,
            "idx is broken: token info was not found for tokid=9999999");
        tokens.clear();

        add(753, "123");
        add(199, "123");
        add(555555, "66666");

        tokens = TOAuthMultiFetcher::FinishProcessingForClientAttributes(
            std::move(tokens),
            TClientIdIndex{{66666, {.Ids = {555555}, .WasFound = false}}});
        UNIT_ASSERT_VALUES_EQUAL(TOAuthError::Ok, tokens[753].Error.Error());
        UNIT_ASSERT_VALUES_EQUAL(TOAuthError::Ok, tokens[199].Error.Error());
        UNIT_ASSERT_VALUES_EQUAL(TOAuthError::ClientNotFound, tokens[555555].Error.Error());
    }
}

template <>
void Out<TClientIdIndex>(IOutputStream& out, const TClientIdIndex& value) {
    for (auto& [clid, tokids] : value) {
        out << "clid=" << clid << " (wars_found=" << tokids.WasFound << ") : {" << Endl;
        for (ui32 id : tokids.Ids) {
            out << id << "," << Endl;
        }
        out << "}" << Endl;
    }
}
