#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/oauth/scopes.h>
#include <passport/infra/daemons/blackbox/src/oauth/token_embedded_info.h>
#include <passport/infra/daemons/blackbox/src/oauth/token_info.h>

#include <passport/infra/libs/cpp/auth_core/keyring.h>
#include <passport/infra/libs/cpp/auth_core/oauth_token.h>
#include <passport/infra/libs/cpp/auth_core/oauth_token_parser.h>
#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/destination.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NBb;
using namespace NPassport::NAuth;

Y_UNIT_TEST_SUITE(OAuthUtils) {
    Y_UNIT_TEST(OAuthScopesTest) {
        TOAuthScopesConfig scopesConfig(TestsDir() + "data/scopes.local.json", 60);

        TOAuthScopes s;
        std::unordered_set<TString> ss;

        // test getScopesFromNums
        s = scopesConfig.GetScopesFromNums("");
        UNIT_ASSERT_VALUES_EQUAL(0, s.Ttl());
        UNIT_ASSERT_VALUES_EQUAL("", s.KeywordList());
        ss = s.ScopeCollection();
        UNIT_ASSERT_VALUES_EQUAL(0, ss.size());

        s = scopesConfig.GetScopesFromNums("1|2");
        UNIT_ASSERT_VALUES_EQUAL(31536000, s.Ttl());
        UNIT_ASSERT_VALUES_EQUAL("auto:auction_api direct:api", s.KeywordList());
        ss = s.ScopeCollection();
        UNIT_ASSERT_VALUES_EQUAL(2, ss.size());
        UNIT_ASSERT_UNEQUAL(ss.end(), ss.find("auto:auction_api"));
        UNIT_ASSERT_UNEQUAL(ss.end(), ss.find("direct:api"));

        s = scopesConfig.GetScopesFromNums("|1|||||2|2|8|5|");
        UNIT_ASSERT_VALUES_EQUAL(7776000, s.Ttl());
        UNIT_ASSERT_VALUES_EQUAL("auto:auction_api direct:api foo:bar", s.KeywordList());
        ss = s.ScopeCollection();
        UNIT_ASSERT_VALUES_EQUAL(3, ss.size());
        UNIT_ASSERT_UNEQUAL(ss.end(), ss.find("auto:auction_api"));
        UNIT_ASSERT_UNEQUAL(ss.end(), ss.find("direct:api"));
        UNIT_ASSERT_UNEQUAL(ss.end(), ss.find("foo:bar"));

        s = scopesConfig.GetScopesFromNums("1a|b");
        UNIT_ASSERT_VALUES_EQUAL(0, s.Ttl());
        UNIT_ASSERT_VALUES_EQUAL("", s.KeywordList());
        ss = s.ScopeCollection();
        UNIT_ASSERT_VALUES_EQUAL(0, ss.size());

        s = scopesConfig.GetScopesFromNums("14||");
        UNIT_ASSERT_VALUES_EQUAL(0, s.Ttl());
        UNIT_ASSERT_VALUES_EQUAL("login:email", s.KeywordList());
        ss = s.ScopeCollection();
        UNIT_ASSERT_VALUES_EQUAL(1, ss.size());

        s = scopesConfig.GetScopesFromNums("|1|14|2||14");
        UNIT_ASSERT_VALUES_EQUAL(31536000, s.Ttl());
        UNIT_ASSERT_VALUES_EQUAL("auto:auction_api direct:api", s.KeywordList());
        ss = s.ScopeCollection();
        UNIT_ASSERT_VALUES_EQUAL(2, ss.size());

        // test getScopesFromIds
        s = scopesConfig.GetScopesFromIds({});
        UNIT_ASSERT_VALUES_EQUAL(0, s.Ttl());
        UNIT_ASSERT_VALUES_EQUAL("", s.KeywordList());
        ss = s.ScopeCollection();
        UNIT_ASSERT_VALUES_EQUAL(0, ss.size());

        s = scopesConfig.GetScopesFromIds({1, 2, 3});
        UNIT_ASSERT_VALUES_EQUAL(31536000, s.Ttl());
        UNIT_ASSERT_VALUES_EQUAL("direct:api auto:auction_api", s.KeywordList());
        ss = s.ScopeCollection();
        UNIT_ASSERT_VALUES_EQUAL(2, ss.size());
        UNIT_ASSERT_UNEQUAL(ss.end(), ss.find("auto:auction_api"));
        UNIT_ASSERT_UNEQUAL(ss.end(), ss.find("direct:api"));

        s = scopesConfig.GetScopesFromIds({2, 2, 2, 3, 1, 5, 1});
        UNIT_ASSERT_VALUES_EQUAL(7776000, s.Ttl());
        UNIT_ASSERT_VALUES_EQUAL("foo:bar auto:auction_api direct:api", s.KeywordList());
        ss = s.ScopeCollection();
        UNIT_ASSERT_VALUES_EQUAL(3, ss.size());
        UNIT_ASSERT_UNEQUAL(ss.end(), ss.find("auto:auction_api"));
        UNIT_ASSERT_UNEQUAL(ss.end(), ss.find("direct:api"));
        UNIT_ASSERT_UNEQUAL(ss.end(), ss.find("foo:bar"));

        s = scopesConfig.GetScopesFromIds({14});
        UNIT_ASSERT_VALUES_EQUAL(0, s.Ttl());
        UNIT_ASSERT_VALUES_EQUAL("login:email", s.KeywordList());
        ss = s.ScopeCollection();
        UNIT_ASSERT_VALUES_EQUAL(1, ss.size());

        s = scopesConfig.GetScopesFromIds({1, 14, 2});
        UNIT_ASSERT_VALUES_EQUAL(31536000, s.Ttl());
        UNIT_ASSERT_VALUES_EQUAL("direct:api auto:auction_api", s.KeywordList());
        ss = s.ScopeCollection();
        UNIT_ASSERT_VALUES_EQUAL(2, ss.size());
    }

    template <class T>
    TString Tos(T num) {
        ui64 n = num;
        TString s;
        for (size_t idx = 0; idx < sizeof(T); ++idx) {
            s.push_back(n & 0xFF);
            n >>= 8;
        }
        std::reverse(s.begin(), s.vend());
        return s;
    }

    Y_UNIT_TEST(OAuthEmbeddedInfo_Random) {
        for (const auto& token : {"", "123", "sdfgsasdfsafsd", "AQAAAADufFocAAAI25K3L0ZoVEFqha",
                                  "AQAAAADufFocAAAI25K3L0ZoVEFqhayQfKOn4iwasdsad"}) {
            TOAuthTokenEmbeddedInfo t = TOAuthTokenEmbeddedInfo::Parse(token);
            UNIT_ASSERT_VALUES_EQUAL_C(true, t.IsOk(), t.ErrMsg());
            UNIT_ASSERT_VALUES_EQUAL(false, t.HasInfo());
        }
    }

    Y_UNIT_TEST(OAuthEmbeddedInfo_size39) {
        TOAuthTokenEmbeddedInfo invalid = TOAuthTokenEmbeddedInfo::Parse("AQAAAADufFocAAAI25K3L0ZoVEFqhayQfKOn4i?");
        UNIT_ASSERT_VALUES_EQUAL(false, invalid.IsOk());
        UNIT_ASSERT_VALUES_EQUAL("broken base64url", invalid.ErrMsg());

        TOAuthTokenEmbeddedInfo t = TOAuthTokenEmbeddedInfo::Parse("AQAAAADufFocAAAI25K3L0ZoVEFqhayQfKOn4iw");
        UNIT_ASSERT_VALUES_EQUAL_C(true, t.IsOk(), t.ErrMsg());
        UNIT_ASSERT_VALUES_EQUAL(true, t.HasInfo());
        UNIT_ASSERT_VALUES_EQUAL(2267U, t.ClientId());
        UNIT_ASSERT_VALUES_EQUAL(ui8(0), t.Shard());
        UNIT_ASSERT_VALUES_EQUAL(4001126940UL, t.Uid());
        UNIT_ASSERT_VALUES_EQUAL(false, t.TokenId().has_value());
        UNIT_ASSERT_VALUES_EQUAL(false, t.Environment().has_value());
    }

    Y_UNIT_TEST(OAuthEmbeddedInfo_size58) {
        struct TTestCase {
            TString Token;
            TString Err;
        };
        std::vector<TTestCase> cases({
            TTestCase{
                .Token = "y0_AQAAAAAAAYiUAAAAgQAAAAA63mixeHh4eHh4eHh4eHh4eHh4eMmOnS?",
                .Err = "broken base64url",
            },
            TTestCase{
                .Token = "t0_AQAAAAAAAYiUAAAAgQAAAAA63mixeHh4eHh4eHh4eHh4eHh4eMmOnSA",
                .Err = "malformed prefix",
            },
            TTestCase{
                .Token = "ye_AQAAAAAAAYiUAAAAgQAAAAA63mixeHh4eHh4eHh4eHh4eHh4eMmOnSA",
                .Err = "malformed environment type",
            },
            TTestCase{
                .Token = "y0_AQAAAAAAAYiUAAAAgQAAAAA63mixeHh4eHh4eHh4eHh4eHh4eMmOnSE",
                .Err = "broken crc",
            },
        });
        for (const auto& testcase : cases) {
            TOAuthTokenEmbeddedInfo token = TOAuthTokenEmbeddedInfo::Parse(testcase.Token);
            UNIT_ASSERT_VALUES_EQUAL(false, token.IsOk());
            UNIT_ASSERT_VALUES_EQUAL(testcase.Err, token.ErrMsg());
        }

        TOAuthTokenEmbeddedInfo t = TOAuthTokenEmbeddedInfo::Parse("y0_AQAAAAAAAYiUAAAAgQAAAAA63mixeHh4eHh4eHh4eHh4eHh4eMmOnSA");
        UNIT_ASSERT_VALUES_EQUAL_C(true, t.IsOk(), t.ErrMsg());
        UNIT_ASSERT_VALUES_EQUAL(true, t.HasInfo());
        UNIT_ASSERT_VALUES_EQUAL(true, t.Environment().has_value());
        UNIT_ASSERT_VALUES_EQUAL(NPassport::NAuth::EEnvironmentType::Production, *t.Environment());
        UNIT_ASSERT_VALUES_EQUAL(129U, t.ClientId());
        UNIT_ASSERT_VALUES_EQUAL(ui8(0), t.Shard());
        UNIT_ASSERT_VALUES_EQUAL(100500UL, t.Uid());
        UNIT_ASSERT_VALUES_EQUAL(true, t.TokenId().has_value());
        UNIT_ASSERT_VALUES_EQUAL(987654321UL, *t.TokenId());
    }

    Y_UNIT_TEST(OAuthTokenConversion) {
        SetLocalTimeZone();

        const NPassport::NDbPool::TDestinationPtr dsn = NPassport::NDbPool::TDestination::CreateSqlite(
            TestsDir() + "db/safeguarddb.sqlite3.sql");
        NPassport::NDbPool::TDbPool pool(NPassport::NDbPool::TDbPoolSettings{
            .Dsn = dsn,
        });
        NPassport::NAuth::TOAuthTokenParser parser(pool, {.Signkeydepth = 2});

        TOAuthTokenInfo ti(false);

        TOAuthScopesConfig scopesConfig(TestsDir() + "data/scopes.local.json", 60);

        // check token with all fields supplied
        TString token = "1.1234567890.7654321.2222222222.1122334455123.12349.CByrQgRDzzqQ2kZ1.oGRSLrjs4wgPAEAuEffyzfdayxazfFmiOirT3wCfnFwXwTHxbxN_XvPoF47GtxkeLYaTCtp4ASaUTt1iUpb1VXhmiKaJk27rs14ReTHW_zSS43TMbl7xODVkRCW1yTKzQ60AtmHdDJ-5YOr7dCtROpBEDHQi5sxJ_U5Wbr8pUf517-r3qLQxFclPPQhuFm826X05QYy7OHY1HbxuL0EH-Z8aORfz2tsehlPLoG-WXQSDZ2dEbA.PRc0ouiNaClypugiJq_hOw";
        TOAuthToken t = parser.ParseToken(token);
        UNIT_ASSERT_VALUES_EQUAL(TOAuthToken::VALID, t.Status());
        ti = TOAuthTokenInfo(t, token, scopesConfig);
        const std::unordered_set<TString>& scopedCollection = ti.GetScopeCollection();

        UNIT_ASSERT_VALUES_EQUAL("11223344551231234567890", ti.TokenId);
        UNIT_ASSERT_VALUES_EQUAL("11223344551231234567890", ti.GetTokenAttr(TOAuthTokenAttr::TOKEN_ID));
        UNIT_ASSERT_VALUES_EQUAL("1234567890", ti.Uid);
        UNIT_ASSERT_VALUES_EQUAL("1234567890", ti.GetTokenAttr(TOAuthTokenAttr::UID));
        UNIT_ASSERT_VALUES_EQUAL("666-555-444-333-222", ti.DeviceId);
        UNIT_ASSERT_VALUES_EQUAL("666-555-444-333-222", ti.GetTokenAttr(TOAuthTokenAttr::DEVICE_ID));

        UNIT_ASSERT_VALUES_EQUAL("foo:bar auto:auction_api", ti.GetScopesKeywordList());
        UNIT_ASSERT_VALUES_EQUAL("foo:bar,auto:auction_api", ti.ScopesComma);
        UNIT_ASSERT_VALUES_EQUAL("|99|7|5|1|", ti.GetTokenAttr(TOAuthTokenAttr::SCOPE_IDS));
        UNIT_ASSERT_VALUES_EQUAL(2, scopedCollection.size());
        UNIT_ASSERT_UNEQUAL(scopedCollection.end(), scopedCollection.find("auto:auction_api"));
        UNIT_ASSERT_UNEQUAL(scopedCollection.end(), scopedCollection.find("foo:bar"));

        UNIT_ASSERT_VALUES_EQUAL("2005-07-26 03:34:15", ti.CreateTime);
        UNIT_ASSERT_VALUES_EQUAL(1122334455l, ti.CreateTimeTs);
        UNIT_ASSERT_VALUES_EQUAL("1122334455", ti.GetTokenAttr(TOAuthTokenAttr::CREATED));
        UNIT_ASSERT_VALUES_EQUAL("2018-01-05 14:25:15", ti.IssueTime);
        UNIT_ASSERT_VALUES_EQUAL(1515151515l, ti.IssueTimeTs);
        UNIT_ASSERT_VALUES_EQUAL("1515151515", ti.GetTokenAttr(TOAuthTokenAttr::ISSUED));
        UNIT_ASSERT_VALUES_EQUAL("2040-06-02 06:57:02", ti.ExpireTime);
        UNIT_ASSERT_VALUES_EQUAL(2222222222l, ti.ExpireTimeTs);
        UNIT_ASSERT_VALUES_EQUAL("2222222222", ti.GetTokenAttr(TOAuthTokenAttr::EXPIRES));

        UNIT_ASSERT_VALUES_EQUAL(false, ti.IsRefreshRequired);
        UNIT_ASSERT_VALUES_EQUAL(false, ti.IsTtlRefreshable);
        UNIT_ASSERT_VALUES_EQUAL(false, ti.NullExpire);
        UNIT_ASSERT_VALUES_EQUAL("7654321", ti.GetTokenAttr(TOAuthTokenAttr::CLIENT_ID));
        UNIT_ASSERT_VALUES_EQUAL("7654321", ti.GetClientAttr(TOAuthClientAttr::CLIENT_ID));
        UNIT_ASSERT_VALUES_EQUAL("12345", ti.GetTokenAttr(TOAuthTokenAttr::X_TOKEN_ID));
        UNIT_ASSERT_VALUES_EQUAL("meta строка", ti.GetTokenAttr(TOAuthTokenAttr::META));
        UNIT_ASSERT_VALUES_EQUAL("t:some:custom.login_id string", ti.GetTokenAttr(TOAuthTokenAttr::LOGIN_ID));
        UNIT_ASSERT_VALUES_EQUAL("{e28d2bc0-77ad-43da-96a0-580ac6333f7d}", ti.GetTokenAttr(TOAuthTokenAttr::PAYMENT_AUTH_CONTEXT_ID));
        UNIT_ASSERT_VALUES_EQUAL("{\"scopes\":'addendum','looks like':\"json\"}", ti.GetTokenAttr(TOAuthTokenAttr::PAYMENT_AUTH_SCOPE_ADDENDUM));

        // check token with most fields omitted
        token = "1.0.1111.2222223333.1231231230456.12349.Rln09yfJAwiVlvCP.PEo.sHm6cPpUcEgeU1xcAQP4uQ";
        t = parser.ParseToken(token);
        UNIT_ASSERT_VALUES_EQUAL(TOAuthToken::VALID, t.Status());
        ti = TOAuthTokenInfo(t, token, scopesConfig);

        UNIT_ASSERT_VALUES_EQUAL("12312312304560", ti.TokenId);
        UNIT_ASSERT_VALUES_EQUAL("12312312304560", ti.GetTokenAttr(TOAuthTokenAttr::TOKEN_ID));
        UNIT_ASSERT_VALUES_EQUAL("", ti.Uid);
        UNIT_ASSERT_VALUES_EQUAL("0", ti.GetTokenAttr(TOAuthTokenAttr::UID));
        UNIT_ASSERT_VALUES_EQUAL("", ti.DeviceId);
        UNIT_ASSERT_VALUES_EQUAL("-", ti.GetTokenAttr(TOAuthTokenAttr::DEVICE_ID));

        UNIT_ASSERT_VALUES_EQUAL("", ti.GetScopesKeywordList());
        UNIT_ASSERT_VALUES_EQUAL("", ti.ScopesComma);
        UNIT_ASSERT_VALUES_EQUAL("", ti.GetTokenAttr(TOAuthTokenAttr::SCOPE_IDS));
        UNIT_ASSERT_VALUES_EQUAL(0, ti.GetScopeCollection().size());

        UNIT_ASSERT_VALUES_EQUAL("2009-01-06 11:40:30", ti.CreateTime);
        UNIT_ASSERT_VALUES_EQUAL(1231231230, ti.CreateTimeTs);
        UNIT_ASSERT_VALUES_EQUAL("1231231230", ti.GetTokenAttr(TOAuthTokenAttr::CREATED));
        UNIT_ASSERT_VALUES_EQUAL("1970-01-01 03:00:00", ti.IssueTime);
        UNIT_ASSERT_VALUES_EQUAL(0, ti.IssueTimeTs);
        UNIT_ASSERT_VALUES_EQUAL("0", ti.GetTokenAttr(TOAuthTokenAttr::ISSUED));
        UNIT_ASSERT_VALUES_EQUAL("2040-06-02 07:15:33", ti.ExpireTime);
        UNIT_ASSERT_VALUES_EQUAL(2222223333l, ti.ExpireTimeTs);
        UNIT_ASSERT_VALUES_EQUAL("2222223333", ti.GetTokenAttr(TOAuthTokenAttr::EXPIRES));

        UNIT_ASSERT_VALUES_EQUAL(false, ti.IsRefreshRequired);
        UNIT_ASSERT_VALUES_EQUAL(false, ti.IsTtlRefreshable);
        UNIT_ASSERT_VALUES_EQUAL(false, ti.NullExpire);

        UNIT_ASSERT(ti.HasTokenAttr(TOAuthTokenAttr::CLIENT_ID));
        UNIT_ASSERT_VALUES_EQUAL("1111", ti.GetTokenAttr(TOAuthTokenAttr::CLIENT_ID));

        UNIT_ASSERT(ti.HasClientAttr(TOAuthClientAttr::CLIENT_ID));
        UNIT_ASSERT_VALUES_EQUAL("1111", ti.GetClientAttr(TOAuthClientAttr::CLIENT_ID));

        UNIT_ASSERT(!ti.HasTokenAttr(TOAuthTokenAttr::X_TOKEN_ID));
        UNIT_ASSERT_VALUES_EQUAL("", ti.GetTokenAttr(TOAuthTokenAttr::X_TOKEN_ID));

        UNIT_ASSERT(!ti.HasTokenAttr(TOAuthTokenAttr::META));
        UNIT_ASSERT_VALUES_EQUAL("", ti.GetTokenAttr(TOAuthTokenAttr::META));

        UNIT_ASSERT(!ti.HasTokenAttr(TOAuthTokenAttr::LOGIN_ID));
        UNIT_ASSERT_VALUES_EQUAL("", ti.GetTokenAttr(TOAuthTokenAttr::LOGIN_ID));

        UNIT_ASSERT(!ti.HasTokenAttr(TOAuthTokenAttr::PAYMENT_AUTH_CONTEXT_ID));
        UNIT_ASSERT_VALUES_EQUAL("", ti.GetTokenAttr(TOAuthTokenAttr::PAYMENT_AUTH_CONTEXT_ID));

        UNIT_ASSERT(!ti.HasTokenAttr(TOAuthTokenAttr::PAYMENT_AUTH_SCOPE_ADDENDUM));
        UNIT_ASSERT_VALUES_EQUAL("", ti.GetTokenAttr(TOAuthTokenAttr::PAYMENT_AUTH_SCOPE_ADDENDUM));
    }

    Y_UNIT_TEST(OAuthTokenInfo_isXtokenTrusted) {
        TOAuthTokenInfo info;

        UNIT_ASSERT(!info.IsXTokenTrusted());

        info.SetTokenAttr("19", "1");
        UNIT_ASSERT(!info.IsXTokenTrusted());

        info.SetScopes(TOAuthScopes("", true));
        UNIT_ASSERT(info.IsXTokenTrusted());

        info.SetTokenAttr("19", "0");
        UNIT_ASSERT(!info.IsXTokenTrusted());

        info.SetTokenAttr("19", "1");
        info.ShowIsXTokenTrusted = false;
        UNIT_ASSERT(!info.IsXTokenTrusted());
    }
}
