#include "generators.h"

#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/output/all_tracks_result.h>
#include <passport/infra/daemons/blackbox/src/output/emails_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/karma_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/login_result.h>
#include <passport/infra/daemons/blackbox/src/output/loginoccupation_result.h>
#include <passport/infra/daemons/blackbox/src/output/new_session_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/oauth_result.h>
#include <passport/infra/daemons/blackbox/src/output/session_result.h>
#include <passport/infra/daemons/blackbox/src/output/table_result.h>
#include <passport/infra/daemons/blackbox/src/output/track_result.h>
#include <passport/infra/daemons/blackbox/src/output/user_info_result.h>

#include <passport/infra/libs/cpp/auth_core/session.h>
#include <passport/infra/libs/cpp/auth_core/sessionparser.h>
#include <passport/infra/libs/cpp/auth_core/sessionsigner.h>
#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/destination.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NBb;
using namespace NPassport::NAuth;

// NOTE: src/output mostly covered in json_serializer/xml_serializer tests
// here are some rare/strange/unusual cases

Y_UNIT_TEST_SUITE(PasspBbOutput) {
    Y_UNIT_TEST(newSessionChunk) {
        const NPassport::NDbPool::TDestinationPtr dsn = NPassport::NDbPool::TDestination::CreateSqlite(
            TestsDir() + "db/safeguarddb.sqlite3.sql");
        NPassport::NDbPool::TDbPool pool(NPassport::NDbPool::TDbPoolSettings{
            .Dsn = dsn,
        });
        TSessionSigner sessSigner(pool);
        UNIT_ASSERT_NO_EXCEPTION(sessSigner.AddKeyspace("yandex.ru"));
        TSessionParser sessParser(sessSigner, EEnvironmentType::Testing);

        TSession sess = sessParser.Create("70500", "0", "yandex.ru");
        UNIT_ASSERT_VALUES_EQUAL((int)TSession::VALID, (int)sess.IsValid());
        sess.SetVersion(3);
        UNIT_ASSERT_VALUES_EQUAL(3, sess.Version());
        UNIT_ASSERT_VALUES_EQUAL("70500", sess.Uid());
        UNIT_ASSERT_VALUES_EQUAL("0", sess.Ttl());

        std::unique_ptr<TNewSessionChunk> chunk = std::make_unique<TNewSessionChunk>(sess, "yandex.ru", true);

        TString value = chunk->Value;
        UNIT_ASSERT_VALUES_EQUAL("3:", value.substr(0, 2));
        UNIT_ASSERT_STRING_CONTAINS(value, ".0.0..1.2:2|70500.-1.0|3:33131.");
        UNIT_ASSERT_VALUES_EQUAL("yandex.ru", chunk->Domain);
        UNIT_ASSERT_VALUES_EQUAL(0, chunk->Expires);

        TString legacyValue = chunk->LegacyValue;
        UNIT_ASSERT_VALUES_EQUAL("3:", legacyValue.substr(0, 2));
        UNIT_ASSERT_STRING_CONTAINS(legacyValue, ".0.0..1.2:2.499:1|70500.-1.0|3:33131.");
    }

    Y_UNIT_TEST(miscChunks) {
        TKarmaChunk karmaChunk;

        UNIT_ASSERT_VALUES_EQUAL(0, karmaChunk.Karma);
        UNIT_ASSERT_VALUES_EQUAL(0, karmaChunk.RawValue);

        TLoginResult loginResult;
        UNIT_ASSERT_VALUES_EQUAL(-1, loginResult.LoginStatus);
        UNIT_ASSERT_VALUES_EQUAL("", loginResult.LoginStatusStr);
        UNIT_ASSERT_VALUES_EQUAL(-1, loginResult.PasswordStatus);
        UNIT_ASSERT_VALUES_EQUAL("", loginResult.PasswordStatusStr);

        UNIT_ASSERT_VALUES_EQUAL(0, loginResult.ResistDelay);
        loginResult.ResistDelay = 666;

        UNIT_ASSERT_VALUES_EQUAL("none", loginResult.ResistPolicyName());
        loginResult.ResistPolicy = TLoginResult::Captcha;
        UNIT_ASSERT_VALUES_EQUAL("captcha", loginResult.ResistPolicyName());
        loginResult.ResistPolicy = TLoginResult::Delay;
        UNIT_ASSERT_VALUES_EQUAL("delay", loginResult.ResistPolicyName());
        loginResult.ResistPolicy = TLoginResult::Expired;
        UNIT_ASSERT_VALUES_EQUAL("password_expired", loginResult.ResistPolicyName());

        UNIT_ASSERT_VALUES_EQUAL("free", TLoginOccupationStatus(TLoginOccupationStatus::Free).StatusName());
        UNIT_ASSERT_VALUES_EQUAL("stoplist", TLoginOccupationStatus(TLoginOccupationStatus::Stop).StatusName());
        UNIT_ASSERT_VALUES_EQUAL("occupied", TLoginOccupationStatus(TLoginOccupationStatus::Busy).StatusName());
        UNIT_ASSERT_VALUES_EQUAL("", TLoginOccupationStatus(TLoginOccupationStatus::Busy).Uid);
        UNIT_ASSERT_VALUES_EQUAL("malformed", TLoginOccupationStatus(TLoginOccupationStatus::Malformed).StatusName());
        UNIT_ASSERT_VALUES_EQUAL("public_id", TLoginOccupationStatus(TLoginOccupationStatus::PublicId, "100").StatusName());
        UNIT_ASSERT_VALUES_EQUAL("100", TLoginOccupationStatus(TLoginOccupationStatus::PublicId, "100").Uid);

        TOAuthResult oauthResult;
        UNIT_ASSERT_VALUES_EQUAL(TOAuthStatus::Invalid, oauthResult.OauthChunk.Status.Status());

        TSessionResult sessionResult;

        UNIT_ASSERT_VALUES_EQUAL("", sessionResult.SpecialKindName());
        sessionResult.SpecialKind = TSessionResult::Support;
        UNIT_ASSERT_VALUES_EQUAL("SUPPORT", sessionResult.SpecialKindName());
        sessionResult.SpecialKind = TSessionResult::Stress;
        UNIT_ASSERT_VALUES_EQUAL("STRESS", sessionResult.SpecialKindName());

        TTableResult::TStringVector columns = {"col1", "col2", "col3"};
        TTableResult tableResult("test_table", columns);
        UNIT_ASSERT_VALUES_EQUAL(0, tableResult.Rows.size());
        UNIT_ASSERT_VALUES_EQUAL(3, tableResult.Columns.size());

        tableResult.Rows.push_back(std::move(columns));
        UNIT_ASSERT_VALUES_EQUAL(1, tableResult.Rows.size());
    }
}
