#include <passport/infra/daemons/blackbox/src/totp/checker.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NBb;
using namespace NPassport::NBb::NTotp;
using namespace NPassport::NBb::NRfc4226;

Y_UNIT_TEST_SUITE(PasspBbTotpChecker) {
    void CheckResult(const TTotpResult& result, TTotpResult::ECode code, time_t ts) {
        UNIT_ASSERT_VALUES_EQUAL(code == TTotpResult::Valid, result.Succeeded());
        UNIT_ASSERT_VALUES_EQUAL((unsigned)code, (unsigned)result.Code());
        UNIT_ASSERT_VALUES_EQUAL(ts, result.TotpTime());
    }

    struct TCheck {
        TString Pwd;
        time_t LastTs;
        TTotpResult::ECode Code;
        time_t CheckTs;
    };

    Y_UNIT_TEST(totpInstanceDigits) {
        TTotpInstance c(12, 2, THotpFactory::Digits, THotpFactory::Sha256, 88, 10);
        TTotpResult r;

        std::vector<TCheck> results = {
            {"712065939585", 0, TTotpResult::Valid, 978}, // exact time slot
            {"712065939585", 978, TTotpResult::AlreadyUsed, 0},
            {"712065939585", 1978, TTotpResult::AlreadyUsed, 0},
            {"859494444155", 1100, TTotpResult::Valid, 1154}, // +2 slots
            {"251567764932", 800, TTotpResult::Valid, 802},   // -2 slots
            {"573477260814", 0, TTotpResult::Invalid, 0},     // +3 slots, out of window
            {"415558847931", 0, TTotpResult::Invalid, 0},     // -3 slots, out of window
            {"712056939585", 0, TTotpResult::Invalid, 0}      // just plain wrong
        };

        for (const auto& s : results) {
            r = c.Check("superKey", s.Pwd, s.LastTs, 1000);
            CheckResult(r, s.Code, s.CheckTs);
        }
    }

    Y_UNIT_TEST(totpInstanceLetters) {
        TTotpInstance c(8, 2, THotpFactory::Letters, THotpFactory::Sha256, 177, 77);
        TTotpResult r;

        std::vector<TCheck> results = {
            {"wvvupmdu", 0, TTotpResult::Valid, 962}, // exact time slot
            {"wvvupmdu", 962, TTotpResult::AlreadyUsed, 0},
            {"wvvupmdu", 2962, TTotpResult::AlreadyUsed, 0},
            {"lwxixfex", 1300, TTotpResult::Valid, 1316}, // +2 slots
            {"odufnbnu", 607, TTotpResult::Valid, 608},   // -2 slots
            {"asbdtiid", 0, TTotpResult::Invalid, 0},     // +3 slots, out of window
            {"aoubnqit", 0, TTotpResult::Invalid, 0},     // -3 slots, out of window
            {"wvwupmdu", 0, TTotpResult::Invalid, 0}      // just plain wrong
        };

        for (const auto& s : results) {
            r = c.Check("superKey", s.Pwd, s.LastTs, 1000);
            CheckResult(r, s.Code, s.CheckTs);
        }
    }

    Y_UNIT_TEST(totpService) {
        TService service(8, 8, 3);
        TTotpResult r;

        // validDigitPassword
        UNIT_ASSERT(service.ValidDigitPassword("12345678"));
        UNIT_ASSERT(!service.ValidDigitPassword(""));
        UNIT_ASSERT(!service.ValidDigitPassword("1234567890"));
        UNIT_ASSERT(!service.ValidDigitPassword("1234567"));
        UNIT_ASSERT(!service.ValidDigitPassword("O12345"));
        UNIT_ASSERT(!service.ValidDigitPassword(" 1 2 345678 "));

        // validLetterPassword
        UNIT_ASSERT(service.ValidLetterPassword("abcdefgh"));
        UNIT_ASSERT(!service.ValidLetterPassword(""));
        UNIT_ASSERT(!service.ValidLetterPassword("abcdefghj"));
        UNIT_ASSERT(!service.ValidLetterPassword("abcdef"));
        UNIT_ASSERT(!service.ValidLetterPassword("0xabcdef"));
        UNIT_ASSERT(!service.ValidLetterPassword(" a bc d efg h "));

        // check
        std::vector<TCheck> results = {
            {"qwerty123", 0, TTotpResult::Invalid, 0},

            {"87037251", 0, TTotpResult::Invalid, 0},
            {"52774986", 0, TTotpResult::Valid, 100410},
            {" 5277\t4986 ", 0, TTotpResult::Valid, 100410},
            {"5 2 7 7 4 9 8 6", 100410, TTotpResult::AlreadyUsed, 0},
            {"152774986", 0, TTotpResult::Invalid, 0},
            {"87411616", 0, TTotpResult::Valid, 100440},
            {"87411616", 100440, TTotpResult::AlreadyUsed, 0},
            {"1 5285731\t", 0, TTotpResult::Valid, 100470},
            {"015285731", 0, TTotpResult::Invalid, 0},
            {"74831289", 0, TTotpResult::Valid, 100500},
            {"04156239", 0, TTotpResult::Valid, 100530},
            {"00347544", 0, TTotpResult::Valid, 100560},
            {"11223140", 0, TTotpResult::Valid, 100590},
            {"17277013", 0, TTotpResult::Invalid, 0},

            {"ydttlfyd", 0, TTotpResult::Invalid, 0},
            {"xaiwkvfx", 0, TTotpResult::Valid, 100410},
            {"Xaiw kvfX ", 0, TTotpResult::Valid, 100410},
            {"Xaiw\tKvFx\t ", 100410, TTotpResult::AlreadyUsed, 0},
            {"xxaiwkvfx", 0, TTotpResult::Invalid, 0},
            {" kvfx", 0, TTotpResult::Invalid, 0},
            {"wrsvtgux", 0, TTotpResult::Valid, 100440},
            {"wrsvtgux", 100440, TTotpResult::AlreadyUsed, 0},
            {"rwvmq qiw\t", 0, TTotpResult::Valid, 100470},
            {"0rwvmqqiw", 0, TTotpResult::Invalid, 0},
            {"jedlwskc", 0, TTotpResult::Valid, 100500},
            {"VBRSUOQC", 0, TTotpResult::Valid, 100530},
            {"vstwhgek", 0, TTotpResult::Valid, 100560},
            {"dpfrifhq", 0, TTotpResult::Valid, 100590},
            {"twnvabbq", 0, TTotpResult::Invalid, 0},
        };

        for (const auto& s : results) {
            r = service.Check("s3cr3t!", s.Pwd, s.LastTs, 100500);
            CheckResult(r, s.Code, s.CheckTs);
        }
    }

    Y_UNIT_TEST(totpRfcService) {
        TServiceRfc service(3);
        TTotpResult r;

        std::vector<TCheck> results = {
            {"461389", 0, TTotpResult::Invalid, 0},
            {"85461389", 0, TTotpResult::Invalid, 0},
            {"025990", 0, TTotpResult::Valid, 100410},
            {"025990", 100420, TTotpResult::AlreadyUsed, 0},
            {"5990", 0, TTotpResult::Invalid, 0},
            {"25990", 0, TTotpResult::Invalid, 0},
            {"7025990", 0, TTotpResult::Invalid, 0},
            {"27025990", 0, TTotpResult::Invalid, 0},
            {"iwkvfx", 0, TTotpResult::Invalid, 0},
            {"185348", 0, TTotpResult::Valid, 100440},
            {"185348", 100440, TTotpResult::AlreadyUsed, 0},
            {"svtgux", 0, TTotpResult::Invalid, 0},
            {"9185348", 0, TTotpResult::Invalid, 0},
            {"99185348", 0, TTotpResult::Invalid, 0},
            {"995271", 0, TTotpResult::Valid, 100470},
            {"0995271", 0, TTotpResult::Invalid, 0},
            {"90995271", 0, TTotpResult::Invalid, 0},
            {"530034", 0, TTotpResult::Valid, 100500},
            {"590464", 0, TTotpResult::Valid, 100530},
            {"841833", 0, TTotpResult::Valid, 100560},
            {"329952", 0, TTotpResult::Valid, 100590},
            {"611544", 0, TTotpResult::Invalid, 0},
        };

        for (const auto& s : results) {
            r = service.Check("s3cr3t!", s.Pwd, s.LastTs, 100500);
            CheckResult(r, s.Code, s.CheckTs);
        }
    }
}
