#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/totp/totp_encryptor.h>

#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/testing/unittest/registar.h>

#include <regex>

using namespace NPassport;
using namespace NPassport::NBb;
using namespace NPassport::NBb::NTotp;
using namespace NPassport::NBb::NRfc4226;
using namespace NPassport::NBb::NRfc6238;

Y_UNIT_TEST_SUITE(PasspBbTotpEncryptor) {
    Y_UNIT_TEST(badDecryptToBlock) {
        TTotpEncryptor enc = CreateTotpEncryptor();
        UNIT_ASSERT(!enc.Decrypt(123, "").IsInited());
        UNIT_ASSERT(!enc.Decrypt(123, "aa:bb:cc:dd:ee:ff").IsInited());
        UNIT_ASSERT(!enc.Decrypt(123, "0:1:1:aaa:bbb:ccc").IsInited());
        UNIT_ASSERT(!enc.Decrypt(123, "4:1:1:aaa:bbb:ccc").IsInited());
        UNIT_ASSERT(!enc.Decrypt(123, "1:1:1:aaa:bbb:ccc").IsInited());
        UNIT_ASSERT(!enc.Decrypt(123, "2:2:1:aaa:bbb:ccc").IsInited());
        UNIT_ASSERT(!enc.Decrypt(123, "2:2:5:aaa:bbb:ccc").IsInited());
        UNIT_ASSERT(!enc.Decrypt(123, "2:2:5:eYGN0dzZcnE39kSGr176tQ==:bbb:ccc").IsInited());
        UNIT_ASSERT(!enc.Decrypt(123, "2:2:5:eYGN0dzZcnE39kSGr176tQ==:Bt5EhruBHkrDOy0we0tqTS9M+3KHh/I1JBqX09ppk/E=:ccc").IsInited());
        UNIT_ASSERT(!enc.Decrypt(123, "2:2:5:eYGN0dzZcnE39kSGr176tQ==:Bt5EhruBHkrDOy0we0tqTS9M+3KHh/I1JBqX09ppk/E=:06YEPNRsVes7JtrDqkE0yPn5n6WrP7ztFmjZi2YZxio=").IsInited());
        UNIT_ASSERT(!enc.Decrypt(123, "1:2:5:eYGN0dzZcnE39kSGr176tQ==:Bt5EhruBHkrDOy0we0tqTS9M+3KHh/I1JBqX09ppk/E=:06YEPNRsVes7JtrDqkE0yPn5n6WrP7ztFmjZi2YZxio=").IsInited());
        UNIT_ASSERT(!enc.Decrypt(123, "2:2:5:n4zibY63/KUHrUqBQ/43aA==:y6jotsoO7MJzKPrYkzB0r59Ih1CI6/4ocui9ebgLK853522=:4zcFXwTxUMVanb8QGvyeoU7CTEXBARYKTMhhucqU+Wo=").IsInited());
    }

    Y_UNIT_TEST(v12) {
        TTotpEncryptor enc = CreateTotpEncryptor();
        TService service(12, 8, 5);

        const std::regex totpV2Regex("2:2:5:[a-zA-Z0-9+/]{22}==:[a-zA-Z0-9+/]{43}=:[a-zA-Z0-9+/]{43}=");

        UNIT_ASSERT(enc.Encrypt(1231234, "tooshort").empty());
        UNIT_ASSERT(enc.Encrypt(1231234, "too long secret (unsupported)").empty());

        const TString passwd = TTotpFactory::GenTotp(8, THotpFactory::Letters, "16-bytes secret!", std::time(nullptr));

        TString val;
        for (int i = 0; i < 100; ++i) {
            val = enc.Encrypt(1231234, "16-bytes secret!");
            UNIT_ASSERT_C(std::regex_match(val.cbegin(), val.cend(), totpV2Regex), val);
            TTotpProfile p = enc.Decrypt(1231234, val);
            UNIT_ASSERT(p.IsInited());
            UNIT_ASSERT_VALUES_EQUAL(1231234, p.Uid());
            UNIT_ASSERT_VALUES_EQUAL("4945", p.Pin());
            UNIT_ASSERT_VALUES_EQUAL(1, p.SecretCount());
            UNIT_ASSERT_VALUES_EQUAL("0:0", p.GetSecretInfo());
            UNIT_ASSERT(p.Check(service, passwd, 0).Succeeded());
        }

        val = enc.Encrypt(1231234, "16-bytes secret!");
        UNIT_ASSERT_C(std::regex_match(val.cbegin(), val.cend(), totpV2Regex), val);
        UNIT_ASSERT(!enc.Decrypt(231234, val).IsInited()); // uid mismatch
    }

    Y_UNIT_TEST(v3) {
        TTotpEncryptor enc = CreateTotpEncryptor();
        TService service(8, 8, 5);

        UNIT_ASSERT(enc.Encrypt(TTotpProfile(0, "pin")).empty());
        UNIT_ASSERT(enc.Encrypt(TTotpProfile(1, "pin")).empty());

        TTotpProfile p(1231234, "test pin");
        TTotpProfile::TSecretData* d;

        const std::regex totpV3Regex("3:2:5:[a-zA-Z0-9+/]{22}==:[a-zA-Z0-9+/]{86}==:[a-zA-Z0-9+/]{43}=");

        UNIT_ASSERT(d = p.AddSecret("real secret", 5, 100));
        UNIT_ASSERT(d = p.AddSecret("another secret", 2, 500));
        const TString secret = d->TotpKey("test pin");
        const TString passwd = TTotpFactory::GenTotp(8, THotpFactory::Letters, secret, std::time(nullptr));

        TString val;
        for (int i = 0; i < 100; ++i) {
            val = enc.Encrypt(p);
            UNIT_ASSERT_C(std::regex_match(val.cbegin(), val.cend(), totpV3Regex), val);
            TTotpProfile p2 = enc.Decrypt(1231234, val);
            UNIT_ASSERT(p2.IsInited());
            UNIT_ASSERT_VALUES_EQUAL(1231234, p2.Uid());
            UNIT_ASSERT_VALUES_EQUAL("test pin", p2.Pin());
            UNIT_ASSERT_VALUES_EQUAL(2, p2.SecretCount());
            UNIT_ASSERT_VALUES_EQUAL("5:100,2:500", p2.GetSecretInfo());
            UNIT_ASSERT(p2.Check(service, passwd, 0).Succeeded());
        }

        val = enc.Encrypt(p);
        UNIT_ASSERT_C(std::regex_match(val.cbegin(), val.cend(), totpV3Regex), val);
        UNIT_ASSERT(!enc.Decrypt(231234, val).IsInited()); // uid mismatch
        UNIT_ASSERT(!enc.Decrypt(1231234, "3:2:5:fwxtSTrK+/yZ/XvS/GOL5Q==:WsDlQUaglPIZRP0HiOWpboRGI3VWYFq93u5mZj7hR/cQ+fCLSjyOw+xPi0PfECDssl0VZcURoowXTi4E/c5PmA==:5kiukIPza4ggcmE1wPI5LwOvP2fkhpxk7tUcIJ0Mnvo=").IsInited());
        UNIT_ASSERT(!enc.Decrypt(1231234, "3:2:5:bNqGQATCwcmfHAjZAjqx4Q==:uZ0K/Uy7LDSJiNWv1O9+ojFXVQxclSKDtUETaxlgAN3sJIGr610CbXMWVZuMguNPuGFUNjxr6Vp2stOufG58WA==:q+B4wUPTJjl9xlZW6y32tJOQ6ZZGV/3BE7IYKS8+vlE=").IsInited());
        UNIT_ASSERT(!enc.Decrypt(1231234, "3:2:5:mcD7NG+8tJ+38J8XBnh3Uw==:/AkRI86X0Kvltc0u6v3L7UBsTsKYoKLbPs8kUO9xvNeXYq7yoBxSjEHfVV0/SMmZVebfzp7ipF+LYJ01Q6nt1w==:62uZV8h0zwVEsfbnWax4N1PEACJDtw161QZu36kq7Sg=").IsInited());
        UNIT_ASSERT(!enc.Decrypt(1231234, "3:2:5:7/dA7COsN44jptio5fpi8g==:tljc2M14VKRfIP+o/u6ESTdDUZ7+ICiyLmAq/CHtvnw=:9OMOQoxPiO3Ne2xbyYMQJa42IjxhCnJmSDFt0oAykWo=").IsInited());
    }
}
