#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/totp/totp_profile.h>

#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NBb;
using namespace NPassport::NBb::NTotp;
using namespace NPassport::NBb::NRfc4226;
using namespace NPassport::NBb::NRfc6238;

Y_UNIT_TEST_SUITE(PasspBbTotpProfile) {
    Y_UNIT_TEST(emptyProfile) {
        TTotpProfile emptyUid(0, "pin");

        UNIT_ASSERT(!emptyUid.IsInited());
        UNIT_ASSERT_VALUES_EQUAL("pin", emptyUid.Pin());
        UNIT_ASSERT_VALUES_EQUAL(0, emptyUid.Uid());
        UNIT_ASSERT_VALUES_EQUAL(0, emptyUid.SecretCount());
        UNIT_ASSERT_VALUES_EQUAL("", emptyUid.LastError());
        UNIT_ASSERT_VALUES_EQUAL(0, emptyUid.OldestSecretId());
        UNIT_ASSERT_VALUES_EQUAL("", emptyUid.GetSecretInfo());

        UNIT_ASSERT(!emptyUid.AddSecret("test", 1, 1));
        UNIT_ASSERT_VALUES_EQUAL("TotpProfile is not inited: can't add secret", emptyUid.LastError());

        TTotpProfile emptyPin(1234, "");

        UNIT_ASSERT(!emptyPin.IsInited());
        UNIT_ASSERT_VALUES_EQUAL("", emptyPin.Pin());
        UNIT_ASSERT_VALUES_EQUAL(1234, emptyPin.Uid());
        UNIT_ASSERT_VALUES_EQUAL(0, emptyPin.SecretCount());
        UNIT_ASSERT_VALUES_EQUAL("", emptyPin.LastError());
        UNIT_ASSERT_VALUES_EQUAL(0, emptyPin.OldestSecretId());
        UNIT_ASSERT_VALUES_EQUAL("", emptyPin.GetSecretInfo());

        UNIT_ASSERT(!emptyPin.AddSecret("test", 1, 1));
        UNIT_ASSERT_VALUES_EQUAL("TotpProfile is not inited: can't add secret", emptyPin.LastError());

        TTestTotpProfile empty(0, "");
        UNIT_ASSERT(!empty.AddSecret());
        UNIT_ASSERT(!empty.AddSecret("test", 1, 1));
    }

    Y_UNIT_TEST(addDelete) {
        TTotpProfile p(1234, "pin");
        TTotpProfile::TSecretData* d;
        UNIT_ASSERT(p.IsInited());
        UNIT_ASSERT_VALUES_EQUAL("pin", p.Pin());
        UNIT_ASSERT_VALUES_EQUAL(1234, p.Uid());
        UNIT_ASSERT_VALUES_EQUAL(0, p.SecretCount());
        UNIT_ASSERT_VALUES_EQUAL("", p.LastError());
        UNIT_ASSERT_VALUES_EQUAL(0, p.OldestSecretId());
        UNIT_ASSERT_VALUES_EQUAL("", p.GetSecretInfo());

        UNIT_ASSERT(d = p.AddSecret("test", 1, 100));
        UNIT_ASSERT_VALUES_EQUAL("", p.LastError());

        UNIT_ASSERT_VALUES_EQUAL("test", d->Secret());
        UNIT_ASSERT_VALUES_EQUAL(1, d->SecretId());
        UNIT_ASSERT_VALUES_EQUAL(100, d->Created());
        UNIT_ASSERT_VALUES_EQUAL("90e2cd6afaf9cdef6eb243188c6d09247658dfc06feb2ca784c67a4220bbd4e4", NUtils::Bin2hex(d->TotpKey("1234")));

        UNIT_ASSERT(d = p.AddSecret("best", 2, 0));
        UNIT_ASSERT_VALUES_EQUAL("", p.LastError());

        UNIT_ASSERT_VALUES_EQUAL("best", d->Secret());
        UNIT_ASSERT_VALUES_EQUAL(2, d->SecretId());
        UNIT_ASSERT_VALUES_EQUAL(0, d->Created());
        UNIT_ASSERT_VALUES_EQUAL("e0f17972aa8b20d4526957b8915505328286c1a87b86f4ae2edd8b6759401a39", NUtils::Bin2hex(d->TotpKey("1234")));

        UNIT_ASSERT(!p.AddSecret("best", 2, 100));
        UNIT_ASSERT_VALUES_EQUAL("Secret id=2 is duplicated. It was first added at: 0", p.LastError());

        UNIT_ASSERT(d = p.AddSecret("", 5, 10000));
        UNIT_ASSERT_VALUES_EQUAL("", p.LastError());

        UNIT_ASSERT_VALUES_EQUAL("", d->Secret());
        UNIT_ASSERT_VALUES_EQUAL(5, d->SecretId());
        UNIT_ASSERT_VALUES_EQUAL(10000, d->Created());
        UNIT_ASSERT_VALUES_EQUAL("fe2592b42a727e977f055947385b709cc82b16b9a87f88c6abf3900d65d0cdc3", NUtils::Bin2hex(d->TotpKey("4321")));

        UNIT_ASSERT_VALUES_EQUAL(3, p.SecretCount());
        UNIT_ASSERT_VALUES_EQUAL(2, p.OldestSecretId());
        UNIT_ASSERT_VALUES_EQUAL("1:100,2:0,5:10000", p.GetSecretInfo());

        UNIT_ASSERT(!p.DropSecret(3));
        UNIT_ASSERT_VALUES_EQUAL("No such secret id", p.LastError());

        UNIT_ASSERT(p.DropSecret(2));
        UNIT_ASSERT_VALUES_EQUAL("", p.LastError());

        UNIT_ASSERT_VALUES_EQUAL(2, p.SecretCount());
        UNIT_ASSERT_VALUES_EQUAL(1, p.OldestSecretId());
        UNIT_ASSERT_VALUES_EQUAL("1:100,5:10000", p.GetSecretInfo());
    }

    void CheckResult(const TTotpResult& result, TTotpResult::ECode code, time_t ts) {
        UNIT_ASSERT_VALUES_EQUAL(code == TTotpResult::Valid, result.Succeeded());
        UNIT_ASSERT_VALUES_EQUAL((unsigned)code, (unsigned)result.Code());
        UNIT_ASSERT_VALUES_EQUAL(ts, result.TotpTime());
    }

    struct TCheckWithId {
        TString Pwd;
        time_t LastTs;
        unsigned SecretId;
        TTotpResult::ECode Code;
        time_t CheckTs;
    };

    struct TCheck {
        TString Pwd;
        time_t LastTs;
        TTotpResult::ECode Code;
        time_t CheckTs;
    };

    Y_UNIT_TEST(checks) {
        TTotpProfile p(1234, "test pin");
        TService service(12, 8, 3);
        TTotpResult r;
        TTotpProfile::TSecretData* d;

        UNIT_ASSERT(d = p.AddSecret("real secret", 5, 100));
        const TString realSecret = d->TotpKey("test pin");
        const TString badpinSecret = d->TotpKey("test_pin");

        UNIT_ASSERT(d = p.AddSecret("another secret", 2, 500));
        const TString anotherSecret = d->TotpKey("test pin");

        UNIT_ASSERT(d = p.AddSecret("somejunk", 11, 1100));

        std::time_t now = std::time(nullptr);
        std::time_t slotStart = now / 30 * 30;

        const TString realPwdLetters = TTotpFactory::GenTotp(8, THotpFactory::Letters, realSecret, now);
        const TString realPwdDigits = TTotpFactory::GenTotp(12, THotpFactory::Digits, realSecret, now);

        const TString badpinPwdLetters = TTotpFactory::GenTotp(8, THotpFactory::Letters, badpinSecret, now);
        const TString badpinPwdDigits = TTotpFactory::GenTotp(12, THotpFactory::Digits, badpinSecret, now);

        const TString anotherPwdLetters = TTotpFactory::GenTotp(8, THotpFactory::Letters, anotherSecret, now);
        const TString anotherPwdDigits = TTotpFactory::GenTotp(12, THotpFactory::Digits, anotherSecret, now);

        // check with given secretId
        std::vector<TCheckWithId> resultsId = {
            // bad key
            {realPwdLetters, 0, 105, TTotpResult::Invalid, 0},
            {realPwdLetters, slotStart, 105, TTotpResult::Invalid, 0},
            {realPwdDigits, 0, 105, TTotpResult::Invalid, 0},
            {realPwdDigits, slotStart, 105, TTotpResult::Invalid, 0},
            // real password, first time
            {realPwdLetters, 0, 2, TTotpResult::Invalid, 0},       // wrong id
            {realPwdLetters, 0, 5, TTotpResult::Valid, slotStart}, // good id
            {realPwdDigits, 0, 2, TTotpResult::Invalid, 0},        // wrong id
            {realPwdDigits, 0, 5, TTotpResult::Valid, slotStart},  // good id
            // real password, second time
            {realPwdLetters, slotStart, 2, TTotpResult::Invalid, 0},     // wrong id
            {realPwdLetters, slotStart, 5, TTotpResult::AlreadyUsed, 0}, // good id
            {realPwdDigits, slotStart, 2, TTotpResult::Invalid, 0},      // wrong id
            {realPwdDigits, slotStart, 5, TTotpResult::AlreadyUsed, 0},  // good id
            // bad pin
            {badpinPwdLetters, 0, 2, TTotpResult::Invalid, 0},
            {badpinPwdLetters, 0, 5, TTotpResult::Invalid, 0},
            {badpinPwdDigits, 0, 2, TTotpResult::Invalid, 0},
            {badpinPwdDigits, 0, 5, TTotpResult::Invalid, 0},
            // another password, first time
            {anotherPwdLetters, 0, 2, TTotpResult::Valid, slotStart}, // good id
            {anotherPwdLetters, 0, 5, TTotpResult::Invalid, 0},       // wrong id
            {anotherPwdDigits, 0, 2, TTotpResult::Valid, slotStart},  // good id
            {anotherPwdDigits, 0, 5, TTotpResult::Invalid, 0},        // wrong id
            // another password, second time
            {anotherPwdLetters, slotStart, 2, TTotpResult::AlreadyUsed, 0}, // good id
            {anotherPwdLetters, slotStart, 5, TTotpResult::Invalid, 0},     // wrong id
            {anotherPwdDigits, slotStart, 2, TTotpResult::AlreadyUsed, 0},  // good id
            {anotherPwdDigits, slotStart, 5, TTotpResult::Invalid, 0},      // wrong id
        };

        for (const auto& s : resultsId) {
            r = p.Check(service, s.Pwd, s.LastTs, s.SecretId);
            CheckResult(r, s.Code, s.CheckTs);
        }

        // check with no secretId
        std::vector<TCheck> results = {
            // letters, first time
            {realPwdLetters, 0, TTotpResult::Valid, slotStart},
            {badpinPwdLetters, 0, TTotpResult::Invalid, 0},
            {anotherPwdLetters, 0, TTotpResult::Valid, slotStart},
            // letters, second time
            {realPwdLetters, slotStart, TTotpResult::AlreadyUsed, 0},
            {badpinPwdLetters, slotStart, TTotpResult::Invalid, 0},
            {anotherPwdLetters, slotStart, TTotpResult::AlreadyUsed, 0},
            // digits, first time
            {realPwdDigits, 0, TTotpResult::Valid, slotStart},
            {badpinPwdDigits, 0, TTotpResult::Invalid, 0},
            {anotherPwdDigits, 0, TTotpResult::Valid, slotStart},
            // digits, second time
            {realPwdDigits, slotStart, TTotpResult::AlreadyUsed, 0},
            {badpinPwdDigits, slotStart, TTotpResult::Invalid, 0},
            {anotherPwdDigits, slotStart, TTotpResult::AlreadyUsed, 0},
        };

        for (const auto& s : results) {
            r = p.Check(service, s.Pwd, s.LastTs);
            CheckResult(r, s.Code, s.CheckTs);
        }
    }

    Y_UNIT_TEST(diagUtils) {
        TTotpProfile p(1234, "test pin");
        TService service(12, 8, 3);
        TTotpProfile::TSecretData* d;

        UNIT_ASSERT(d = p.AddSecret("real secret", 5, 100));
        const TString realSecret = d->TotpKey("test pin");
        const TString badpinSecret = d->TotpKey("best pin");

        UNIT_ASSERT(d = p.AddSecret("somejunk", 11, 1100));

        UNIT_ASSERT(d = p.AddSecret("another secret", 2, 500));
        const TString anotherSecret = d->TotpKey("test pin");

        const TString salt1("some random salt");
        const TString salt2("another random salt");

        using Pair = std::pair<bool, ui64>;
        using TimePair = std::pair<time_t, ui64>;

        // diagSecret
        {
            const TString realSecret_hash = NUtils::TCrypto::Sha256(salt1 + "real secret");
            const TString anotherSecretHash = NUtils::TCrypto::Sha256(salt1 + "another secret");
            const TString missingSecretHash = NUtils::TCrypto::Sha256(salt1 + "something unknown");

            // unknown secret
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagSecret(missingSecretHash, salt1));
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagSecret(missingSecretHash, salt2));
            // real secret
            UNIT_ASSERT_VALUES_EQUAL(Pair({true, 5}), p.DiagSecret(realSecret_hash, salt1));
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagSecret(realSecret_hash, salt2));
            // another secret
            UNIT_ASSERT_VALUES_EQUAL(Pair({true, 2}), p.DiagSecret(anotherSecretHash, salt1));
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagSecret(anotherSecretHash, salt2));
        }

        // diagPinSecret
        {
            const TString realPinSecretHash = NUtils::TCrypto::Sha256(salt2 + "test pin" + "real secret");
            const TString badPinSecretHash = NUtils::TCrypto::Sha256(salt2 + "best pin" + "real secret");
            const TString anotherPinSecretHash = NUtils::TCrypto::Sha256(salt2 + "test pin" + "another secret");

            // real secret with pin
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagPinSecret(realPinSecretHash, salt1));
            UNIT_ASSERT_VALUES_EQUAL(Pair({true, 5}), p.DiagPinSecret(realPinSecretHash, salt2));
            // real secret with bad pin
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagPinSecret(badPinSecretHash, salt1));
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagPinSecret(badPinSecretHash, salt2));
            // another secret with pin
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagPinSecret(anotherPinSecretHash, salt1));
            UNIT_ASSERT_VALUES_EQUAL(Pair({true, 2}), p.DiagPinSecret(anotherPinSecretHash, salt2));
        }

        // diagTotp
        {
            std::time_t time1 = 1234512345;
            std::time_t time2 = 987654321;

            const TString realTotpT1 = TTotpFactory::GenTotp(8, THotpFactory::Letters, realSecret, time1);
            const TString realTotpT2 = TTotpFactory::GenTotp(8, THotpFactory::Letters, realSecret, time2);
            const TString badpinTotp = TTotpFactory::GenTotp(8, THotpFactory::Letters, badpinSecret, time1);
            const TString anotherTotpT1 = TTotpFactory::GenTotp(8, THotpFactory::Letters, anotherSecret, time1);
            const TString anotherTotpT2 = TTotpFactory::GenTotp(8, THotpFactory::Letters, anotherSecret, time2);

            const TString realTotpT1Hash = NUtils::TCrypto::Sha256(salt1 + realTotpT1);
            const TString realTotpT2Hash = NUtils::TCrypto::Sha256(salt1 + realTotpT2);
            const TString badpinTotpHash = NUtils::TCrypto::Sha256(salt1 + badpinTotp);
            const TString anotherTotpT1Hash = NUtils::TCrypto::Sha256(salt2 + anotherTotpT1);
            const TString anotherTotpT2Hash = NUtils::TCrypto::Sha256(salt2 + anotherTotpT2);

            // badpin secret
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagTotp(service, badpinTotpHash, time1, salt1));
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagTotp(service, badpinTotpHash, time1, salt2));
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagTotp(service, badpinTotpHash, time2, salt1));
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagTotp(service, badpinTotpHash, time2, salt2));
            // real secret
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagTotp(service, realTotpT1Hash, time1, salt2));
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagTotp(service, realTotpT1Hash, time2, salt1));
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagTotp(service, realTotpT1Hash, time2, salt2));
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagTotp(service, realTotpT2Hash, time1, salt1));
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagTotp(service, realTotpT2Hash, time1, salt2));
            UNIT_ASSERT_VALUES_EQUAL(Pair({true, 5}), p.DiagTotp(service, realTotpT2Hash, time2, salt1));
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagTotp(service, realTotpT2Hash, time2, salt2));
            // another secret
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagTotp(service, anotherTotpT1Hash, time1, salt1));
            UNIT_ASSERT_VALUES_EQUAL(Pair({true, 2}), p.DiagTotp(service, anotherTotpT1Hash, time1, salt2));
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagTotp(service, anotherTotpT1Hash, time2, salt1));
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagTotp(service, anotherTotpT1Hash, time2, salt2));
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagTotp(service, anotherTotpT2Hash, time1, salt1));
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagTotp(service, anotherTotpT2Hash, time1, salt2));
            UNIT_ASSERT_VALUES_EQUAL(Pair({false, 0}), p.DiagTotp(service, anotherTotpT2Hash, time2, salt1));
            UNIT_ASSERT_VALUES_EQUAL(Pair({true, 2}), p.DiagTotp(service, anotherTotpT2Hash, time2, salt2));
        }

        // diagTime
        {
            std::time_t now = 1234512345;
            std::time_t slot = now / 30 * 30;
            const TString skew1("150");
            const TString skew2("700");

            const TString realNow = TTotpFactory::GenTotp(8, THotpFactory::Letters, realSecret, now);
            const TString real5before = TTotpFactory::GenTotp(8, THotpFactory::Letters, realSecret, now - 30 * 5);
            const TString real5after = TTotpFactory::GenTotp(8, THotpFactory::Letters, realSecret, now + 30 * 5);
            const TString badpinNow = TTotpFactory::GenTotp(8, THotpFactory::Letters, badpinSecret, now);
            const TString badpin3before = TTotpFactory::GenTotp(8, THotpFactory::Letters, badpinSecret, now - 30 * 3);
            const TString badpin3after = TTotpFactory::GenTotp(8, THotpFactory::Letters, badpinSecret, now + 30 * 3);
            const TString anotherNow = TTotpFactory::GenTotp(8, THotpFactory::Letters, anotherSecret, now);
            const TString another17before = TTotpFactory::GenTotp(8, THotpFactory::Letters, anotherSecret, now - 30 * 17);
            const TString another17after = TTotpFactory::GenTotp(8, THotpFactory::Letters, anotherSecret, now + 30 * 17);

            const TString realNowHash = NUtils::TCrypto::Sha256(salt1 + realNow);
            const TString real5beforeHash = NUtils::TCrypto::Sha256(salt1 + real5before);
            const TString real5afterHash = NUtils::TCrypto::Sha256(salt1 + real5after);
            const TString badpinNowHash = NUtils::TCrypto::Sha256(salt1 + badpinNow);
            const TString badpin3beforeHash = NUtils::TCrypto::Sha256(salt1 + badpin3before);
            const TString badpin3afterHash = NUtils::TCrypto::Sha256(salt1 + badpin3after);
            const TString anotherNowHash = NUtils::TCrypto::Sha256(salt2 + anotherNow);
            const TString another17beforeHash = NUtils::TCrypto::Sha256(salt2 + another17before);
            const TString another17afterHash = NUtils::TCrypto::Sha256(salt2 + another17after);

            // bad skew
            UNIT_ASSERT_EXCEPTION_CONTAINS(p.DiagTime(service, realNow, "", now, salt1), TBlackboxError, "invalid skew value");
            UNIT_ASSERT_EXCEPTION_CONTAINS(p.DiagTime(service, realNow, "a", now, salt1), TBlackboxError, "invalid skew value");
            UNIT_ASSERT_EXCEPTION_CONTAINS(p.DiagTime(service, realNow, "-1", now, salt1), TBlackboxError, "invalid skew value");
            // badpin secret
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, badpinNowHash, skew1, now, salt1));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, badpinNowHash, skew1, now, salt2));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, badpinNowHash, skew2, now, salt1));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, badpinNowHash, skew2, now, salt2));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, badpin3beforeHash, skew1, now, salt1));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, badpin3beforeHash, skew1, now, salt2));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, badpin3beforeHash, skew2, now, salt1));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, badpin3beforeHash, skew2, now, salt2));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, badpin3afterHash, skew1, now, salt1));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, badpin3afterHash, skew1, now, salt2));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, badpin3afterHash, skew2, now, salt1));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, badpin3afterHash, skew2, now, salt2));
            // real secret
            UNIT_ASSERT_VALUES_EQUAL(TimePair({slot, 5}), p.DiagTime(service, realNowHash, skew1, now, salt1));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, realNowHash, skew1, now, salt2));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({slot, 5}), p.DiagTime(service, realNowHash, skew2, now, salt1));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, realNowHash, skew2, now, salt2));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({slot - 30 * 5, 5}), p.DiagTime(service, real5beforeHash, skew1, now, salt1));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, real5beforeHash, skew1, now, salt2));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({slot - 30 * 5, 5}), p.DiagTime(service, real5beforeHash, skew2, now, salt1));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, real5beforeHash, skew2, now, salt2));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({slot + 30 * 5, 5}), p.DiagTime(service, real5afterHash, skew1, now, salt1));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, real5afterHash, skew1, now, salt2));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({slot + 30 * 5, 5}), p.DiagTime(service, real5afterHash, skew2, now, salt1));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, real5afterHash, skew2, now, salt2));
            // another secret
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, anotherNowHash, skew1, now, salt1));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({slot, 2}), p.DiagTime(service, anotherNowHash, skew1, now, salt2));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, anotherNowHash, skew2, now, salt1));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({slot, 2}), p.DiagTime(service, anotherNowHash, skew2, now, salt2));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, another17beforeHash, skew1, now, salt1));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, another17beforeHash, skew1, now, salt2));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, another17beforeHash, skew2, now, salt1));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({slot - 30 * 17, 2}), p.DiagTime(service, another17beforeHash, skew2, now, salt2));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, another17afterHash, skew1, now, salt1));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, another17afterHash, skew1, now, salt2));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({0, 0}), p.DiagTime(service, another17afterHash, skew2, now, salt1));
            UNIT_ASSERT_VALUES_EQUAL(TimePair({slot + 30 * 17, 2}), p.DiagTime(service, another17afterHash, skew2, now, salt2));
        }
    }
}
