#include <passport/infra/daemons/blackbox/src/totp/totp.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NBb;
using namespace NPassport::NBb::NRfc4226;
using namespace NPassport::NBb::NRfc6238;

Y_UNIT_TEST_SUITE(PasspBbTotpTotp) {
    Y_UNIT_TEST(totpProduceDigits) {
        THotpFactory hotp(THotpFactory::defaultHotpDigitsLength, THotpFactory::Digits);
        TTotpFactory totp(hotp, 30, 0);
        UNIT_ASSERT_VALUES_EQUAL("459110", totp.Produce("superKey", 9));
        UNIT_ASSERT_VALUES_EQUAL("118981", totp.Produce("superKey", 59));

        std::vector<TTotp> passwords;
        totp.Produce("superKey", 120, 2, std::back_inserter(passwords));
        UNIT_ASSERT_VALUES_EQUAL(passwords, std::vector<TTotp>({"358441", "361071", "515226", "515202", "440748"}));

        UNIT_ASSERT_EXCEPTION_CONTAINS(totp.Produce("superKey", 30, 2, std::back_inserter(passwords)),
                                       std::invalid_argument, "TotpFactory::produce() window too large - underflow!");
        UNIT_ASSERT_EXCEPTION_CONTAINS(totp.Produce("superKey", 500, std::numeric_limits<std::time_t>::max() - 1, std::back_inserter(passwords)),
                                       std::invalid_argument, "TotpFactory::produce() window too large - underflow!");

        TTotpFactory strangeTotp(hotp, 1, 0);
        std::time_t halfTime = std::numeric_limits<std::time_t>::max() / 2;
        UNIT_ASSERT_EXCEPTION_CONTAINS(strangeTotp.Produce("superKey", halfTime - 5, halfTime, std::back_inserter(passwords)),
                                       std::invalid_argument, "TotpFactory::produce() window too large - underflow!");

        UNIT_ASSERT_EXCEPTION_CONTAINS(strangeTotp.Produce("superKey", -500, halfTime, std::back_inserter(passwords)),
                                       std::invalid_argument, "TotpFactory::produce() window too large - overflow!");

        TTotpFactory totpDefault(hotp);
        UNIT_ASSERT_VALUES_EQUAL("459110", totpDefault.Produce("superKey", 9));
        UNIT_ASSERT_VALUES_EQUAL("118981", totpDefault.Produce("superKey", 59));
    }

    Y_UNIT_TEST(totpProduceLetters) {
        THotpFactory hotp(THotpFactory::defaultHotpLettersLength, THotpFactory::Letters);
        TTotpFactory totp(hotp, 30, 0);
        UNIT_ASSERT_VALUES_EQUAL("wpzotibu", totp.Produce("s3cret", 9));
        UNIT_ASSERT_VALUES_EQUAL("unshwiqy", totp.Produce("s3cret", 59));

        std::vector<TTotp> passwords;
        totp.Produce("s3cret", 120, 2, std::back_inserter(passwords));
        UNIT_ASSERT_VALUES_EQUAL(passwords, std::vector<TTotp>({"yivmhfww", "afwgjcvb", "uivbmivd", "jfgrwdsu", "ymmvuphi"}));

        UNIT_ASSERT_EXCEPTION_CONTAINS(totp.Produce("s3cret", 30, 2, std::back_inserter(passwords)),
                                       std::invalid_argument, "TotpFactory::produce() window too large - underflow!");
        UNIT_ASSERT_EXCEPTION_CONTAINS(totp.Produce("s3cret", 500, std::numeric_limits<std::time_t>::max() - 1, std::back_inserter(passwords)),
                                       std::invalid_argument, "TotpFactory::produce() window too large - underflow!");

        TTotpFactory strangeTotp(hotp, 1, 0);
        std::time_t halfTime = std::numeric_limits<std::time_t>::max() / 2;
        UNIT_ASSERT_EXCEPTION_CONTAINS(strangeTotp.Produce("s3cret", halfTime - 5, halfTime, std::back_inserter(passwords)),
                                       std::invalid_argument, "TotpFactory::produce() window too large - underflow!");

        UNIT_ASSERT_EXCEPTION_CONTAINS(strangeTotp.Produce("s3cret", -500, halfTime, std::back_inserter(passwords)),
                                       std::invalid_argument, "TotpFactory::produce() window too large - overflow!");

        TTotpFactory totpDefault(hotp);
        UNIT_ASSERT_VALUES_EQUAL("wpzotibu", totpDefault.Produce("s3cret", 9));
        UNIT_ASSERT_VALUES_EQUAL("unshwiqy", totpDefault.Produce("s3cret", 59));
    }

    Y_UNIT_TEST(totpGenTotp) {
        UNIT_ASSERT_VALUES_EQUAL("459110", TTotpFactory::GenTotp(THotpFactory::defaultHotpDigitsLength,
                                                                 THotpFactory::Digits,
                                                                 "superKey",
                                                                 9));
        UNIT_ASSERT_VALUES_EQUAL("118981", TTotpFactory::GenTotp(THotpFactory::defaultHotpDigitsLength,
                                                                 THotpFactory::Digits,
                                                                 "superKey",
                                                                 59));

        UNIT_ASSERT_VALUES_EQUAL("27459110", TTotpFactory::GenTotp(8,
                                                                   THotpFactory::Digits,
                                                                   "superKey",
                                                                   9));
        UNIT_ASSERT_VALUES_EQUAL("29118981", TTotpFactory::GenTotp(8,
                                                                   THotpFactory::Digits,
                                                                   "superKey",
                                                                   59));

        UNIT_ASSERT_VALUES_EQUAL("wpzotibu", TTotpFactory::GenTotp(8,
                                                                   THotpFactory::Letters,
                                                                   "s3cret",
                                                                   9));
        UNIT_ASSERT_VALUES_EQUAL("unshwiqy", TTotpFactory::GenTotp(8,
                                                                   THotpFactory::Letters,
                                                                   "s3cret",
                                                                   59));

        UNIT_ASSERT_VALUES_EQUAL("741964", TTotpFactory::GenRfcTotp("superKey", 9));
        UNIT_ASSERT_VALUES_EQUAL("838740", TTotpFactory::GenRfcTotp("superKey", 59));
        UNIT_ASSERT_VALUES_EQUAL("871667", TTotpFactory::GenRfcTotp("s3cret", 9));
        UNIT_ASSERT_VALUES_EQUAL("058613", TTotpFactory::GenRfcTotp("s3cret", 59));
    }

    Y_UNIT_TEST(totpCustomTime) {
        THotpFactory hotp(THotpFactory::defaultHotpDigitsLength, THotpFactory::Digits);
        TTotpFactory totp(hotp, 77, 13);
        UNIT_ASSERT_VALUES_EQUAL(77, totp.GetStepTime());

        UNIT_ASSERT_VALUES_EQUAL(0, totp.SkewToWindow(0));
        UNIT_ASSERT_VALUES_EQUAL(0, totp.SkewToWindow(76));
        UNIT_ASSERT_VALUES_EQUAL(1, totp.SkewToWindow(77));
        UNIT_ASSERT_VALUES_EQUAL(2, totp.SkewToWindow(200));
        UNIT_ASSERT_VALUES_EQUAL(0, totp.SkewToWindow(-7));
        UNIT_ASSERT_VALUES_EQUAL(-1, totp.SkewToWindow(-100));

        UNIT_ASSERT_VALUES_EQUAL("525437", totp.Produce("superKey", 1));
        UNIT_ASSERT_VALUES_EQUAL("459110", totp.Produce("superKey", 13));
        UNIT_ASSERT_VALUES_EQUAL("459110", totp.Produce("superKey", 20));
        UNIT_ASSERT_VALUES_EQUAL("459110", totp.Produce("superKey", 89));
        UNIT_ASSERT_VALUES_EQUAL("118981", totp.Produce("superKey", 90));
        UNIT_ASSERT_VALUES_EQUAL("118981", totp.Produce("superKey", 100));
        UNIT_ASSERT_VALUES_EQUAL("118981", totp.Produce("superKey", 166));
        UNIT_ASSERT_VALUES_EQUAL("358441", totp.Produce("superKey", 167));
    }

    Y_UNIT_TEST(totpIterator) {
        THotpFactory hotp(12, THotpFactory::Digits);
        TTotpFactory totp(hotp, 88, 10);
        TTotpFactory::TIterator it = totp.GetIterator(1000, 3, "superKey");

        struct TIteratorState {
            TString Val;
            std::time_t Ts;
            std::time_t Sh;
        };

        std::vector<TIteratorState> results = {
            {"712065939585", 978, 3},
            {"394098516253", 890, 2},
            {"334039319176", 1066, 4},
            {"251567764932", 802, 1},
            {"859494444155", 1154, 5},
            {"415558847931", 714, 0},
            {"573477260814", 1242, 6},
        };

        for (const auto& s : results) {
            UNIT_ASSERT_VALUES_EQUAL(s.Val, it.Value());
            UNIT_ASSERT_VALUES_EQUAL(s.Ts, it.Timestamp());
            UNIT_ASSERT_VALUES_EQUAL(s.Sh, it.Shift());
            UNIT_ASSERT(s.Sh != 6 ? it.Next() : !it.Next());
        }
    }

    // Integration test, according to http://www.rfc-editor.org/rfc/rfc6238.txt
    // Appendix B.  Test Vectors
    // For the maximum standard length = 8
    // SHA-256 items are used
    using TTotpVector = std::vector<std::pair<std::time_t, TTotp>>;

    Y_UNIT_TEST(totpIntegrationStandard) {
        THotpFactory hotp(8);
        TTotpFactory totp(hotp);
        const TString secret("12345678901234567890123456789012");

        TTotpVector results = {
            {59, "46119246"},
            {1111111109, "68084774"},
            {1111111111, "67062674"},
            {1234567890, "91819424"},
            {2000000000, "90698825"},
            {20000000000, "77737706"}};

        for (const auto& pair : results) {
            UNIT_ASSERT_VALUES_EQUAL(pair.second, totp.Produce(secret, pair.first));
        }
    }

    // Integration test, similar to http://www.rfc-editor.org/rfc/rfc6238.txt
    // but for our nonstandard maximum length = 12
    //
    Y_UNIT_TEST(totpIntegrationCustom) {
        THotpFactory hotp(12);
        TTotpFactory totp(hotp);
        const TString secret("12345678901234567890123456789012");

        TTotpVector results = {
            {59, "760870385432"},
            {1111111109, "825007741896"},
            {1111111111, "019898070882"},
            {1234567890, "425740522484"},
            {2000000000, "893279495488"},
            {20000000000, "013173827260"}};

        for (const auto& pair : results) {
            UNIT_ASSERT_VALUES_EQUAL(pair.second, totp.Produce(secret, pair.first));
        }
    }

    // Integration test, similar to http://www.rfc-editor.org/rfc/rfc6238.txt
    // but for our custom letters function
    //
    Y_UNIT_TEST(totpIntegrationLetters) {
        THotpFactory hotp(8, THotpFactory::Letters);
        TTotpFactory totp(hotp);
        const TString secret("12345678901234567890123456789012");

        TTotpVector results = {
            {59, "ueitlebg"},
            {1111111109, "sbeefdpo"},
            {1111111111, "ssovlrqw"},
            {1234567890, "jmrgwbxk"},
            {2000000000, "inwneuaa"},
            {20000000000, "vrwjvqqe"}};

        for (const auto& pair : results) {
            UNIT_ASSERT_VALUES_EQUAL(pair.second, totp.Produce(secret, pair.first));
        }
    }
}
