package api

import (
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/grants"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/hbaseapi"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/ytc"
	"a.yandex-team.ru/passport/infra/libs/go/keys"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/httpdtvm"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/middlewares"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/passport/shared/golibs/unistat"
)

type API struct {
	grants    *grants.Grants
	tvmClient *tvmauth.Client
	accessLog log.Logger
	oldAPI    *hbaseapi.Client
	yt        *ytc.Client
	pingFile  string
	stats     Stats
	keys      Keyring
	config    Config
}

type Keyring struct {
	Restore      *keys.KeyMap
	YasmsPrivate *keys.KeyMap
}

type Stats struct {
	invalidParams *unistat.SignalDiff
	accessDenied  *unistat.SignalDiff
	tmpErr        *unistat.SignalDiff
	decryptErr    *unistat.SignalDiff
	unknown       *unistat.SignalDiff
}

type KeysConfig struct {
	Restore      keys.Config `json:"restore"`
	YasmsPrivate keys.Config `json:"yasms_private"`
}

type CommonConfig struct {
	PingFile  string        `json:"ping_file"`
	AccessLog logger.Config `json:"access_log"`

	AuthsYtThreshold      uint64 `json:"auths_yt_threshold"`
	YasmsSmsHistoryFromYt bool   `json:"yasms_sms_history_from_yt"`
	PushSubscriptionWidth uint64 `json:"push_subscription_width"`
}

type Config struct {
	Common   CommonConfig       `json:"common"`
	Grants   grants.Config      `json:"grants"`
	HbaseCfg hbaseapi.Config    `json:"hbase_api"`
	Tvm      httpdtvm.TvmConfig `json:"tvm"`
	Yt       ytc.Config         `json:"yt"`
	Keys     KeysConfig         `json:"keys"`
}

const (
	xYaUserTicket = "X-Ya-User-Ticket"
)

type APIFactory struct{}

func (f *APIFactory) NewService(config httpdaemon.ServiceConfig) (httpdaemon.Service, error) {
	var cfg Config
	if err := httpdaemon.ParseServiceConfig(config, &cfg); err != nil {
		return nil, xerrors.Errorf("Failed to parse config: %w", err)
	}

	tvmClient, err := httpdtvm.InitTvm(cfg.Tvm)
	if err != nil {
		return nil, xerrors.Errorf("Failed to create client for TVM: %w", err)
	}

	api, err := hbaseapi.NewClient(cfg.HbaseCfg)
	if err != nil {
		return nil, xerrors.Errorf("Failed to create client for hbaseApi: %w", err)
	}

	yt, err := ytc.NewClient(cfg.Yt, tvmClient)
	if err != nil {
		return nil, xerrors.Errorf("Failed to create client for YT: %w", err)
	}

	gr, err := grants.NewGrants(cfg.Grants)
	if err != nil {
		return nil, xerrors.Errorf("Failed to get grants: %w", err)
	}

	accessLog, err := logger.CreateLog(cfg.Common.AccessLog)
	if err != nil {
		return nil, err
	}

	restoreKeys, err := keys.InitKeyMap(cfg.Keys.Restore)
	if err != nil {
		return nil, err
	}

	yasmsPrivateKeys, err := keys.InitKeyMap(cfg.Keys.YasmsPrivate)
	if err != nil {
		return nil, err
	}

	return &API{
		grants:    gr,
		tvmClient: tvmClient,
		accessLog: accessLog,
		oldAPI:    api,
		yt:        yt,
		pingFile:  cfg.Common.PingFile,
		stats: Stats{
			invalidParams: unistat.DefaultChunk.CreateSignalDiff("errors.requests.invalid_params"),
			accessDenied:  unistat.DefaultChunk.CreateSignalDiff("errors.requests.access_denied"),
			tmpErr:        unistat.DefaultChunk.CreateSignalDiff("errors.requests.tmp_err"),
			decryptErr:    unistat.DefaultChunk.CreateSignalDiff("errors.requests.decrypt_err"),
			unknown:       unistat.DefaultChunk.CreateSignalDiff("errors.requests.unknown"),
		},
		keys: Keyring{
			Restore:      restoreKeys,
			YasmsPrivate: yasmsPrivateKeys,
		},
		config: cfg,
	}, nil
}

func (s *API) AddHandlers(e *echo.Echo) {
	e.Pre(s.middlewareAccessLog())
	e.Use(
		s.middlewareTvmServiceTicketAuthentication(),
		s.middlewareTvmUserTicketAuthentication(),
	)

	e.GET(
		"/ping",
		s.PingHandler(),
	)

	e.GET(
		"/push/2/by_push_id/",
		s.PushByPushIDHandler(),
	)
	e.GET(
		"/push/2/by_fields/",
		s.PushByFieldsHandler(),
	)

	e.GET(
		"/push_subscription/2/actual/",
		s.PushSubscriptionHandler(),
	)

	e.GET(
		"/mail/2/user_history/",
		s.MailUserHistoryHandler(),
	)

	e.GET(
		"/sender/2/last_letter/",
		s.LastLetterHandler(),
	)

	e.GET(
		"/2/lastauth/",
		s.LastAuthHandler(),
	)
	e.GET(
		"/2/lastauth/bulk/",
		s.LastAuthBulkHandler(),
	)

	e.GET(
		"/2/events/",
		s.EventsHandler(),
	)
	e.GET(
		"/2/events/restore/",
		s.EventsRestoreHandler(),
	)
	e.GET(
		"/2/events/by_ip/registrations/",
		s.EventsByIPRegistrationsHandler(),
	)
	e.POST(
		"/2/events/passwords/",
		s.EventsPasswordsHandler(),
	)

	e.GET(
		"/yasms/2/sms/by_globalid/",
		s.SmsByGlobalIDHandler(),
	)
	e.GET(
		"/yasms/2/sms/by_phone/",
		s.SmsByPhoneHandler(),
	)
	e.GET(
		"/yasms/2/sms/laststatus/by_globalid/",
		s.SmsLastStatusByGlobalIDHandler(),
	)

	e.GET(
		"/2/auths/",
		s.AuthsHandler(false),
	)
	e.GET(
		"/2/auths/failed/",
		s.AuthsHandler(true),
	)
	e.GET(
		"/3/auths/aggregated/",
		s.AuthsAggregatedHandler(),
	)
	e.GET(
		"/2/auths/aggregated/runtime/",
		s.AuthsRuntimeAggregatedHandler(),
	)
}

func (s *API) PingHandler() echo.HandlerFunc {
	return func(c echo.Context) error {
		pingBody, err := ioutil.ReadFile(s.pingFile)
		if err != nil {
			ctxlog.Debugf(c.Request().Context(), logger.Log(),
				"Ping: Service is forced down: %s", err)
			return c.String(http.StatusForbidden, "Service is forced down")
		}

		if err := s.oldAPI.Ping(c.Request().Context()); err != nil {
			msg := fmt.Sprintf("Old API is down: %s", err)
			ctxlog.Debugf(c.Request().Context(), logger.Log(),
				"Ping: %s", msg)
			return c.String(http.StatusForbidden, msg)
		}

		if err := s.yt.Ping(c.Request().Context()); err != nil {
			msg := fmt.Sprintf("YT is down: %s", err)
			ctxlog.Debugf(c.Request().Context(), logger.Log(),
				"Ping: %s", msg)
			return c.String(http.StatusInternalServerError, msg)
		}

		ctxlog.Debugf(c.Request().Context(), logger.Log(), "Ping: service is up")
		return c.String(http.StatusOK, string(pingBody))
	}
}

func (s *API) middlewareAccessLog() echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			c.Response().After(func() {
				startTime := middlewares.ContextStartInstant(c.Request().Context())

				st := tvm.ContextServiceTicket(c.Request().Context())
				ut := tvm.ContextUserTicket(c.Request().Context())

				srcID := "-"
				if st != nil {
					srcID = fmt.Sprintf("%d", st.SrcID)
				}

				defaultUID := "-"
				if ut != nil {
					defaultUID = fmt.Sprintf("%d", ut.DefaultUID)
				}

				query := c.Request().URL.RawQuery
				if query == "" {
					query = "-"
				}

				s.accessLog.Debugf(
					"%s\t%s\t%d\t%.2fms\ttvmid=%s\tuser=%s\t%s\t%s\t%s",
					middlewares.ContextReqID(c.Request().Context()),
					c.RealIP(),
					c.Response().Status,
					float64(time.Since(startTime).Microseconds())/1000,
					srcID,
					defaultUID,
					c.Request().Method,
					c.Request().URL.Path,
					query,
				)
			})

			return next(c)
		}
	}
}

func (s *API) middlewareTvmServiceTicketAuthentication() echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			st := c.Request().Header.Get("X-Ya-Service-Ticket")
			if st == "" {
				return next(c)
			}

			ticket, err := s.tvmClient.CheckServiceTicket(c.Request().Context(), st)
			if err != nil {
				ctxlog.Debugf(c.Request().Context(), logger.Log(), "TVM error: %+v", err)
				e := &errs.AccessDeniedError{
					Message:      "service ticket is invalid",
					Description:  err.Error(),
					TicketStatus: err.(*tvm.TicketError).Status.String(),
				}
				if ticket != nil {
					e.LoggablePart = ticket.LogInfo
				}
				if err.(*tvm.TicketError).Status == tvm.TicketInvalidDst {
					e.Description = fmt.Sprintf(
						"%s; expected dst is %d",
						e.Description, s.config.Tvm.SelfID,
					)
				}

				return s.sendErrorResponse(c, e)
			}

			c.SetRequest(c.Request().WithContext(
				tvm.WithServiceTicket(c.Request().Context(), ticket),
			))

			return next(c)
		}
	}
}

func (s *API) middlewareTvmUserTicketAuthentication() echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			ut := c.Request().Header.Get(xYaUserTicket)
			if ut == "" {
				return next(c)
			}

			ticket, err := s.tvmClient.CheckUserTicket(c.Request().Context(), ut)
			if err != nil {
				ctxlog.Debugf(c.Request().Context(), logger.Log(), "TVM error: %+v", err)
				e := &errs.AccessDeniedError{
					Message:      "user ticket is invalid",
					Description:  err.Error(),
					TicketStatus: err.(*tvm.TicketError).Status.String(),
				}
				if ticket != nil {
					e.LoggablePart = ticket.LogInfo
				}

				return s.sendErrorResponse(c, e)
			}

			c.SetRequest(c.Request().WithContext(
				tvm.WithUserTicket(c.Request().Context(), ticket),
			))

			return next(c)
		}
	}
}
