package api

import (
	"net/http"
	"sync"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/grants"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/hbaseapi"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/reqs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/resps"
)

func parseAuthsRequest(c echo.Context) (*reqs.AuthsRequest, error) {
	query := c.Request().URL.Query()

	uid, err := getRequiredUIntParam(query, "uid", nil)
	if err != nil {
		return nil, err
	}

	fromTS, err := getRequiredUIntParam(query, "from_ts", nil)
	if err != nil {
		return nil, err
	}

	toTS, err := getRequiredUIntParam(query, "to_ts", []uintValidator{&uintPositiveValidator})
	if err != nil {
		return nil, err
	}

	limit, err := getDefaultUIntParam(query, "limit", 1000, []uintValidator{&uintPositiveValidator})
	if err != nil {
		return nil, err
	}

	orderBy, err := getDefaultEnumParam(query, "order_by", orderByMapping, reqs.OrderByDesc)
	if err != nil {
		return nil, err
	}

	return &reqs.AuthsRequest{
		UID:        uid,
		FromTS:     fromTS,
		ToTS:       toTS,
		OrderBy:    orderBy,
		Limit:      limit,
		Status:     getStringListValue(query, "status"),
		Type:       getStringListValue(query, "type"),
		ClientName: getStringListValue(query, "client_name"),
	}, nil
}

func (s *API) AuthsHandler(failed bool) echo.HandlerFunc {
	return func(c echo.Context) error {
		consumer, err := s.grants.GetConsumer(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}
		if err = consumer.CheckAllowed(grants.KeyAuths, false); err != nil {
			return s.sendErrorResponse(c, err)
		}

		req, err := parseAuthsRequest(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		ytThreshold := s.config.Common.AuthsYtThreshold
		if failed {
			ytThreshold = 0
		}

		var oldAPIWaitGroup sync.WaitGroup
		oldAPIWaitGroup.Add(1)

		var oldAPIResponse *resps.AuthsResponse
		var oldAPIErr error
		go func() {
			oldAPIResponse, oldAPIErr = s.oldAPI.GetAuths(c.Request().Context(), req, failed, ytThreshold)
			oldAPIWaitGroup.Done()
		}()

		ytAuths, err := s.yt.GetAuths(c.Request().Context(), req, failed, ytThreshold)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		oldAPIWaitGroup.Wait()
		if oldAPIErr != nil {
			return s.sendErrorResponse(c, oldAPIErr)
		}

		return c.JSON(http.StatusOK, resps.AuthsResponse{
			Status: errs.ScalaStatusOk,
			UID:    req.UID,
			Auths:  mergeOrderedSlices(ytAuths, oldAPIResponse.Auths, int(req.Limit), req.OrderBy),
		})
	}
}

func parseAuthsAggregatedRequest(c echo.Context) (*reqs.AuthsAggregatedRequest, error) {
	query := c.Request().URL.Query()

	uid, err := getRequiredUIntParam(query, "uid", nil)
	if err != nil {
		return nil, err
	}

	limit, err := getOptionalUIntParam(query, "limit", []uintValidator{&uintPositiveValidator})
	if err != nil {
		return nil, err
	}

	hoursLimit, err := getOptionalUIntParam(query, "hours_limit", []uintValidator{&uintPositiveValidator})
	if err != nil {
		return nil, err
	}

	passwordAuths, err := getDefaultBoolParam(query, "password_auths", false)
	if err != nil {
		return nil, err
	}

	return &reqs.AuthsAggregatedRequest{
		UID:           uid,
		Limit:         limit,
		HoursLimit:    hoursLimit,
		From:          getOptionalStringParam(query, "from"),
		PasswordAuths: passwordAuths,
	}, nil
}

func (s *API) AuthsAggregatedHandler() echo.HandlerFunc {
	return func(c echo.Context) error {
		consumer, err := s.grants.GetConsumer(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}
		if err = consumer.CheckAllowed(grants.KeyAuths, false); err != nil {
			return s.sendErrorResponse(c, err)
		}

		req, err := parseAuthsAggregatedRequest(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		result, err := s.oldAPI.GetAuthsAggregated(c.Request().Context(), req)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		return c.JSON(http.StatusOK, result)
	}
}

func parseAuthsRuntimeAggregatedRequest(c echo.Context) (*reqs.AuthsRuntimeAggregatedRequest, error) {
	query := c.Request().URL.Query()

	uid, err := getRequiredUIntParam(query, "uid", nil)
	if err != nil {
		return nil, err
	}

	fromTS, err := getRequiredUIntParam(query, "from_ts", nil)
	if err != nil {
		return nil, err
	}

	toTS, err := getRequiredUIntParam(query, "to_ts", []uintValidator{&uintPositiveValidator})
	if err != nil {
		return nil, err
	}

	limit, err := getDefaultUIntParam(query, "limit", 1000, []uintValidator{&uintPositiveValidator})
	if err != nil {
		return nil, err
	}

	return &reqs.AuthsRuntimeAggregatedRequest{
		UID:    uid,
		FromTS: fromTS,
		ToTS:   toTS,
		Limit:  limit,
	}, nil
}

func (s *API) AuthsRuntimeAggregatedHandler() echo.HandlerFunc {
	return func(c echo.Context) error {
		consumer, err := s.grants.GetConsumer(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}
		if err = consumer.CheckAllowed(grants.KeyAuths, false); err != nil {
			return s.sendErrorResponse(c, err)
		}

		req, err := parseAuthsRuntimeAggregatedRequest(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		ytThreshold := s.config.Common.AuthsYtThreshold + hbaseapi.AuthsRuntimeAggregatedWindowWidth - 1
		ytThreshold = ytThreshold - ytThreshold%hbaseapi.AuthsRuntimeAggregatedWindowWidth

		var oldAPIWaitGroup sync.WaitGroup
		oldAPIWaitGroup.Add(1)

		var oldAPIResponse *resps.AuthsRuntimeAggregatedResponse
		var oldAPIErr error
		go func() {
			oldAPIResponse, oldAPIErr = s.oldAPI.GetAuthsRuntimeAggregated(c.Request().Context(), req, ytThreshold)
			oldAPIWaitGroup.Done()
		}()

		ytAuths, err := s.yt.GetAuthsRuntimeAggregated(c.Request().Context(), req, hbaseapi.AuthsRuntimeAggregatedWindowWidth, ytThreshold)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		oldAPIWaitGroup.Wait()
		if oldAPIErr != nil {
			return s.sendErrorResponse(c, oldAPIErr)
		}

		return c.JSON(http.StatusOK, resps.AuthsRuntimeAggregatedResponse{
			Status:  errs.ScalaStatusOk,
			UID:     req.UID,
			History: mergeOrderedSlices(ytAuths, oldAPIResponse.History, int(req.Limit), reqs.OrderByDesc),
		})
	}
}
