package api

import (
	"fmt"
	"net/http"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/grants"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/reqs"
)

func parseEventsRequest(c echo.Context) (*reqs.EventsRequest, error) {
	query := c.Request().URL.Query()

	uid, err := getRequiredUIntParam(query, "uid", nil)
	if err != nil {
		return nil, err
	}

	fromTS, err := getRequiredUIntParam(query, "from_ts", nil)
	if err != nil {
		return nil, err
	}

	toTS, err := getRequiredUIntParam(query, "to_ts", nil)
	if err != nil {
		return nil, err
	}

	limit, err := getOptionalUIntParam(query, "limit", []uintValidator{&uintPositiveValidator})
	if err != nil {
		return nil, err
	}

	orderBy, err := getDefaultEnumParam(query, "order_by", orderByMapping, reqs.OrderByDesc)
	if err != nil {
		return nil, err
	}

	return &reqs.EventsRequest{
		UID:     uid,
		FromTS:  fromTS,
		ToTS:    toTS,
		OrderBy: orderBy,
		Limit:   limit,
		Name:    getStringListValue(query, "name"),
	}, nil
}

func (s *API) EventsHandler() echo.HandlerFunc {
	return func(c echo.Context) error {
		consumer, err := s.grants.GetConsumer(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}
		if err = consumer.CheckAllowed(grants.KeyEvents, false); err != nil {
			return s.sendErrorResponse(c, err)
		}

		req, err := parseEventsRequest(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		result, err := s.oldAPI.GetEvents(c.Request().Context(), req)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		return c.JSON(http.StatusOK, result)
	}
}

func parseEventsRestoreRequest(c echo.Context) (*reqs.EventsRestoreRequest, error) {
	query := c.Request().URL.Query()

	uid, err := getRequiredUIntParam(query, "uid", nil)
	if err != nil {
		return nil, err
	}

	fromTS, err := getRequiredUIntParam(query, "from_ts", nil)
	if err != nil {
		return nil, err
	}

	toTS, err := getRequiredUIntParam(query, "to_ts", nil)
	if err != nil {
		return nil, err
	}

	limit, err := getDefaultUIntParam(query, "limit", 1000, []uintValidator{&uintPositiveValidator})
	if err != nil {
		return nil, err
	}

	orderBy, err := getDefaultEnumParam(query, "order_by", orderByMapping, reqs.OrderByDesc)
	if err != nil {
		return nil, err
	}

	return &reqs.EventsRestoreRequest{
		UID:     uid,
		FromTS:  fromTS,
		ToTS:    toTS,
		OrderBy: orderBy,
		Limit:   limit,
	}, nil
}

func (s *API) EventsRestoreHandler() echo.HandlerFunc {
	return func(c echo.Context) error {
		consumer, err := s.grants.GetConsumer(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}
		if err = consumer.CheckAllowed(grants.KeyEvents, false); err != nil {
			return s.sendErrorResponse(c, err)
		}

		req, err := parseEventsRestoreRequest(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		result, err := s.yt.GetRestore(c.Request().Context(), req, s.keys.Restore)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		return c.JSON(http.StatusOK, result)
	}
}

func parseEventsByIPRegistrationsRequest(c echo.Context) (*reqs.EventsByIPRegistrationsRequest, error) {
	query := c.Request().URL.Query()

	subnet, err := getRequiredStringParam(query, "subnet")
	if err != nil {
		return nil, err
	}

	fromTS, err := getRequiredUIntParam(query, "from_ts", nil)
	if err != nil {
		return nil, err
	}

	toTS, err := getRequiredUIntParam(query, "to_ts", nil)
	if err != nil {
		return nil, err
	}

	limit, err := getOptionalUIntParam(query, "limit", []uintValidator{
		&uintPositiveValidator,
		&uintMaxLimitValidator{max: 10000},
	})
	if err != nil {
		return nil, err
	}

	orderBy, err := getDefaultEnumParam(query, "order_by", orderByMapping, reqs.OrderByDesc)
	if err != nil {
		return nil, err
	}

	return &reqs.EventsByIPRegistrationsRequest{
		SubNet:  subnet,
		FromTS:  fromTS,
		ToTS:    toTS,
		OrderBy: orderBy,
		Limit:   limit,
	}, nil
}

func (s *API) EventsByIPRegistrationsHandler() echo.HandlerFunc {
	return func(c echo.Context) error {
		consumer, err := s.grants.GetConsumer(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}
		if err = consumer.CheckAllowed(grants.KeyEvents, false); err != nil {
			return s.sendErrorResponse(c, err)
		}

		req, err := parseEventsByIPRegistrationsRequest(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		result, err := s.oldAPI.GetEventsByIPRegistrations(c.Request().Context(), req)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		return c.JSON(http.StatusOK, result)
	}
}

func parseEventsPasswordsRequest(c echo.Context) (*reqs.EventsPasswordsRequest, error) {
	if err := c.Request().ParseForm(); err != nil {
		return nil, &errs.BadRequestError{
			Message: fmt.Sprintf("failed to parse form: %s", err),
		}
	}
	query := c.Request().Form

	password, err := getRequiredStringParam(query, "password")
	if err != nil {
		return nil, err
	}

	uid, err := getRequiredUIntParam(query, "uid", nil)
	if err != nil {
		return nil, err
	}

	fromTS, err := getRequiredUIntParam(query, "from_ts", nil)
	if err != nil {
		return nil, err
	}

	toTS, err := getRequiredUIntParam(query, "to_ts", nil)
	if err != nil {
		return nil, err
	}

	limit, err := getOptionalUIntParam(query, "limit", []uintValidator{&uintPositiveValidator})
	if err != nil {
		return nil, err
	}

	return &reqs.EventsPasswordsRequest{
		Password: password,
		UID:      uid,
		FromTS:   fromTS,
		ToTS:     toTS,
		Limit:    limit,
	}, nil
}

func (s *API) EventsPasswordsHandler() echo.HandlerFunc {
	return func(c echo.Context) error {
		consumer, err := s.grants.GetConsumer(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}
		if err = consumer.CheckAllowed(grants.KeyEvents, false); err != nil {
			return s.sendErrorResponse(c, err)
		}

		req, err := parseEventsPasswordsRequest(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		result, err := s.oldAPI.GetEventsPasswords(c.Request().Context(), req)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		return c.JSON(http.StatusOK, result)
	}
}
