package api

import (
	"net/http"
	"sync"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/grants"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/reqs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/resps"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

func ParseMailUserHistoryRequest(c echo.Context) (*reqs.MailUserHistoryRequest, error) {
	query := c.Request().URL.Query()

	uid, err := getRequiredUIntParam(query, "uid", []uintValidator{&uintPositiveValidator})
	if err != nil {
		return nil, err
	}

	corp, err := getDefaultBoolParam(query, "corp", false)
	if err != nil {
		return nil, err
	}

	fromTS, err := getRequiredUIntParam(query, "from_ts", nil)
	if err != nil {
		return nil, err
	}

	toTS, err := getRequiredUIntParam(query, "to_ts", []uintValidator{&uintPositiveValidator})
	if err != nil {
		return nil, err
	}

	limit, err := getDefaultUIntParam(query, "limit", 1000, []uintValidator{&uintPositiveValidator})
	if err != nil {
		return nil, err
	}

	return &reqs.MailUserHistoryRequest{
		UID:       uid,
		Corp:      corp,
		FromTS:    fromTS,
		ToTS:      toTS,
		Limit:     limit,
		Operation: getStringListValue(query, "operation"),
		Module:    getStringListValue(query, "module"),
	}, nil
}

func (s *API) MailUserHistoryHandler() echo.HandlerFunc {
	return func(c echo.Context) error {
		consumer, err := s.grants.GetConsumer(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}
		if err = consumer.CheckAllowed(grants.KeyMailUserHistory, false); err != nil {
			return s.sendErrorResponse(c, err)
		}

		req, err := ParseMailUserHistoryRequest(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		var oldAPIResponse sync.WaitGroup
		oldAPIResponse.Add(1)

		var oldAPIResult *resps.MailUserHistoryResult
		var oldAPIErr error
		go func() {
			oldAPIResult, oldAPIErr = s.oldAPI.GetMailUserHistory(c.Request().Context(), req)
			oldAPIResponse.Done()
		}()

		ytResult, err := s.yt.GetMailUserHistory(c.Request().Context(), req)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		oldAPIResponse.Wait()
		if oldAPIErr != nil {
			return s.sendErrorResponse(c, oldAPIErr)
		}

		ctxlog.Debugf(c.Request().Context(), logger.Log(),
			"MailUserHistory: Got %d events from YT, %d events from HBase",
			len(ytResult.Items), len(oldAPIResult.Items),
		)
		result, err := mergeMailUserHistoryResults(c.Request().Context(), ytResult, oldAPIResult)
		if err != nil {
			// should never happen
			return s.sendErrorResponse(c, err)
		}

		return c.JSON(http.StatusOK, result)
	}
}
