package api

import (
	"context"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/reqs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/resps"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

func mergeOrderedSlices[Slice ~[]Item, Item any](
	yt, hbase Slice,
	limit int,
	order reqs.OrderByType,
) Slice {
	ytRows := len(yt)
	hbaseRows := len(hbase)

	if hbaseRows == 0 || ytRows >= limit {
		return yt
	}
	if ytRows == 0 {
		return hbase
	}

	res := make(Slice, 0, min(hbaseRows+ytRows, limit))

	switch order {
	case reqs.OrderByAsc:
		res = append(hbase[max(hbaseRows+ytRows-limit, 0):], yt...)
	case reqs.OrderByDesc:
		res = append(yt, hbase[:min(hbaseRows, limit-ytRows)]...)
	}

	return res
}

func mergeMailUserHistoryResults(
	ctx context.Context,
	yt, hbase *resps.MailUserHistoryResult,
) (*resps.MailUserHistoryResult, error) {
	if yt.Status != hbase.Status {
		return nil, xerrors.Errorf("statuses are not equal; yt=%s, hbase=%s", yt.Status, hbase.Status)
	}
	if yt.UID != hbase.UID {
		return nil, xerrors.Errorf("uids are not equal; yt=%d, hbase=%d", yt.UID, hbase.UID)
	}

	if len(hbase.Items) == 0 {
		return yt, nil
	}
	if len(yt.Items) == 0 {
		return hbase, nil
	}

	res := &resps.MailUserHistoryResult{
		Status: yt.Status,
		UID:    yt.UID,
		Items:  make([]*resps.MailUserHistoryItem, 0, max(len(yt.Items), len(hbase.Items))),
	}

	mergeMailUserHistoryItems(ctx, yt.Items, hbase.Items, &res.Items)

	return res, nil
}

func mergeMailUserHistoryItems(
	ctx context.Context,
	yt, hbase []*resps.MailUserHistoryItem,
	out *[]*resps.MailUserHistoryItem,
) {
	ytIdx := 0
	hbaseIdx := 0

	for ytIdx < len(yt) && hbaseIdx < len(hbase) {
		ytItem := yt[ytIdx]
		hbaseItem := hbase[hbaseIdx]

		if ytItem.After(hbaseItem) {
			(*out) = append((*out), ytItem)
			ytIdx++
		} else if hbaseItem.After(ytItem) {
			(*out) = append((*out), hbaseItem)
			hbaseIdx++
		} else {
			if !ytItem.Equals(hbaseItem) {
				ctxlog.Warnf(ctx, logger.Log(),
					"Items have same date but have not equal key: yt(%d,%s,%s) vs hbase(%d,%s,%s)",
					ytItem.Common.Date, ytItem.Common.Operation, ytItem.Common.Module,
					hbaseItem.Common.Date, hbaseItem.Common.Operation, hbaseItem.Common.Module,
				)
			}
			(*out) = append((*out), ytItem)
			ytIdx++
			hbaseIdx++
		}
	}

	for ytIdx < len(yt) {
		(*out) = append((*out), yt[ytIdx])
		ytIdx++
	}
	for hbaseIdx < len(hbase) {
		(*out) = append((*out), hbase[hbaseIdx])
		hbaseIdx++
	}
}

func min(l, r int) int {
	if l < r {
		return l
	}
	return r
}

func max(l, r int) int {
	if l > r {
		return l
	}
	return r
}
