package api

import (
	"net/http"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/grants"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/reqs"
)

func ParsePushByPushIDRequest(c echo.Context) (*reqs.PushByPushIDRequest, error) {
	query := c.Request().URL.Query()

	push, err := getRequiredStringParam(query, "push")
	if err != nil {
		return nil, err
	}

	uid, err := getOptionalUIntParam(query, "uid", []uintValidator{&uintPositiveValidator})
	if err != nil {
		return nil, err
	}

	return &reqs.PushByPushIDRequest{
		Push:         push,
		Subscription: getOptionalStringParam(query, "subscription"),
		UID:          uid,
		Device:       getOptionalStringParam(query, "device"),
		App:          getOptionalStringParam(query, "app"),
	}, nil
}

func (s *API) PushByPushIDHandler() echo.HandlerFunc {
	return func(c echo.Context) error {
		consumer, err := s.grants.GetConsumer(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}
		if err = consumer.CheckAllowed(grants.KeyPushBase, true); err != nil {
			return s.sendErrorResponse(c, err)
		}

		req, err := ParsePushByPushIDRequest(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		result, err := s.yt.GetPushByPushID(c.Request().Context(), req)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		return c.JSON(http.StatusOK, result)
	}
}

func ParsePushByFieldsRequest(c echo.Context) (*reqs.PushByFieldsRequest, error) {
	query := c.Request().URL.Query()

	uid, err := getOptionalUIntParam(query, "uid", []uintValidator{&uintPositiveValidator})
	if err != nil {
		return nil, err
	}

	device := getOptionalStringParam(query, "device")

	if uid == nil && device == nil {
		return nil, &errs.InvalidParamError{
			ScalaKey:  "uid/device",
			ScalaCode: errs.ScalaErrorRequired,
			Message:   "missing required push parameter: require at least one of 'uid' or 'device' parameter",
		}
	}

	fromTS, err := getRequiredUIntParam(query, "from_ts", nil)
	if err != nil {
		return nil, err
	}

	toTS, err := getRequiredUIntParam(query, "to_ts", nil)
	if err != nil {
		return nil, err
	}

	limit, err := getDefaultUIntParam(query, "limit", 1000, []uintValidator{&uintPositiveValidator})
	if err != nil {
		return nil, err
	}

	return &reqs.PushByFieldsRequest{
		UID:          uid,
		Device:       device,
		App:          getOptionalStringParam(query, "app"),
		Subscription: getOptionalStringParam(query, "subscription"),
		FromTS:       fromTS,
		ToTS:         toTS,
		Limit:        limit,
	}, nil
}

func (s *API) PushByFieldsHandler() echo.HandlerFunc {
	return func(c echo.Context) error {
		consumer, err := s.grants.GetConsumer(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}
		if err = consumer.CheckAllowed(grants.KeyPushBase, true); err != nil {
			return s.sendErrorResponse(c, err)
		}

		req, err := ParsePushByFieldsRequest(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		result, err := s.yt.GetPushByFields(c.Request().Context(), req)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		return c.JSON(http.StatusOK, result)
	}
}
