package api

import (
	"net/http"
	"sort"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/grants"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/reqs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/resps"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/ytc"
)

const (
	maxAgeMonth = uint64(86400 * 31)
)

func ParsePushSubscriptionRequest(c echo.Context) (*reqs.PushSubscriptionRequest, error) {
	query := c.Request().URL.Query()

	uid, err := getRequiredUIntParam(query, "uid", []uintValidator{&uintPositiveValidator})
	if err != nil {
		return nil, err
	}

	magAgeParam, err := getDefaultUIntParam(query, "max_age", maxAgeMonth, []uintValidator{&uintPositiveValidator})
	if err != nil {
		return nil, err
	}

	limit, err := getDefaultUIntParam(query, "limit", 100, []uintValidator{&uintPositiveValidator})
	if err != nil {
		return nil, err
	}

	return &reqs.PushSubscriptionRequest{
		UID:    uid,
		MaxAge: magAgeParam,
		Limit:  limit,
	}, nil
}

func (s *API) PushSubscriptionHandler() echo.HandlerFunc {
	return func(c echo.Context) error {
		consumer, err := s.grants.GetConsumer(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}
		if err = consumer.CheckAllowed(grants.KeyPushSubscriptionBase, true); err != nil {
			return s.sendErrorResponse(c, err)
		}

		req, err := ParsePushSubscriptionRequest(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		// Counters are aggregated by window:
		//   let's add window width to get counter from whole window
		req.MaxAge += s.config.Common.PushSubscriptionWidth

		rows, err := s.yt.GetPushSubscription(c.Request().Context(), req)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		result := buildPushSubscriptionResult(rows, req.UID, req.Limit)

		return c.JSON(http.StatusOK, result)
	}
}

func buildPushSubscriptionResult(rows []ytc.PushSubscriptionRow, uid, limit uint64) *resps.PushSubscriptionResult {
	type RowKey struct {
		AppID    string `json:"app_id"`
		DeviceID string `json:"device_id"`
	}
	counters := make(map[RowKey]uint64)

	for _, row := range rows {
		key := RowKey{AppID: row.AppID, DeviceID: row.DeviceID}
		counters[key] = counters[key] + row.Count
	}

	result := &resps.PushSubscriptionResult{
		Status: errs.ScalaStatusOk,
		UID:    uid,
		Items:  make([]resps.PushSubscriptionItem, 0, len(counters)),
	}
	for k, v := range counters {
		result.Items = append(result.Items, resps.PushSubscriptionItem{
			AppID:    k.AppID,
			DeviceID: k.DeviceID,
			Count:    v,
		})
	}

	sort.Slice(result.Items, func(i, j int) bool {
		return result.Items[i].Count > result.Items[j].Count
	})
	if uint64(len(result.Items)) > limit {
		result.Items = result.Items[:limit]
	}

	return result
}
