package api

import (
	"fmt"
	"net/url"
	"strconv"
	"strings"

	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/reqs"
)

var orderByMapping = map[string]int{
	"asc":  reqs.OrderByAsc,
	"desc": reqs.OrderByDesc,
}

var uintPositiveValidator = uintMinLimitValidator{min: 1}

type uintValidator interface {
	validate(value uint64, key string) error
}

type uintMinLimitValidator struct {
	min uint64
}

func (v *uintMinLimitValidator) validate(value uint64, key string) error {
	if value < v.min {
		return &errs.InvalidParamError{
			ScalaKey:  key,
			ScalaCode: errs.ScalaErrorMin,
			Message:   fmt.Sprintf("'%s' param should be greater than '%d', got '%d'", key, v.min, value),
		}
	}

	return nil
}

type uintMaxLimitValidator struct {
	max uint64
}

func (v *uintMaxLimitValidator) validate(value uint64, key string) error {
	if value > v.max {
		return &errs.InvalidParamError{
			ScalaKey:  key,
			ScalaCode: errs.ScalaErrorMax,
			Message:   fmt.Sprintf("'%s' param should be lesser than '%d', got '%d'", key, v.max, value),
		}
	}

	return nil
}

func getOptionalUIntParam(values url.Values, key string, validators []uintValidator) (*uint64, error) {
	value := values.Get(key)
	if value == "" {
		return nil, nil
	}

	valueInt, err := strconv.ParseUint(value, 10, 64)
	if err != nil {
		return nil, &errs.InvalidParamError{
			ScalaKey:  key,
			ScalaCode: errs.ScalaErrorNumber,
			Message:   fmt.Sprintf("invalid param: '%s' must be unsigned int, got '%s'", key, value),
		}
	}

	for _, validator := range validators {
		if err := validator.validate(valueInt, key); err != nil {
			return nil, err
		}
	}

	return &valueInt, nil
}

func getDefaultUIntParam(values url.Values, key string, def uint64, validators []uintValidator) (uint64, error) {
	val, err := getOptionalUIntParam(values, key, validators)
	if err != nil {
		return 0, err
	}

	if val == nil {
		return def, nil
	}

	return *val, nil
}

func getRequiredUIntParam(values url.Values, key string, validators []uintValidator) (uint64, error) {
	val, err := getOptionalUIntParam(values, key, validators)
	if err != nil {
		return 0, err
	}

	if val == nil {
		return 0, &errs.InvalidParamError{
			ScalaKey:  key,
			ScalaCode: errs.ScalaErrorRequired,
			Message:   fmt.Sprintf("missing param: '%s'", key),
		}
	}

	return *val, nil
}

func getOptionalStringParam(values url.Values, key string) *string {
	if !values.Has(key) {
		return nil
	}

	value := values.Get(key)

	return &value
}

func getRequiredStringParam(values url.Values, key string) (string, error) {
	if !values.Has(key) {
		return "", &errs.InvalidParamError{
			ScalaKey:  key,
			ScalaCode: errs.ScalaErrorRequired,
			Message:   fmt.Sprintf("missing param: '%s'", key),
		}
	}

	return values.Get(key), nil
}

func getOptionalBoolParam(values url.Values, key string) (*bool, error) {
	value := values.Get(key)
	if value == "" {
		return nil, nil
	}

	var res bool
	if strings.EqualFold(value, "false") {
		res = false
	} else if strings.EqualFold(value, "true") {
		res = true
	} else {
		return nil, &errs.InvalidParamError{
			ScalaKey:  key,
			ScalaCode: errs.ScalaErrorBoolean,
			Message:   fmt.Sprintf("invalid param: '%s' must be boolean, got '%s'", key, value),
		}
	}

	return &res, nil
}

func getDefaultBoolParam(values url.Values, key string, def bool) (bool, error) {
	value, err := getOptionalBoolParam(values, key)
	if err != nil {
		return false, err
	}

	if value == nil {
		return def, nil
	}

	return *value, nil
}

func getStringListValue(values url.Values, key string) []string {
	val := values.Get(key)
	if len(val) == 0 {
		return nil
	}
	return strings.Split(val, ",")
}

func getUIntListValue(values url.Values, key string) ([]uint64, error) {
	valueRaw := values.Get(key)
	if valueRaw == "" {
		return nil, nil
	}

	list := strings.Split(valueRaw, ",")
	result := make([]uint64, 0, len(list))
	for _, valueStr := range list {
		valueInt, err := strconv.ParseUint(valueStr, 10, 64)
		if err != nil {
			return nil, &errs.InvalidParamError{
				ScalaKey:  key,
				ScalaCode: errs.ScalaErrorNumber,
				Message:   fmt.Sprintf("invalid param: '%s' elements must be unsigned int, got '%s'", key, valueStr),
			}
		}

		result = append(result, valueInt)
	}

	return result, nil
}

func getOptionalEnumParam(values url.Values, key string, mapping map[string]int) (*int, error) {
	value := getOptionalStringParam(values, key)
	if value == nil || *value == "" {
		return nil, nil
	}

	res, exists := mapping[*value]
	if !exists {
		return nil, &errs.InvalidParamError{
			ScalaKey:  key,
			ScalaCode: errs.ScalaErrorUnknown,
			Message:   fmt.Sprintf("unknown '%s' param value: '%s'", key, *value),
		}
	}

	return &res, nil
}

func getDefaultEnumParam(values url.Values, key string, mapping map[string]int, def int) (int, error) {
	value, err := getOptionalEnumParam(values, key, mapping)
	if err != nil {
		return -1, err
	}

	if value == nil {
		return def, nil
	}

	return *value, nil
}
