package api

import (
	"fmt"
	"net/http"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type ScalaErrorResponse struct {
	Status errs.ScalaStatus                 `json:"status"`
	Errors map[string][]errs.ScalaErrorCode `json:"errors"`
}

type TmpErrorResponse struct {
	Error string `json:"error"`
}

const (
	tmpErrorMessage = "Historydb API backend error"
)

func (s *API) sendErrorResponse(c echo.Context, err error) error {
	if err == nil {
		err = xerrors.New("some error occurred but original message was lost")
	}

	switch err := err.(type) {
	case *errs.InvalidParamError:
		s.stats.invalidParams.Inc()
		ctxlog.Debugf(c.Request().Context(), logger.Log(), "%s", err)
		return c.JSON(http.StatusOK, buildScalaErrorResponse(err.ScalaKey, err.ScalaCode))

	case *errs.BadRequestError:
		s.stats.invalidParams.Inc()
		ctxlog.Debugf(c.Request().Context(), logger.Log(), "%s", err)
		return c.String(http.StatusBadRequest, err.Error())

	case *errs.AccessDeniedError:
		s.stats.accessDenied.Inc()
		txt := fmt.Sprintf("Access denied for ip: %s; %s; '%s'", c.RealIP(), err, err.LoggablePart)
		ctxlog.Debugf(c.Request().Context(), logger.Log(), "%s", txt)
		return c.String(http.StatusForbidden, txt)

	case *errs.TemporaryError:
		switch err.Type {
		case errs.DecryptError:
			s.stats.decryptErr.Inc()
		default:
			s.stats.tmpErr.Inc()
		}
		ctxlog.Debugf(c.Request().Context(), logger.Log(), "%s", err)
		return c.JSON(http.StatusInternalServerError, TmpErrorResponse{Error: tmpErrorMessage})
	}

	s.stats.unknown.Inc()
	ctxlog.Errorf(c.Request().Context(), logger.Log(), "Unexpected error: %s", err)
	return c.JSON(http.StatusInternalServerError, TmpErrorResponse{Error: err.Error()})
}

func buildScalaErrorResponse(key string, code errs.ScalaErrorCode) ScalaErrorResponse {
	return ScalaErrorResponse{
		Status: errs.ScalaStatusError,
		Errors: map[string][]errs.ScalaErrorCode{
			key: {code},
		},
	}
}
