package api

import (
	"net/http"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/grants"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/hbaseapi"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/model"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/reqs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/ytc"
)

func (s *API) SmsByGlobalIDHandler() echo.HandlerFunc {
	return func(c echo.Context) error {
		consumer, err := s.grants.GetConsumer(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}
		if err = consumer.CheckAllowed(grants.KeySmsHistory, false); err != nil {
			return s.sendErrorResponse(c, err)
		}

		globalSmsID, err := getRequiredStringParam(c.Request().URL.Query(), "global_smsid")
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		req := &reqs.SmsByGlobalIDRequest{
			GlobalSmsID: globalSmsID,
			Limit:       1000,
		}

		var result *hbaseapi.SmsByGlobalIDResult
		if s.config.Common.YasmsSmsHistoryFromYt {
			collector := model.NewSmsItemsCollector(s.keys.YasmsPrivate)
			if err := s.yt.SelectYasmsSmsHistory(c.Request().Context(), req, ytc.YasmsSmsHistoryFull, collector); err != nil {
				return s.sendErrorResponse(c, err)
			}

			result = &hbaseapi.SmsByGlobalIDResult{
				Status:      errs.ScalaStatusOk,
				GlobalSmsID: req.GlobalSmsID,
				Items:       collector.Finish(),
			}
		} else {
			result, err = s.oldAPI.GetSmsByGlobalID(c.Request().Context(), req)
			if err != nil {
				return s.sendErrorResponse(c, err)
			}
		}

		return c.JSON(http.StatusOK, result)
	}
}

func parseSmsByPhoneRequest(c echo.Context) (*reqs.SmsByFieldsRequest, error) {
	query := c.Request().URL.Query()

	phone, err := getRequiredStringParam(query, "phone")
	if err != nil {
		return nil, err
	}

	fromTS, err := getOptionalUIntParam(query, "from_ts", nil)
	if err != nil {
		return nil, err
	}

	toTS, err := getOptionalUIntParam(query, "to_ts", nil)
	if err != nil {
		return nil, err
	}

	limit, err := getDefaultUIntParam(query, "limit", 1000, []uintValidator{&uintPositiveValidator})
	if err != nil {
		return nil, err
	}

	return &reqs.SmsByFieldsRequest{
		Phone:  &phone,
		FromTS: fromTS,
		ToTS:   toTS,
		Limit:  limit,
	}, nil
}

func (s *API) SmsByPhoneHandler() echo.HandlerFunc {
	return func(c echo.Context) error {
		consumer, err := s.grants.GetConsumer(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}
		if err = consumer.CheckAllowed(grants.KeySmsHistory, false); err != nil {
			return s.sendErrorResponse(c, err)
		}

		req, err := parseSmsByPhoneRequest(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		var result *hbaseapi.SmsByPhoneResult
		if s.config.Common.YasmsSmsHistoryFromYt {
			collector := model.NewSmsItemsByGlobalIDCollector(s.keys.YasmsPrivate)
			if err := s.yt.SelectYasmsSmsHistoryByFields(c.Request().Context(), req, ytc.YasmsSmsHistoryFull, collector); err != nil {
				return s.sendErrorResponse(c, err)
			}

			result = &hbaseapi.SmsByPhoneResult{
				Status: errs.ScalaStatusOk,
				Phone:  *req.Phone,
				Items:  collector.Finish(),
			}
		} else {
			result, err = s.oldAPI.GetSmsByPhone(c.Request().Context(), req)
			if err != nil {
				return s.sendErrorResponse(c, err)
			}
		}

		return c.JSON(http.StatusOK, result)
	}
}

type SmsLastStatusByGlobalIDResponse struct {
	Status      errs.ScalaStatus   `json:"status"`
	GlobalSmsID string             `json:"global_smsid"`
	LastStatus  *model.SmsItemBase `json:"laststatus"`
}

func (s *API) SmsLastStatusByGlobalIDHandler() echo.HandlerFunc {
	return func(c echo.Context) error {
		consumer, err := s.grants.GetConsumer(c)
		if err != nil {
			return s.sendErrorResponse(c, err)
		}
		if err = consumer.CheckAllowed(grants.KeySmsLastStatus, false); err != nil {
			return s.sendErrorResponse(c, err)
		}

		globalSmsID, err := getRequiredStringParam(c.Request().URL.Query(), "global_smsid")
		if err != nil {
			return s.sendErrorResponse(c, err)
		}

		req := &reqs.SmsByGlobalIDRequest{
			GlobalSmsID: globalSmsID,
			Limit:       1,
		}

		collector := model.NewSmsItemsCollector(nil)
		if err := s.yt.SelectYasmsSmsHistory(c.Request().Context(), req, ytc.YasmsSmsHistoryBase, collector); err != nil {
			return s.sendErrorResponse(c, err)
		}

		result := SmsLastStatusByGlobalIDResponse{
			Status:      errs.ScalaStatusOk,
			GlobalSmsID: req.GlobalSmsID,
		}

		res := collector.Finish()
		if len(res) > 0 {
			result.LastStatus = res[0].Common.SmsItemBase
		}

		return c.JSON(http.StatusOK, result)
	}
}
