package grants

import (
	"fmt"
	"sync"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

func NewGrants(cfg Config) (*Grants, error) {
	res := Grants{
		cfg:   cfg,
		mutex: &sync.RWMutex{},
	}
	if err := res.update(); err != nil {
		return nil, err
	}

	go func() {
		for range time.NewTicker(time.Duration(cfg.Period) * time.Second).C {
			if err := res.update(); err != nil {
				logger.Log().Warnf("Failed to update grants: %s", err)
			}
		}
	}()

	return &res, nil
}

func (g *Grants) GetConsumer(c echo.Context) (*Consumer, error) {
	addr := c.RealIP()

	name := c.Request().URL.Query().Get("consumer")
	if name == "" {
		return nil, &errs.AccessDeniedError{
			Message: "missing 'consumer' in request",
		}
	}

	state := g.getState()
	consumer, ok := state.consumers[name]
	if !ok {
		return nil, &errs.AccessDeniedError{
			Message: fmt.Sprintf("unknown consumer: '%s'", name),
		}
	}

	if !consumer.nets.Contains(addr) {
		return nil, &errs.AccessDeniedError{
			Message: fmt.Sprintf("consumer '%s' cannot use address", name),
		}
	}

	if consumer.tvmClientID != nil {
		st := tvm.ContextServiceTicket(c.Request().Context())
		if st == nil {
			return nil, &errs.AccessDeniedError{
				Message: "missing service ticket",
			}
		}
		if *consumer.tvmClientID != st.SrcID {
			return nil, &errs.AccessDeniedError{
				Message: fmt.Sprintf(
					"service ticket from client_id %d does not match consumer %s",
					st.SrcID,
					consumer.name,
				),
			}
		}
	}

	return consumer, nil
}

func (g *Grants) getState() *State {
	g.mutex.RLock()
	defer g.mutex.RUnlock()

	return g.state
}

func (c *Consumer) CheckAllowed(key string, isTVMRequired bool) error {
	if isTVMRequired && c.tvmClientID == nil {
		return &errs.AccessDeniedError{
			Message: fmt.Sprintf("consumer '%s' is not allowed (TVM required)", c.name),
		}
	}

	_, ok := c.keys[key]
	if !ok {
		return &errs.AccessDeniedError{
			Message: fmt.Sprintf("consumer '%s' missing required grant: %s", c.name, key),
		}
	}

	return nil
}
