package grants

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"os"

	"golang.org/x/xerrors"

	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/libs/go/grantnets"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

func (g *Grants) update() error {
	info, err := os.Stat(g.cfg.Path)
	if err != nil {
		return xerrors.Errorf("failed to stat file: %s: %w", g.cfg.Path, err)
	}

	if g.getState() != nil && g.lastMTime == info.ModTime() {
		// grants are the same
		return nil
	}

	blob, err := ioutil.ReadFile(g.cfg.Path)
	if err != nil {
		return xerrors.Errorf("failed to get file as blob: %s: %w", g.cfg.Path, err)
	}

	s, err := parseState(blob)
	if err != nil {
		return xerrors.Errorf("failed to parse grants: %s: %w", g.cfg.Path, err)
	}

	g.mutex.Lock()
	defer g.mutex.Unlock()

	g.state = s
	g.lastMTime = info.ModTime()
	logger.Log().Info("Grants were updated")

	return nil
}

func parseState(blob []byte) (*State, error) {
	type (
		client struct {
			ID   *tvm.ClientID `json:"client_id"`
			Name string        `json:"client_name"`
		}
		consumer struct {
			Client   client              `json:"client"`
			Gr       map[string][]string `json:"grants"`
			Networks []string            `json:"networks"`
		}
	)

	grants := make(map[string]consumer)
	if err := json.Unmarshal(blob, &grants); err != nil {
		return nil, err
	}

	res := &State{
		consumers: make(map[string]*Consumer),
	}

	for name, cons := range grants {
		c := &Consumer{
			name: name,
			nets: grantnets.NewNets(),
			keys: make(map[string]interface{}),
		}

		c.tvmClientID = cons.Client.ID

		for k1, v := range cons.Gr {
			for _, k2 := range v {
				c.keys[fmt.Sprintf("%s.%s", k1, k2)] = 0
			}
		}

		for _, netw := range cons.Networks {
			if err := c.nets.Add(netw); err != nil {
				return nil, xerrors.Errorf("bad network for '%s': %w", name, err)
			}
		}

		res.consumers[name] = c
	}

	return res, nil
}
