package hbaseapi

import (
	"context"
	"encoding/json"
	"fmt"
	"strconv"
	"strings"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/reqs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/resps"
)

const (
	AuthsRuntimeAggregatedWindowWidth = uint64(24 * 60 * 60)
)

func (c *Client) GetAuths(ctx context.Context, req *reqs.AuthsRequest, failed bool, upperTSLimit uint64) (*resps.AuthsResponse, error) {
	res := &resps.AuthsResponse{
		Status: errs.ScalaStatusOk,
		UID:    req.UID,
		Auths:  make([]*resps.PlainAuth, 0),
	}

	toTS := chooseToTS(req.ToTS, upperTSLimit)
	if toTS < req.FromTS {
		return res, nil
	}

	orderBy, err := getOrderByParam(req.OrderBy)
	if err != nil {
		return nil, xerrors.Errorf("Incorrect request: %s", err.Error())
	}

	newHTTPReq := c.newRequestWithConsumer(ctx).
		SetQueryParam("uid", strconv.FormatUint(req.UID, 10)).
		SetQueryParam("from_ts", strconv.FormatUint(req.FromTS, 10)).
		SetQueryParam("to_ts", strconv.FormatUint(toTS, 10)).
		SetQueryParam("order_by", orderBy).
		SetQueryParam("limit", strconv.FormatUint(req.Limit, 10))

	if len(req.Status) != 0 {
		newHTTPReq.SetQueryParam("status", strings.Join(req.Status, ","))
	}
	if len(req.Type) != 0 {
		newHTTPReq.SetQueryParam("type", strings.Join(req.Type, ","))
	}
	if len(req.ClientName) != 0 {
		newHTTPReq.SetQueryParam("client_name", strings.Join(req.ClientName, ","))
	}

	url := "/2/auths/"
	if failed {
		url = "/2/auths/failed/"
	}
	err = c.withRetries(newHTTPReq, resty.MethodGet, url, func(resp *resty.Response) error {
		if err := json.Unmarshal(resp.Body(), &res); err != nil {
			return xerrors.Errorf("Failed to parse json: [%d] '%s'. Error: %s", resp.StatusCode(), resp.String(), err)
		}

		if err := c.checkStatusOk(res.Status); err != nil {
			return xerrors.Errorf("Request failed: [%d] %s. Error: %s", resp.StatusCode(), resp.String(), err)
		}

		return nil
	})
	if err != nil {
		c.unistat.errs.Inc()
		return nil, &errs.TemporaryError{
			Message: fmt.Sprintf("Failed to fetch auths from Old API: %v", err),
		}
	}

	c.unistat.authsRows.Add(float64(len(res.Auths)))

	return res, nil
}

func (c *Client) GetAuthsAggregated(ctx context.Context, req *reqs.AuthsAggregatedRequest) (*resps.AuthsAggregatedResponse, error) {
	newHTTPReq := c.newRequestWithConsumer(ctx).
		SetQueryParam("uid", strconv.FormatUint(req.UID, 10))

	if req.Limit != nil {
		newHTTPReq.SetQueryParam("limit", strconv.FormatUint(*req.Limit, 10))
	}
	if req.HoursLimit != nil {
		newHTTPReq.SetQueryParam("hours_limit", strconv.FormatUint(*req.HoursLimit, 10))
	}
	if req.From != nil {
		newHTTPReq.SetQueryParam("from", *req.From)
	}
	if req.PasswordAuths {
		newHTTPReq.SetQueryParam("password_auths", "true")
	}

	res := &resps.AuthsAggregatedResponse{}

	err := c.withRetries(newHTTPReq, resty.MethodGet, "/3/auths/aggregated/", func(resp *resty.Response) error {
		if err := json.Unmarshal(resp.Body(), &res); err != nil {
			return xerrors.Errorf("Failed to parse json: [%d] '%s'. Error: %s", resp.StatusCode(), resp.String(), err)
		}

		if err := c.checkStatusOk(res.Status); err != nil {
			return xerrors.Errorf("Request failed: [%d] %s. Error: %s", resp.StatusCode(), resp.String(), err)
		}

		return nil
	})
	if err != nil {
		c.unistat.errs.Inc()
		return nil, &errs.TemporaryError{
			Message: fmt.Sprintf("Failed to fetch aggregated auths from Old API: %v", err),
		}
	}

	c.unistat.authsRows.Add(float64(len(res.Auths)))

	return res, nil
}

func (c *Client) GetAuthsRuntimeAggregated(ctx context.Context, req *reqs.AuthsRuntimeAggregatedRequest, upperTSLimit uint64) (*resps.AuthsRuntimeAggregatedResponse, error) {
	res := &resps.AuthsRuntimeAggregatedResponse{
		Status:  errs.ScalaStatusOk,
		UID:     req.UID,
		History: make([]resps.AuthsRuntimeAggregatedEntry, 0),
	}

	toTS := chooseToTS(req.ToTS, upperTSLimit)
	if toTS < req.FromTS {
		return res, nil
	}

	newHTTPReq := c.newRequestWithConsumer(ctx).
		SetQueryParam("uid", strconv.FormatUint(req.UID, 10)).
		SetQueryParam("from_ts", strconv.FormatUint(req.FromTS, 10)).
		SetQueryParam("to_ts", strconv.FormatUint(toTS, 10)).
		SetQueryParam("limit", strconv.FormatUint(req.Limit, 10))

	err := c.withRetries(newHTTPReq, resty.MethodGet, "/2/auths/aggregated/runtime/", func(resp *resty.Response) error {
		if err := json.Unmarshal(resp.Body(), &res); err != nil {
			return xerrors.Errorf("Failed to parse json: [%d] '%s'. Error: %s", resp.StatusCode(), resp.String(), err)
		}

		if err := c.checkStatusOk(res.Status); err != nil {
			return xerrors.Errorf("Request failed: [%d] %s. Error: %s", resp.StatusCode(), resp.String(), err)
		}

		return nil
	})
	if err != nil {
		c.unistat.errs.Inc()
		return nil, &errs.TemporaryError{
			Message: fmt.Sprintf("Failed to fetch runtime aggregated auths from Old API: %v", err),
		}
	}

	var rows int
	for _, entry := range res.History {
		rows += len(entry.Auths)
	}
	c.unistat.authsRows.Add(float64(rows))

	return res, nil
}
