package hbaseapi

import (
	"context"
	"encoding/json"
	"fmt"
	"strconv"
	"strings"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/reqs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/resps"
)

func (c *Client) GetEvents(ctx context.Context, req *reqs.EventsRequest) (*resps.EventsResult, error) {
	orderBy, err := getOrderByParam(req.OrderBy)
	if err != nil {
		return nil, xerrors.Errorf("Incorrect request: %s", err.Error())
	}

	newHTTPReq := c.newRequestWithConsumer(ctx).
		SetQueryParam("uid", strconv.FormatUint(req.UID, 10)).
		SetQueryParam("from_ts", strconv.FormatUint(req.FromTS, 10)).
		SetQueryParam("to_ts", strconv.FormatUint(req.ToTS, 10)).
		SetQueryParam("order_by", orderBy)

	if req.Limit != nil {
		newHTTPReq.SetQueryParam("limit", strconv.FormatUint(*req.Limit, 10))
	}
	if len(req.Name) != 0 {
		newHTTPReq.SetQueryParam("name", strings.Join(req.Name, ","))
	}

	res := &resps.EventsResult{}

	err = c.withRetries(newHTTPReq, resty.MethodGet, "/2/events/", func(resp *resty.Response) error {
		if err := json.Unmarshal(resp.Body(), &res); err != nil {
			return xerrors.Errorf("Failed to parse json: [%d] '%s'. Error: %s", resp.StatusCode(), resp.String(), err)
		}

		if err := c.checkStatusOk(res.Status); err != nil {
			return xerrors.Errorf("Request failed: [%d] %s. Error: %s", resp.StatusCode(), resp.String(), err)
		}

		return nil
	})
	if err != nil {
		c.unistat.errs.Inc()
		return nil, &errs.TemporaryError{
			Message: fmt.Sprintf("Failed to fetch events from Old API: %v", err),
		}
	}

	c.unistat.eventsRows.Add(float64(len(res.Events)))

	return res, nil
}

func (c *Client) GetEventsRestore(ctx context.Context, req *reqs.EventsRestoreRequest) (*resps.EventsRestoreResult, error) {
	orderBy, err := getOrderByParam(req.OrderBy)
	if err != nil {
		return nil, xerrors.Errorf("Incorrect request: %s", err.Error())
	}

	newHTTPReq := c.newRequestWithConsumer(ctx).
		SetQueryParam("uid", strconv.FormatUint(req.UID, 10)).
		SetQueryParam("from_ts", strconv.FormatUint(req.FromTS, 10)).
		SetQueryParam("to_ts", strconv.FormatUint(req.ToTS, 10)).
		SetQueryParam("order_by", orderBy).
		SetQueryParam("limit", strconv.FormatUint(req.Limit, 10))

	res := &resps.EventsRestoreResult{}

	err = c.withRetries(newHTTPReq, resty.MethodGet, "/2/events/restore/", func(resp *resty.Response) error {
		if err := json.Unmarshal(resp.Body(), &res); err != nil {
			return xerrors.Errorf("Failed to parse json: [%d] '%s'. Error: %s", resp.StatusCode(), resp.String(), err)
		}

		if err := c.checkStatusOk(res.Status); err != nil {
			return xerrors.Errorf("Request failed: [%d] %s. Error: %s", resp.StatusCode(), resp.String(), err)
		}

		return nil
	})
	if err != nil {
		c.unistat.errs.Inc()
		return nil, &errs.TemporaryError{
			Message: fmt.Sprintf("Failed to fetch restore events from Old API: %v", err),
		}
	}

	c.unistat.eventsRows.Add(float64(len(res.Events)))

	return res, nil
}

func (c *Client) GetEventsByIPRegistrations(ctx context.Context, req *reqs.EventsByIPRegistrationsRequest) (*resps.EventsByIPRegistrationsResult, error) {
	orderBy, err := getOrderByParam(req.OrderBy)
	if err != nil {
		return nil, xerrors.Errorf("Incorrect request: %s", err.Error())
	}

	newHTTPReq := c.newRequestWithConsumer(ctx).
		SetQueryParam("subnet", req.SubNet).
		SetQueryParam("from_ts", strconv.FormatUint(req.FromTS, 10)).
		SetQueryParam("to_ts", strconv.FormatUint(req.ToTS, 10)).
		SetQueryParam("order_by", orderBy)

	if req.Limit != nil {
		newHTTPReq.SetQueryParam("limit", strconv.FormatUint(*req.Limit, 10))
	}

	res := &resps.EventsByIPRegistrationsResult{}

	err = c.withRetries(newHTTPReq, resty.MethodGet, "/2/events/by_ip/registrations/", func(resp *resty.Response) error {
		if err := json.Unmarshal(resp.Body(), &res); err != nil {
			return xerrors.Errorf("Failed to parse json: [%d] '%s'. Error: %s", resp.StatusCode(), resp.String(), err)
		}

		if err := c.checkStatusOk(res.Status); err != nil {
			return xerrors.Errorf("Request failed: [%d] %s. Error: %s", resp.StatusCode(), resp.String(), err)
		}

		return nil
	})
	if err != nil {
		c.unistat.errs.Inc()
		return nil, &errs.TemporaryError{
			Message: fmt.Sprintf("Failed to fetch events by ip from Old API: %v", err),
		}
	}

	c.unistat.eventsRows.Add(float64(len(res.Events)))

	return res, nil
}

func (c *Client) GetEventsPasswords(ctx context.Context, req *reqs.EventsPasswordsRequest) (*resps.EventsPasswordsResult, error) {
	formData := map[string]string{
		"password": req.Password,
		"uid":      strconv.FormatUint(req.UID, 10),
		"from_ts":  strconv.FormatUint(req.FromTS, 10),
		"to_ts":    strconv.FormatUint(req.ToTS, 10),
	}
	if req.Limit != nil {
		formData["limit"] = strconv.FormatUint(*req.Limit, 10)
	}

	newHTTPReq := c.newRequestWithConsumer(ctx).
		SetFormData(formData)

	res := &resps.EventsPasswordsResult{}

	err := c.withRetries(newHTTPReq, resty.MethodPost, "/2/events/passwords/", func(resp *resty.Response) error {
		if err := json.Unmarshal(resp.Body(), &res); err != nil {
			return xerrors.Errorf("Failed to parse json: [%d] '%s'. Error: %s", resp.StatusCode(), resp.String(), err)
		}

		if err := c.checkStatusOk(res.Status); err != nil {
			return xerrors.Errorf("Request failed: [%d] %s. Error: %s", resp.StatusCode(), resp.String(), err)
		}

		return nil
	})
	if err != nil {
		c.unistat.errs.Inc()
		return nil, &errs.TemporaryError{
			Message: fmt.Sprintf("Failed to fetch passwords events from Old API: %v", err),
		}
	}

	c.unistat.eventsRows.Add(float64(len(res.Ranges)))

	return res, nil
}
