package hbaseapi

import (
	"context"
	"fmt"
	"net/http"
	"time"

	"github.com/cenkalti/backoff/v4"
	"github.com/go-resty/resty/v2"
	"golang.org/x/xerrors"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/reqs"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/middlewares"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/passport/shared/golibs/unistat"
)

const (
	xRealIPHeader         = "X-Real-IP"
	backoffInitInternal   = 500 * time.Millisecond
	backoffMaxInterval    = 2000 * time.Millisecond
	backoffMaxElapsedTime = 10 * time.Second
)

type Config struct {
	Consumer string `json:"consumer_name"`
	Host     string `json:"host"`
	Port     uint16 `json:"port"`
	Retries  uint64 `json:"retries"`
	Timeout  uint32 `json:"timeout_ms"`
}

type Client struct {
	http     *resty.Client
	retries  uint64
	consumer string
	unistat  stats
}

type stats struct {
	responseTimings     *unistat.TimeStat
	requests            *unistat.SignalDiff
	errs                *unistat.SignalDiff
	mailUserHistoryRows *unistat.SignalDiff
	eventsRows          *unistat.SignalDiff
	yasmsRows           *unistat.SignalDiff
	authsRows           *unistat.SignalDiff
}

func NewClient(cfg Config) (*Client, error) {
	if cfg.Consumer == "" {
		return nil, xerrors.Errorf("consumer cannot be empty")
	}

	httpc := resty.New().
		SetBaseURL(fmt.Sprintf("http://%s:%d/", cfg.Host, cfg.Port)).
		SetTimeout(time.Duration(cfg.Timeout) * time.Millisecond).
		SetRedirectPolicy(resty.NoRedirectPolicy())

	responseTimings, err := unistat.DefaultChunk.CreateTimeStats(
		"hbase.response_time",
		unistat.CreateTimeBoundsFromMaxValue(10*time.Second),
	)
	if err != nil {
		return nil, xerrors.Errorf("Failed to create time stats: %w", err)
	}

	res := &Client{
		http:     httpc,
		retries:  cfg.Retries,
		consumer: cfg.Consumer,
		unistat: stats{
			responseTimings:     responseTimings,
			requests:            unistat.DefaultChunk.CreateSignalDiff("hbase.requests"),
			errs:                unistat.DefaultChunk.CreateSignalDiff("hbase.errors"),
			mailUserHistoryRows: unistat.DefaultChunk.CreateSignalDiff("hbase.rows.mail_user_history"),
			eventsRows:          unistat.DefaultChunk.CreateSignalDiff("hbase.rows.events"),
			yasmsRows:           unistat.DefaultChunk.CreateSignalDiff("hbase.rows.yasms"),
			authsRows:           unistat.DefaultChunk.CreateSignalDiff("hbase.rows.auths"),
		},
	}

	res.http.
		OnBeforeRequest(func(c *resty.Client, r *resty.Request) error {
			res.unistat.requests.Inc()
			r.SetContext(middlewares.WithStartInstant(r.Context(), time.Now()))
			return nil
		}).
		OnAfterResponse(func(c *resty.Client, r *resty.Response) error {
			res.unistat.responseTimings.Insert(
				time.Since(middlewares.ContextStartInstant(r.Request.Context())))
			return nil
		})

	return res, nil
}

func (c *Client) Ping(ctx context.Context) error {
	newHTTPReq := c.http.R().SetContext(ctx)

	resp, err := newHTTPReq.Get("/ping")
	if err != nil {
		return xerrors.Errorf("failed to perform /ping: %w", err)
	}

	if resp.StatusCode() != http.StatusOK {
		return xerrors.Errorf("not available: %d: %s", resp.StatusCode(), resp.String())
	}

	return nil
}

func (c *Client) newRequestWithConsumer(ctx context.Context) *resty.Request {
	return c.http.R().
		SetContext(ctx).
		SetHeader(xRealIPHeader, "::1").
		SetQueryParam("consumer", c.consumer)
}

func (c *Client) checkStatusOk(status errs.ScalaStatus) error {
	if status != errs.ScalaStatusOk {
		return xerrors.Errorf("Old API error: got status '%s'", status)
	}

	return nil
}

func (c *Client) withRetries(req *resty.Request, method, url string, handle func(*resty.Response) error) error {
	op := func() error {
		resp, err := req.Execute(method, url)
		if err != nil {
			return xerrors.Errorf("Failed to perform request to Old API: %s", err)
		}

		if resp.StatusCode() != http.StatusOK {
			err := xerrors.Errorf("Bad response code: [%d] %s", resp.StatusCode(), resp.String())
			if resp.StatusCode() >= 400 && resp.StatusCode() < 500 {
				return backoff.Permanent(err)
			}
			return err
		}

		return handle(resp)
	}

	notify := func(err error, delay time.Duration) {
		ctxlog.Warnf(req.Context(), logger.Log(),
			"Old API: new retry: sleep for %s: %v", delay, err)
	}

	backoffPolicy := c.newBackoff(req.Context())
	return backoff.RetryNotify(op, backoffPolicy, notify)
}

func (c *Client) newBackoff(ctx context.Context) backoff.BackOff {
	if c.retries <= 1 {
		return &backoff.StopBackOff{}
	}
	exp := backoff.NewExponentialBackOff()
	exp.InitialInterval = backoffInitInternal
	exp.MaxInterval = backoffMaxInterval
	exp.MaxElapsedTime = backoffMaxElapsedTime
	// Reset() must have been called after re-setting fields of ExponentialBackoff.
	// https://github.com/cenkalti/backoff/issues/69#issuecomment-448923118
	exp.Reset()

	// we doing "c.retries - 1", because of "backoff.WithMaxRetries(backoff, 2)" mean repeat operation 3 times.
	return backoff.WithContext(backoff.WithMaxRetries(exp, c.retries-1), ctx)
}

func getOrderByParam(orderBy reqs.OrderByType) (string, error) {
	switch orderBy {
	case reqs.OrderByDesc:
		return "desc", nil
	case reqs.OrderByAsc:
		return "asc", nil
	default:
		return "", xerrors.Errorf("unknown order by type: %v", orderBy)
	}
}

func chooseToTS(toTS, upperTSLimit uint64) uint64 {
	// Старое API возвращает список авторизаций до верхней границы ToTS *включительно*,
	// поэтому, что бы в ответ не попали значения с TS = upperTSLimit, запрос нужно обрезать по upperTSLimit - 1
	if toTS < upperTSLimit {
		return toTS
	}
	if upperTSLimit > 1 {
		return upperTSLimit - 1
	}
	return 1
}
