package hbaseapi

import (
	"context"
	"encoding/json"
	"fmt"
	"strconv"
	"strings"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/reqs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/resps"
)

func (c *Client) GetMailUserHistory(ctx context.Context, req *reqs.MailUserHistoryRequest) (*resps.MailUserHistoryResult, error) {
	newHTTPReq := c.newRequestWithConsumer(ctx).
		SetQueryParam("uid", strconv.FormatUint(req.UID, 10)).
		SetQueryParam("from_ts", strconv.FormatUint(req.FromTS, 10)).
		SetQueryParam("to_ts", strconv.FormatUint(req.ToTS, 10)).
		SetQueryParam("limit", strconv.FormatUint(req.Limit, 10))

	if req.Corp {
		newHTTPReq.SetQueryParam("corp", "true")
	}
	if len(req.Operation) != 0 {
		newHTTPReq.SetQueryParam("operation", strings.Join(req.Operation, ","))
	}
	if len(req.Module) != 0 {
		newHTTPReq.SetQueryParam("module", strings.Join(req.Module, ","))
	}

	res := &resps.MailUserHistoryResult{}

	err := c.withRetries(newHTTPReq, resty.MethodGet, "/mail/2/user_history/", func(resp *resty.Response) error {
		if err := json.Unmarshal(resp.Body(), &res); err != nil {
			return xerrors.Errorf("Failed to parse json: [%d] '%s'. Error: %s", resp.StatusCode(), resp.String(), err)
		}

		if err := c.checkStatusOk(res.Status); err != nil {
			return xerrors.Errorf("Request failed: [%d] %s. Error: %s", resp.StatusCode(), resp.String(), err)
		}

		return nil
	})
	if err != nil {
		c.unistat.errs.Inc()
		return nil, &errs.TemporaryError{
			Message: fmt.Sprintf("Failed to fetch mail user history from Old API: %v", err),
		}
	}

	c.unistat.mailUserHistoryRows.Add(float64(len(res.Items)))

	return res, nil
}
