package hbaseapi

import (
	"context"
	"encoding/json"
	"fmt"
	"strconv"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/model"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/reqs"
)

type SmsByGlobalIDResult struct {
	Status      errs.ScalaStatus `json:"status"`
	GlobalSmsID string           `json:"global_smsid"`
	Items       []*model.SmsItem `json:"items"`
}

type SmsByPhoneResult struct {
	Status errs.ScalaStatus         `json:"status"`
	Phone  string                   `json:"number"`
	Items  model.SmsItemsByGlobalID `json:"items"`
}

func (c *Client) GetSmsByGlobalID(ctx context.Context, req *reqs.SmsByGlobalIDRequest) (*SmsByGlobalIDResult, error) {
	newHTTPReq := c.newRequestWithConsumer(ctx).
		SetQueryParam("global_smsid", req.GlobalSmsID)

	var res SmsByGlobalIDResult
	err := c.withRetries(newHTTPReq, resty.MethodGet, "/yasms/2/sms/by_globalid/", func(resp *resty.Response) error {
		if err := json.Unmarshal(resp.Body(), &res); err != nil {
			return xerrors.Errorf("Failed to parse json: [%d] '%s'. Error: %s", resp.StatusCode(), resp.String(), err)
		}

		if err := c.checkStatusOk(res.Status); err != nil {
			return xerrors.Errorf("Request failed: [%d] %s. Error: %s", resp.StatusCode(), resp.String(), err)
		}

		return nil
	})
	if err != nil {
		c.unistat.errs.Inc()
		return nil, &errs.TemporaryError{
			Message: fmt.Sprintf("Failed to fetch sms by global ID from Old API: %v", err),
		}
	}

	c.unistat.yasmsRows.Add(float64(len(res.Items)))

	return &res, nil
}

func (c *Client) GetSmsByPhone(ctx context.Context, req *reqs.SmsByFieldsRequest) (*SmsByPhoneResult, error) {
	res := SmsByPhoneResult{
		Status: errs.ScalaStatusOk,
		Phone:  "",
		Items:  make(model.SmsItemsByGlobalID),
	}

	if req.Phone == nil {
		return &res, nil
	}

	newHTTPReq := c.newRequestWithConsumer(ctx).
		SetQueryParam("phone", *req.Phone)

	if req.FromTS != nil {
		newHTTPReq.SetQueryParam("from_ts", strconv.FormatUint(*req.FromTS, 10))
	}
	if req.ToTS != nil {
		newHTTPReq.SetQueryParam("to_ts", strconv.FormatUint(*req.ToTS, 10))
	}

	err := c.withRetries(newHTTPReq, resty.MethodGet, "/yasms/2/sms/by_phone/", func(resp *resty.Response) error {
		if err := json.Unmarshal(resp.Body(), &res); err != nil {
			return xerrors.Errorf("Failed to parse json: [%d] '%s'. Error: %s", resp.StatusCode(), resp.String(), err)
		}

		if err := c.checkStatusOk(res.Status); err != nil {
			return xerrors.Errorf("Request failed: [%d] %s. Error: %s", resp.StatusCode(), resp.String(), err)
		}

		return nil
	})
	if err != nil {
		c.unistat.errs.Inc()
		return nil, &errs.TemporaryError{
			Message: fmt.Sprintf("Failed to fetch sms by phone from Old API: %v", err),
		}
	}

	c.unistat.yasmsRows.Add(float64(len(res.Items)))

	return &res, nil
}
