package model

import (
	"encoding/json"
	"strconv"

	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/ytc"
	"a.yandex-team.ru/passport/infra/libs/go/keys"
)

type SmsItemsByGlobalID = map[string][]*SmsItem

type SmsItemBase struct {
	GlobalSmsID string  `json:"global_smsid"`
	Action      string  `json:"action"`
	Timestamp   float64 `json:"timestamp"`
}

type SmsItemCommon struct {
	*SmsItemBase
	Number string  `json:"number,omitempty"`
	UID    *uint64 `json:"uid,string,omitempty"`
	Text   string  `json:"text,omitempty"`
}

type SmsItem struct {
	Common *SmsItemCommon
	Extra  map[string]interface{}
}

func (item *SmsItem) UnmarshalJSON(data []byte) error {
	if err := json.Unmarshal(data, &item.Common); err != nil {
		return err
	}
	return json.Unmarshal(data, &item.Extra)
}

func (item *SmsItem) MarshalJSON() ([]byte, error) {
	common, err := json.Marshal(item.Common)
	if err != nil {
		return nil, err
	}

	if item.Extra == nil {
		return common, nil
	}
	if err := json.Unmarshal(common, &item.Extra); err != nil {
		return nil, err
	}

	return json.Marshal(item.Extra)
}

func smsItemBaseFromYt(row *ytc.YasmsSmsHistoryRowBase) *SmsItemBase {
	return &SmsItemBase{
		GlobalSmsID: row.GlobalSmsID,
		Action:      row.Action,
		Timestamp:   float64(row.UnixtimeMicro) / 1000000,
	}
}

func smsItemCommonFromYt(row *ytc.YasmsSmsHistoryRow, keys *keys.KeyMap) (*SmsItemCommon, error) {
	res := &SmsItemCommon{
		SmsItemBase: smsItemBaseFromYt(&row.YasmsSmsHistoryRowBase),
		UID:         row.UID,
	}

	if row.Phone != nil {
		res.Number = "+" + strconv.FormatUint(*row.Phone, 10)
	}

	if keys != nil && row.Data.EncryptedText != nil {
		text, err := row.Data.EncryptedText.Decrypt(keys)
		if err != nil {
			return nil, err
		}

		res.Text = string(text)
	}

	return res, nil
}

func smsItemFromYt(row *ytc.YasmsSmsHistoryRow, keys *keys.KeyMap) (*SmsItem, error) {
	common, err := smsItemCommonFromYt(row, keys)
	if err != nil {
		return nil, err
	}

	return &SmsItem{
		Common: common,
		Extra:  row.Data.Extra,
	}, nil
}

type SmsItemsCollector struct {
	keys  *keys.KeyMap
	items []*SmsItem
}

func NewSmsItemsCollector(keys *keys.KeyMap) *SmsItemsCollector {
	return &SmsItemsCollector{
		keys:  keys,
		items: make([]*SmsItem, 0, 8),
	}
}

func (collector *SmsItemsCollector) CollectFromYt(row *ytc.YasmsSmsHistoryRow) error {
	item, err := smsItemFromYt(row, collector.keys)
	if err != nil {
		return err
	}

	collector.items = append(collector.items, item)
	return nil
}

func (collector *SmsItemsCollector) Finish() []*SmsItem {
	return collector.items
}

type SmsItemsByGlobalIDCollector struct {
	keys  *keys.KeyMap
	items SmsItemsByGlobalID
}

func NewSmsItemsByGlobalIDCollector(keys *keys.KeyMap) *SmsItemsByGlobalIDCollector {
	return &SmsItemsByGlobalIDCollector{
		keys:  keys,
		items: make(SmsItemsByGlobalID),
	}
}

func (collector *SmsItemsByGlobalIDCollector) CollectFromYt(row *ytc.YasmsSmsHistoryRow) error {
	item, err := smsItemFromYt(row, collector.keys)
	if err != nil {
		return err
	}

	items := collector.items[item.Common.GlobalSmsID]
	if items == nil {
		items = make([]*SmsItem, 0, 8)
	}

	collector.items[item.Common.GlobalSmsID] = append(items, item)
	return nil
}

func (collector *SmsItemsByGlobalIDCollector) Finish() SmsItemsByGlobalID {
	return collector.items
}
