package resps

import (
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
)

type AuthsResponse struct {
	Status errs.ScalaStatus `json:"status"`
	UID    uint64           `json:"uid"`
	Auths  []*PlainAuth     `json:"auths"`
}

type AuthsRuntimeAggregatedResponse struct {
	Status  errs.ScalaStatus              `json:"status"`
	UID     uint64                        `json:"uid"`
	History []AuthsRuntimeAggregatedEntry `json:"history"`
}

type AuthsAggregatedResponse struct {
	Status errs.ScalaStatus         `json:"status"`
	UID    uint64                   `json:"uid"`
	Next   *string                  `json:"next"`
	Auths  []map[string]interface{} `json:"auths"`
}

type AuthsRuntimeAggregatedEntry struct {
	Timestamp uint64                 `json:"timestamp"`
	Auths     []AuthsAggregatedEntry `json:"auths"`
}

type AuthsAggregatedEntry struct {
	Count uint64    `json:"count"`
	Auth  *AuthItem `json:"auth"`
}

type PlainAuth struct {
	Timestamp float64 `json:"timestamp"`

	Type       string `json:"type,omitempty"`
	Status     string `json:"status,omitempty"`
	ClientName string `json:"client_name,omitempty"`

	HostID    string `json:"host_id,omitempty"`
	Login     string `json:"login,omitempty"`
	SID       string `json:"sid,omitempty"`
	YandexUID string `json:"yandexuid,omitempty"`
	Comment   string `json:"comment,omitempty"`

	UserIP     string `json:"user_ip,omitempty"`
	IPGeoID    string `json:"ip.geoid,omitempty"`
	IPAsList   string `json:"ip.as_list,omitempty"`
	IPIsYandex string `json:"ip.is_yandex,omitempty"`

	BrowserName    string `json:"browser.name,omitempty"`
	BrowserVersion string `json:"browser.version,omitempty"`

	OsName    string `json:"os.name,omitempty"`
	OsFamily  string `json:"os.family,omitempty"`
	OsVersion string `json:"os.version,omitempty"`
}

type IPItem struct {
	GeoID uint64 `json:"geoid,omitempty"`
	AS    uint64 `json:"AS,omitempty"`
	IP    string `json:"ip,omitempty"`
}

type BrowserItem struct {
	Name      string `json:"name,omitempty"`
	Version   string `json:"version,omitempty"`
	YandexUID string `json:"yandexuid,omitempty"`
}

type OsItem struct {
	Name    string `json:"name,omitempty"`
	Version string `json:"version,omitempty"`
}

type AuthItem struct {
	AuthType string `json:"authtype"`
	Status   string `json:"status,omitempty"`

	IP      IPItem      `json:"ip"`
	Browser BrowserItem `json:"browser"`
	Os      OsItem      `json:"os"`
}

type AuthsAggregated map[AuthItem]uint64

func (auths AuthsAggregated) ToList() []AuthsAggregatedEntry {
	asList := make([]AuthsAggregatedEntry, 0, len(auths))
	for auth, count := range auths {
		localAuth := auth
		asList = append(asList, AuthsAggregatedEntry{
			Count: count,
			Auth:  &localAuth,
		})
	}

	return asList
}

type AuthsRuntimeAggregatedBuilder struct {
	windowWidth uint64

	currentWindow      AuthsAggregated
	currentWindowStart uint64

	result []AuthsRuntimeAggregatedEntry
}

func NewAuthsRuntimeAggregatedBuilder(windowWidth uint64) *AuthsRuntimeAggregatedBuilder {
	return &AuthsRuntimeAggregatedBuilder{
		windowWidth: windowWidth,
		result:      make([]AuthsRuntimeAggregatedEntry, 0),
	}
}

func (builder *AuthsRuntimeAggregatedBuilder) flush() {
	if len(builder.currentWindow) > 0 {
		builder.result = append(builder.result, AuthsRuntimeAggregatedEntry{
			Timestamp: builder.currentWindowStart,
			Auths:     builder.currentWindow.ToList(),
		})
	}

	builder.currentWindow = nil
}

func (builder *AuthsRuntimeAggregatedBuilder) Collect(ts uint64, item *AuthItem) {
	windowStart := ts - ts%builder.windowWidth
	if windowStart != builder.currentWindowStart || builder.currentWindow == nil {
		builder.flush()
		builder.currentWindow = make(AuthsAggregated)
		builder.currentWindowStart = windowStart
	}

	builder.currentWindow[*item] += 1
}

func (builder *AuthsRuntimeAggregatedBuilder) Finish() []AuthsRuntimeAggregatedEntry {
	builder.flush()
	return builder.result
}
