package resps

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"reflect"

	"github.com/andybalholm/brotli"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/yt/go/yson"
)

type MailUserHistoryResult struct {
	Status errs.ScalaStatus       `json:"status"`
	UID    uint64                 `json:"uid"`
	Items  []*MailUserHistoryItem `json:"items"`
}

type MailUserHistoryItemCommon struct {
	Date      uint64 `json:"date"`
	Operation string `json:"operation"`
	Module    string `json:"module"`
}

type MailUserHistoryItem struct {
	Common MailUserHistoryItemCommon

	all map[string]interface{}
}

func (u *MailUserHistoryItem) MarshalJSON() ([]byte, error) {
	return json.Marshal(u.all)
}

func (u *MailUserHistoryItem) UnmarshalJSON(data []byte) error {
	// keep everything from old historydbapi
	if err := json.Unmarshal(data, &u.all); err != nil {
		return err
	}

	// store significant fields
	if err := json.Unmarshal(data, &u.Common); err != nil {
		return err
	}

	return nil
}

func (u *MailUserHistoryItem) UnmarshalYSON(data []byte) error {
	if err := yson.Unmarshal(data, &u.all); err != nil {
		return err
	}

	move := func(entryKey, env, field string) {
		e, ok := u.all[entryKey]
		if !ok {
			return
		}

		en, ok := u.all[env]
		if ok {
			m, ok := en.(map[string]interface{})
			if !ok {
				// something strange stored there
				return
			}
			m[field] = e
		} else {
			u.all[env] = map[string]interface{}{
				field: e,
			}
		}

		delete(u.all, entryKey)
	}

	move("operationSystem.name", "operationSystem", "name")
	move("operationSystem.version", "operationSystem", "version")
	move("browser.name", "browser", "name")
	move("browser.version", "browser", "version")

	return nil
}

func (u *MailUserHistoryItem) After(other *MailUserHistoryItem) bool {
	return u.Common.Date > other.Common.Date
}

func (u *MailUserHistoryItem) Equals(other *MailUserHistoryItem) bool {
	return u.Common.Date == other.Common.Date &&
		u.Common.Operation == other.Common.Operation &&
		u.Common.Module == other.Common.Module
}

func (u *MailUserHistoryItem) SetDate(d uint64) {
	u.lazyInit()
	u.Common.Date = d

	u.all["date"] = d
	u.all["unixtime"] = fmt.Sprintf("%d", d/1000)
}

func (u *MailUserHistoryItem) SetOperation(value string) {
	u.lazyInit()
	u.Common.Operation = value
	u.all["operation"] = value
}

func (u *MailUserHistoryItem) SetModule(value string) {
	u.lazyInit()
	u.Common.Module = value
	u.all["module"] = value
}

// for tests mostly
func (u *MailUserHistoryItem) lazyInit() {
	if u.all == nil {
		u.all = make(map[string]interface{})
	}
}

const (
	compressedKey = "_compressed"
)

func (u *MailUserHistoryItem) Decompress() error {
	compressed, ok := u.all[compressedKey]
	if !ok {
		return nil
	}
	defer func() { delete(u.all, compressedKey) }()

	compressedMap, ok := compressed.(map[string]interface{})
	if !ok {
		return xerrors.Errorf(
			"'%s' is not map[string]interface{}: %s",
			compressedKey,
			reflect.TypeOf(compressed),
		)
	}

	var err error
	for key, field := range compressedMap {
		val, er := decompressMailUserHistoryItemField(key, field)
		if er != nil {
			err = xerrors.Errorf("failed to decompress '%s': %w; %s", key, er, err)
			continue
		}

		u.all[key] = val
	}

	return err
}

func decompressMailUserHistoryItemField(key string, field interface{}) (string, error) {
	fieldMap, ok := field.(map[string]interface{})
	if !ok {
		// something strange stored there
		return "", xerrors.Errorf("'%s' in '%s' is not map[string]interface{}: %s",
			key,
			compressedKey,
			reflect.TypeOf(field),
		)
	}

	size, ok := fieldMap["size"]
	if !ok {
		return "", xerrors.Errorf("missing 'size'")
	}
	sizeNum, ok := size.(uint64)
	if !ok {
		return "", xerrors.Errorf("'size' is not uint64 but %s", reflect.TypeOf(size))
	}

	value, ok := fieldMap["value"]
	if !ok {
		return "", xerrors.Errorf("missing 'value'")
	}
	valueBlob, ok := value.(string)
	if !ok {
		return "", xerrors.Errorf("'value' is not string but %s", reflect.TypeOf(value))
	}

	return decompressBrotli(sizeNum, valueBlob)
}

func decompressBrotli(expectedLen uint64, encoded string) (string, error) {
	decoded := make([]byte, expectedLen)
	r := brotli.NewReader(bytes.NewBufferString(encoded))
	_, err := io.ReadFull(r, decoded)
	if err != nil {
		return "", xerrors.Errorf("failed to decompress blob: %w", err)
	}

	return string(decoded), nil
}
