package ytc

import (
	"context"
	"fmt"
	"strconv"
	"strings"
	"time"

	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/reqs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/resps"
	"a.yandex-team.ru/passport/infra/libs/go/ytsimple"
	"a.yandex-team.ru/yt/go/yt"
)

type AuthRow struct {
	ReversedTimestamp uint64            `yson:"reversed_timestamp"`
	Type              string            `yson:"type"`
	Status            string            `yson:"status"`
	ClientName        string            `yson:"client_name"`
	Data              map[string]string `yson:"data"`
}

const (
	authsDir         = "auths"
	failedAuthsTable = "failed_auths/failed_auths"
)

func (c *Client) GetAuths(ctx context.Context, req *reqs.AuthsRequest, failed bool, lowerTSLimit uint64) ([]*resps.PlainAuth, error) {
	fromTS := max(req.FromTS, lowerTSLimit)
	if fromTS >= req.ToTS {
		return make([]*resps.PlainAuth, 0), nil
	}

	if failed {
		return c.getFailedAuthsImpl(ctx, req, fromTS)
	} else {
		return c.getAuthsImpl(ctx, req, fromTS)
	}
}

func (c *Client) getAuthsImpl(ctx context.Context, req *reqs.AuthsRequest, fromTS uint64) ([]*resps.PlainAuth, error) {
	auths := make([]*resps.PlainAuth, 0, min(10001, req.Limit))
	scanner := plainAuthsScanner(&auths)

	tables := c.tablesTTLConfig.getMonthlyTables(
		fromTS,
		req.ToTS,
		c.tablesTTLConfig.Auths,
		uint64(time.Now().Add(60*time.Second).Unix()),
	)
	condition := buildAuthsCondition(req, fromTS)
	for uint64(len(auths)) < req.Limit && tables.Next() {
		query := buildAuthsQuery(buildNodePath(c.dir, authsDir, tables.TableName()), condition, req.Limit-uint64(len(auths)))
		if err := c.selectAll(ctx, query, scanner); err != nil {
			return nil, &errs.TemporaryError{
				Message: fmt.Sprintf("Failed to fetch auths from YT: %v", err),
			}
		}
	}
	c.unistat.authsRows.Add(float64(len(auths)))

	if req.OrderBy == reqs.OrderByAsc {
		reverse(auths)
	}

	return auths, nil
}

func (c *Client) getFailedAuthsImpl(ctx context.Context, req *reqs.AuthsRequest, fromTS uint64) ([]*resps.PlainAuth, error) {
	auths := make([]*resps.PlainAuth, 0, min(10001, req.Limit))

	query := buildAuthsQuery(buildNodePath(c.dir, failedAuthsTable), buildAuthsCondition(req, fromTS), req.Limit)
	if err := c.selectAll(ctx, query, plainAuthsScanner(&auths)); err != nil {
		return nil, &errs.TemporaryError{
			Message: fmt.Sprintf("Failed to fetch failed auths from YT: %v", err),
		}
	}
	c.unistat.failedAuthsRows.Add(float64(len(auths)))

	if req.OrderBy == reqs.OrderByAsc {
		reverse(auths)
	}

	return auths, nil
}

func (c *Client) GetAuthsRuntimeAggregated(
	ctx context.Context,
	req *reqs.AuthsRuntimeAggregatedRequest,
	windowWidth uint64,
	lowerTSLimit uint64,
) ([]resps.AuthsRuntimeAggregatedEntry, error) {
	fromTS := max(req.FromTS, lowerTSLimit)
	if fromTS >= req.ToTS {
		return make([]resps.AuthsRuntimeAggregatedEntry, 0), nil
	}

	var rowsTotal uint64
	builder := resps.NewAuthsRuntimeAggregatedBuilder(windowWidth)
	scanner := authsRuntimeAggregatedScanner(builder, &rowsTotal)

	tables := c.tablesTTLConfig.getMonthlyTables(
		fromTS,
		req.ToTS,
		c.tablesTTLConfig.Auths,
		uint64(time.Now().Add(60*time.Second).Unix()),
	)
	condition := buildAuthsRuntimeAggregatedCondition(req, fromTS)
	for rowsTotal < req.Limit && tables.Next() {
		query := buildAuthsQuery(buildNodePath(c.dir, authsDir, tables.TableName()), condition, req.Limit-rowsTotal)
		if err := c.selectAll(ctx, query, scanner); err != nil {
			return nil, &errs.TemporaryError{
				Message: fmt.Sprintf("Failed to fetch auths from YT: %v", err),
			}
		}
	}
	c.unistat.authsRows.Add(float64(rowsTotal))

	return builder.Finish(), nil
}

func plainAuthsScanner(out *[]*resps.PlainAuth) func(reader yt.TableReader) error {
	return func(reader yt.TableReader) error {
		row := AuthRow{}
		if err := ytsimple.ScanRow(reader, &row); err != nil {
			return err
		}

		*out = append(*out, &resps.PlainAuth{
			Timestamp: convertFromAuthsReversedTimestamp[float64](row.ReversedTimestamp),

			Type:       row.Type,
			Status:     row.Status,
			ClientName: row.ClientName,

			HostID:    row.Data["host_id"],
			Login:     row.Data["login"],
			SID:       row.Data["sid"],
			YandexUID: row.Data["yandexuid"],
			Comment:   row.Data["comment"],

			UserIP:     row.Data["user_ip"],
			IPGeoID:    row.Data["ip.geoid"],
			IPAsList:   row.Data["ip.as_list"],
			IPIsYandex: row.Data["ip.is_yandex"],

			BrowserName:    row.Data["browser.name"],
			BrowserVersion: row.Data["browser.version"],

			OsName:    row.Data["os.name"],
			OsFamily:  row.Data["os.family"],
			OsVersion: row.Data["os.version"],
		})
		return nil
	}
}

func authsRuntimeAggregatedScanner(builder *resps.AuthsRuntimeAggregatedBuilder, rowsTotal *uint64) func(reader yt.TableReader) error {
	return func(reader yt.TableReader) error {
		row := AuthRow{}
		if err := ytsimple.ScanRow(reader, &row); err != nil {
			return err
		}

		optionalUint := func(value string) uint64 {
			res, err := strconv.ParseUint(value, 10, 64)
			if err != nil {
				return 0
			}
			return res
		}

		builder.Collect(
			convertFromAuthsReversedTimestamp[uint64](row.ReversedTimestamp),
			&resps.AuthItem{
				AuthType: row.Type,
				Status:   row.Status,
				IP: resps.IPItem{
					GeoID: optionalUint(row.Data["ip.geoid"]),
					AS:    optionalUint(strings.TrimPrefix(row.Data["ip.as_list"], "AS")),
					IP:    row.Data["user_ip"],
				},
				Browser: resps.BrowserItem{
					Name:      row.Data["browser.name"],
					Version:   row.Data["browser.version"],
					YandexUID: row.Data["yandexuid"],
				},
				Os: resps.OsItem{
					Name:    row.Data["os.name"],
					Version: row.Data["os.version"],
				},
			},
		)

		*rowsTotal += 1
		return nil
	}
}

func buildAuthsCondition(req *reqs.AuthsRequest, fromTS uint64) string {
	condition := fmt.Sprintf(
		"uid = %d AND %d < reversed_timestamp AND reversed_timestamp <= %d",
		req.UID,
		convertToAuthsReversedTimestamp(req.ToTS),
		convertToAuthsReversedTimestamp(fromTS),
	)
	if len(req.Type) > 0 {
		condition += fmt.Sprintf(" AND type IN (%s)", serializeConditionIN(req.Type))
	}
	if len(req.Status) > 0 {
		condition += fmt.Sprintf(" AND status IN (%s)", serializeConditionIN(req.Status))
	}
	if len(req.ClientName) > 0 {
		condition += fmt.Sprintf(" AND client_name IN (%s)", serializeConditionIN(req.ClientName))
	}

	return condition
}

func buildAuthsRuntimeAggregatedCondition(req *reqs.AuthsRuntimeAggregatedRequest, fromTS uint64) string {
	return fmt.Sprintf(
		`uid = %d AND %d < reversed_timestamp AND reversed_timestamp <= %d AND (status IN ("ses_create","ses_update") OR (status = "successful" AND not type = "web"))`,
		req.UID,
		convertToAuthsReversedTimestamp(req.ToTS),
		convertToAuthsReversedTimestamp(fromTS),
	)
}

func buildAuthsQuery(table, condition string, limit uint64) string {
	return fmt.Sprintf(
		`
reversed_timestamp,type,status,client_name,data
FROM [%s]
WHERE %s
ORDER BY uid,reversed_timestamp
LIMIT %d`,
		table,
		condition,
		limit,
	)
}

func convertToAuthsReversedTimestamp(ts uint64) uint64 {
	return reverseUnixtime(ts * 1000000)
}

func convertFromAuthsReversedTimestamp[Num float64 | uint64](ts uint64) Num {
	return Num(reverseUnixtime(ts)) / 1000000
}
