package ytc

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/resps"
	"a.yandex-team.ru/passport/infra/libs/go/ytsimple"
	"a.yandex-team.ru/yt/go/yt"
)

type LastAuthRow struct {
	UID       uint64 `yson:"uid"`
	AuthType  string `yson:"authtype"`
	Timestamp uint64 `yson:"timestamp"`
}

const (
	lastAuthTable = "lastauth/lastauth"
)

func convertLastAuthTimestamp(ts uint64) float64 {
	return float64(ts) / 1000000
}

func (c *Client) GetLastAuth(ctx context.Context, uid uint64) (*resps.LastAuthResult, error) {
	var err error

	start := time.Now()
	defer func() {
		c.unistat.responseTimings.Insert(time.Since(start))
		if err != nil {
			c.unistat.errs.Inc()
		}
	}()

	result := &resps.LastAuthResult{
		Status: errs.ScalaStatusOk,
		UID:    uid,
	}

	c.unistat.requests.Inc()
	query := buildLastAuthQuery(uid, c.dir)

	err = ytsimple.SelectAll(ctx, c.yc, query, c.timeout, func(reader yt.TableReader) error {
		row := LastAuthRow{}
		if err := ytsimple.ScanRow(reader, &row); err != nil {
			return err
		}

		result.Value = resps.LastAuthValue{
			Timestamp: convertLastAuthTimestamp(row.Timestamp),
			Type:      row.AuthType,
		}
		return nil
	})
	if err != nil {
		return nil, &errs.TemporaryError{
			Message: fmt.Sprintf("Failed to fetch lastauth from YT: %v", err),
		}
	}

	c.unistat.lastAuthRows.Inc()

	return result, nil
}

func buildLastAuthQuery(uid uint64, dir string) string {
	return fmt.Sprintf(`
authtype, timestamp
FROM [%s]
WHERE uid = %d
ORDER BY timestamp DESC
LIMIT 1
`,
		buildNodePath(dir, lastAuthTable),
		uid,
	)
}

func (c *Client) GetLastAuthBulk(ctx context.Context, uids []uint64) (*resps.LastAuthBulkResult, error) {
	var err error

	start := time.Now()
	defer func() {
		c.unistat.responseTimings.Insert(time.Since(start))
		if err != nil {
			c.unistat.errs.Inc()
		}
	}()

	result := &resps.LastAuthBulkResult{
		Status: errs.ScalaStatusOk,
		Items:  make(resps.LastAuthValueMap, len(uids)),
	}
	if len(uids) == 0 {
		return result, nil
	}
	for _, uid := range uids {
		result.Items[uid] = resps.LastAuthValue{}
	}

	query := buildLastAuthBulkQuery(uids, c.dir)

	var rowsNumber uint64
	c.unistat.requests.Inc()
	err = ytsimple.SelectAll(ctx, c.yc, query, c.timeout, func(reader yt.TableReader) error {
		row := LastAuthRow{}
		if err := ytsimple.ScanRow(reader, &row); err != nil {
			return err
		}

		result.Items[row.UID] = resps.LastAuthValue{
			Timestamp: convertLastAuthTimestamp(row.Timestamp),
		}

		rowsNumber++

		return nil
	})
	if err != nil {
		return nil, &errs.TemporaryError{
			Message: fmt.Sprintf("Failed to fetch lastauth from YT: %v", err),
		}
	}

	c.unistat.lastAuthRows.Add(float64(rowsNumber))

	return result, nil
}

func buildLastAuthBulkQuery(uids []uint64, dir string) string {
	return fmt.Sprintf(`
uid, MAX(timestamp) AS timestamp
FROM [%s]
WHERE uid IN (%s)
GROUP BY uid
`,
		buildNodePath(dir, lastAuthTable),
		serializeUIntListIN(uids),
	)
}
