package ytc

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/reqs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/resps"
	"a.yandex-team.ru/passport/infra/libs/go/ytsimple"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/yt/go/yt"
)

type MailUserHistoryRow struct {
	ReversedUnixtime uint64                     `yson:"reversed_unixtime"`
	Operation        string                     `yson:"operation"`
	Module           string                     `yson:"module"`
	Data             *resps.MailUserHistoryItem `yson:"data"`
}

const (
	userHistoryDir     = "users_history"
	corpUserHistoryDir = "corp_users_history"
)

func (c *Client) GetMailUserHistory(ctx context.Context, req *reqs.MailUserHistoryRequest) (*resps.MailUserHistoryResult, error) {
	limit := req.Limit
	result := &resps.MailUserHistoryResult{
		Status: errs.ScalaStatusOk,
		UID:    req.UID,
		Items:  make([]*resps.MailUserHistoryItem, 0, limit),
	}

	tables := c.tablesTTLConfig.getMonthlyTables(
		req.FromTS,
		req.ToTS,
		c.tablesTTLConfig.chooseUsersHistoryTTL(req.Corp),
		uint64(time.Now().Add(60*time.Second).Unix()),
	)
	for uint64(len(result.Items)) < limit && tables.Next() {
		query := buildMailUserHistoryQuery(
			req,
			c.dir,
			tables.TableName(),
			limit-uint64(len(result.Items)),
		)

		if err := c.getMailUserHistoryFromTable(ctx, query, &result.Items); err != nil {
			return nil, &errs.TemporaryError{
				Message: fmt.Sprintf("Failed to fetch mail user history from YT: %v", err),
			}
		}
	}

	c.unistat.mailUserHistoryRows.Add(float64(len(result.Items)))

	return result, nil
}

func (c *Client) getMailUserHistoryFromTable(ctx context.Context, query string, out *[]*resps.MailUserHistoryItem) error {
	var err error

	start := time.Now()
	defer func() {
		c.unistat.responseTimings.Insert(time.Since(start))
		if err != nil {
			c.unistat.errs.Inc()
		}
	}()

	c.unistat.requests.Inc()
	err = ytsimple.SelectAll(ctx, c.yc, query, c.timeout, func(reader yt.TableReader) error {
		row := MailUserHistoryRow{}
		if err := ytsimple.ScanRow(reader, &row); err != nil {
			return err
		}

		row.Data.SetOperation(row.Operation)
		row.Data.SetModule(row.Module)
		row.Data.SetDate(reverseUnixtime(row.ReversedUnixtime))

		if err := row.Data.Decompress(); err != nil {
			c.unistat.decompressErrs.Inc()
			ctxlog.Warnf(ctx, logger.Log(),
				"failed to decompress mail user history item: %v", err,
			)
		}

		(*out) = append((*out), row.Data)
		return nil
	})
	if err != nil {
		return err
	}

	return nil
}

func buildMailUserHistoryQuery(
	req *reqs.MailUserHistoryRequest,
	dir string,
	tableName string,
	limit uint64,
) string {
	condition := fmt.Sprintf(
		`uid=%d AND %d < reversed_unixtime AND reversed_unixtime <= %d`,
		req.UID,
		reverseUnixtime(1000*req.ToTS),
		reverseUnixtime(1000*req.FromTS),
	)
	if len(req.Operation) > 0 {
		condition += fmt.Sprintf(" AND operation IN (%s)", serializeConditionIN(req.Operation))
	}
	if len(req.Module) > 0 {
		condition += fmt.Sprintf(" AND module IN (%s)", serializeConditionIN(req.Module))
	}

	return fmt.Sprintf(
		`
reversed_unixtime,operation,module,data
FROM [%s]
WHERE %s
ORDER BY uid,reversed_unixtime
LIMIT %d`,
		buildNodePath(dir, chooseUserHistoryDir(req.Corp), tableName),
		condition,
		limit,
	)
}

func chooseUserHistoryDir(corp bool) string {
	if corp {
		return corpUserHistoryDir
	}

	return userHistoryDir
}
