package ytc

import (
	"context"
	"fmt"
	"strings"
	"time"

	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/reqs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/resps"
	"a.yandex-team.ru/passport/infra/libs/go/ytsimple"
	"a.yandex-team.ru/yt/go/yt"
)

type PushRow struct {
	Push         string                 `yson:"push_id"`
	Subscription string                 `yson:"subscription_id"`
	UID          uint64                 `yson:"uid"`
	Device       string                 `yson:"device_id"`
	App          string                 `yson:"app_id"`
	Unixtime     int64                  `yson:"unixtime"`
	Data         map[string]interface{} `yson:"data"`
}

const (
	pushDir           = "push"
	pushByUIDDir      = "push-by-uid"
	pushByAppIDDir    = "push-by-app-id"
	pushByDeviceIDDir = "push-by-device-id"
)

func (c *Client) GetPushByPushID(ctx context.Context, req *reqs.PushByPushIDRequest) (*resps.PushResult, error) {
	result := &resps.PushResult{
		Status: errs.ScalaStatusOk,
		Items:  make([]*resps.PushItem, 0),
	}

	tables := c.tablesTTLConfig.getAllMonthlyTables(c.tablesTTLConfig.Push, uint64(time.Now().Add(60*time.Second).Unix()))
	for tables.Next() {
		query := buildPushByPushIDQuery(
			req,
			buildNodePath(c.dir, pushDir, tables.TableName()),
		)

		if err := c.getPushFromTable(ctx, query, &result.Items); err != nil {
			return nil, &errs.TemporaryError{
				Message: fmt.Sprintf("Failed to fetch pushes from YT: %v", err),
			}
		}

		// push_id + subscription_id уникальны - можно досрочно завершать поиск
		if req.Subscription != nil && len(result.Items) > 0 {
			break
		}
	}

	c.unistat.pushRows.Add(float64(len(result.Items)))

	return result, nil
}

func buildPushByPushIDQuery(req *reqs.PushByPushIDRequest, node string) string {
	condition := fmt.Sprintf("push_id = %s", escapeString(req.Push))
	if req.Subscription != nil {
		condition += fmt.Sprintf(" AND subscription_id = %s", escapeString(*req.Subscription))
	}
	if req.UID != nil {
		condition += fmt.Sprintf(" AND uid = %d", *req.UID)
	}
	if req.Device != nil {
		condition += fmt.Sprintf(" AND device_id = %s", escapeString(*req.Device))
	}
	if req.App != nil {
		condition += fmt.Sprintf(" AND app_id = %s", escapeString(*req.App))
	}

	return fmt.Sprintf(
		`push_id,subscription_id,uid,device_id,app_id,unixtime,data
FROM [%s]
WHERE %s`,
		node,
		condition,
	)
}

func (c *Client) GetPushByFields(ctx context.Context, req *reqs.PushByFieldsRequest) (*resps.PushResult, error) {
	limit := req.Limit
	result := &resps.PushResult{
		Status: errs.ScalaStatusOk,
		Items:  make([]*resps.PushItem, 0),
	}

	tables := c.tablesTTLConfig.getAllMonthlyTables(c.tablesTTLConfig.Push, uint64(time.Now().Add(60*time.Second).Unix()))
	for uint64(len(result.Items)) < limit && tables.Next() {
		query := buildPushByFieldsQuery(
			req,
			c.dir,
			tables.TableName(),
			limit-uint64(len(result.Items)),
		)

		if err := c.getPushFromTable(ctx, query, &result.Items); err != nil {
			return nil, &errs.TemporaryError{
				Message: fmt.Sprintf("Failed to fetch pushes from YT: %v", err),
			}
		}
	}

	c.unistat.pushIndexRows.Add(float64(len(result.Items)))

	return result, nil
}

func buildPushByFieldsQuery(req *reqs.PushByFieldsRequest, dir, tableName string, limit uint64) string {
	condition := fmt.Sprintf(
		"%d < reversed_timestamp AND reversed_timestamp <= %d",
		reverseUnixtime(req.ToTS),
		reverseUnixtime(req.FromTS),
	)
	if req.UID != nil {
		condition += fmt.Sprintf(" AND uid = %d", *req.UID)
	}
	if req.Device != nil {
		condition += fmt.Sprintf(" AND device_id = %s", escapeString(*req.Device))
	}
	if req.App != nil {
		condition += fmt.Sprintf(" AND app_id = %s", escapeString(*req.App))
	}
	if req.Subscription != nil {
		condition += fmt.Sprintf(" AND subscription_id = %s", escapeString(*req.Subscription))
	}

	indexDir, keyColumnsList := chooseIndexTable(req)
	keyColumns := strings.Join(keyColumnsList, ",")

	return fmt.Sprintf(
		`push_id,subscription_id,uid,device_id,app_id,unixtime,data
FROM [%s]
JOIN [%s] USING push_id,subscription_id,%s
WHERE %s
ORDER BY %s,reversed_timestamp
LIMIT %d`,
		buildNodePath(dir, indexDir, tableName),
		buildNodePath(dir, pushDir, tableName),
		keyColumns,
		condition,
		keyColumns,
		limit,
	)
}

func chooseIndexTable(req *reqs.PushByFieldsRequest) (string, []string) {
	if req.UID != nil {
		if req.App != nil {
			return pushByAppIDDir, []string{"uid", "app_id"}
		}
		return pushByUIDDir, []string{"uid"}
	}
	return pushByDeviceIDDir, []string{"device_id"}
}

func (c *Client) getPushFromTable(ctx context.Context, query string, out *[]*resps.PushItem) error {
	var err error

	start := time.Now()
	defer func() {
		c.unistat.responseTimings.Insert(time.Since(start))
		if err != nil {
			c.unistat.errs.Inc()
		}
	}()

	c.unistat.requests.Inc()
	err = ytsimple.SelectAll(ctx, c.yc, query, c.timeout, func(reader yt.TableReader) error {
		row := PushRow{}
		if err := ytsimple.ScanRow(reader, &row); err != nil {
			return err
		}

		item := &resps.PushItem{
			Push:         row.Push,
			Subscription: row.Subscription,
			UID:          row.UID,
			Device:       row.Device,
			App:          row.App,
			Unixtime:     row.Unixtime,
		}

		get := func(key string, to *string) {
			if value, ok := row.Data[key]; ok {
				*to = fmt.Sprintf("%v", value)
			}
		}

		get("context", &item.Context)
		get("status", &item.Status)
		get("push_service", &item.Service)
		get("push_event", &item.Event)
		get("details", &item.Details)

		(*out) = append((*out), item)
		return nil
	})
	if err != nil {
		return err
	}

	return nil
}
