package ytc

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/reqs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/resps"
	"a.yandex-team.ru/passport/infra/libs/go/keys"
	"a.yandex-team.ru/passport/infra/libs/go/ytsimple"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/yt/go/yt"
)

type RestoreRow struct {
	ReversedTimestamp uint64 `yson:"reversed_timestamp"`

	Data RestoreRowData `yson:"data"`
}

type RestoreRowData struct {
	Action    string        `yson:"action"`
	RestoreID string        `yson:"restore_id"`
	DataJSON  EncryptedData `yson:"data_json"`
}

const (
	restoreTable = "restore/restore"
)

func (c *Client) GetRestore(
	ctx context.Context,
	request *reqs.EventsRestoreRequest,
	keyMap *keys.KeyMap,
) (*resps.EventsRestoreResult, error) {
	var err error

	start := time.Now()
	defer func() {
		c.unistat.responseTimings.Insert(time.Since(start))
		if err != nil {
			c.unistat.errs.Inc()
		}
	}()

	result := &resps.EventsRestoreResult{
		Status: errs.ScalaStatusOk,
		UID:    request.UID,
		Events: make([]*resps.RestoreEvent, 0, min(100, request.Limit)),
	}

	c.unistat.requests.Inc()
	query := buildRestoreQuery(request, c.dir)

	err = ytsimple.SelectAll(ctx, c.yc, query, c.timeout, func(reader yt.TableReader) error {
		row := &RestoreRow{}
		if err := ytsimple.ScanRow(reader, row); err != nil {
			return err
		}

		result.Events = append(result.Events, c.convertRestoreRow(row, keyMap))
		return nil
	})
	if err != nil {
		return nil, &errs.TemporaryError{
			Message: fmt.Sprintf("Failed to fetch restore from YT: %v", err),
		}
	}

	if request.OrderBy == reqs.OrderByAsc {
		for i, j := 0, len(result.Events)-1; i < j; i, j = i+1, j-1 {
			result.Events[i], result.Events[j] = result.Events[j], result.Events[i]
		}
	}

	c.unistat.restoreRows.Add(float64(len(result.Events)))

	return result, nil
}

func buildRestoreQuery(request *reqs.EventsRestoreRequest, dir string) string {
	return fmt.Sprintf(`
reversed_timestamp, data
FROM [%s]
WHERE uid = %d AND %d < reversed_timestamp AND reversed_timestamp <= %d
ORDER BY uid, reversed_timestamp
LIMIT %d
`,
		buildNodePath(dir, restoreTable),
		request.UID,
		convertToRestoreReversedTimestamp(request.ToTS),
		convertToRestoreReversedTimestamp(request.FromTS),
		request.Limit,
	)
}

func convertToRestoreReversedTimestamp(ts uint64) uint64 {
	return reverseUnixtime(ts * 1000000)
}

func convertToRestoreTimestamp(ts uint64) float64 {
	return float64(reverseUnixtime(ts)) / 1000000
}

func (c *Client) convertRestoreRow(row *RestoreRow, keyMap *keys.KeyMap) *resps.RestoreEvent {
	dataJSON, err := row.Data.DataJSON.Decrypt(keyMap)
	if err != nil {
		c.unistat.decryptErrs.Inc()
		logger.Log().Warnf("failed to decrypt restore.data_json: %v", err)
	}

	return &resps.RestoreEvent{
		Timestamp: convertToRestoreTimestamp(row.ReversedTimestamp),
		Action:    row.Data.Action,
		RestoreID: row.Data.RestoreID,
		DataJSON:  string(dataJSON),
	}
}
