package ytc

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/reqs"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/resps"
	"a.yandex-team.ru/passport/infra/libs/go/ytsimple"
	"a.yandex-team.ru/yt/go/yt"
)

type SenderLastLetterRow struct {
	UnsibscribeList uint64 `yson:"unsibscribe_list"`
	Timestamp       uint64 `yson:"timestamp"`
}

const (
	senderLastLetterTable = "sendr/last_letter"
)

func (c *Client) GetSenderLastLetter(ctx context.Context, req *reqs.SenderLastLetterRequest) (*resps.SenderLastLetterResult, error) {
	var err error

	start := time.Now()
	defer func() {
		c.unistat.responseTimings.Insert(time.Since(start))
		if err != nil {
			c.unistat.errs.Inc()
		}
	}()

	result := &resps.SenderLastLetterResult{
		Status: errs.ScalaStatusOk,
		UID:    req.UID,
		Items:  make(map[uint64]resps.SenderLastLetterItem),
	}

	c.unistat.requests.Inc()
	query := buildSenderLastLetterQuery(req, c.dir, time.Now())
	err = ytsimple.SelectAll(ctx, c.yc, query, c.timeout, func(reader yt.TableReader) error {
		row := SenderLastLetterRow{}
		if err := ytsimple.ScanRow(reader, &row); err != nil {
			return err
		}

		result.Items[row.UnsibscribeList] = resps.SenderLastLetterItem{
			EmailSent: row.Timestamp,
		}
		return nil
	})
	if err != nil {
		return nil, &errs.TemporaryError{
			Message: fmt.Sprintf("Failed to fetch sender last letter from YT: %v", err),
		}
	}

	return result, nil
}

func buildSenderLastLetterQuery(req *reqs.SenderLastLetterRequest, dir string, now time.Time) string {
	return fmt.Sprintf(`
unsibscribe_list,timestamp
FROM [%s]
WHERE uid = %d AND timestamp > %d`,
		buildNodePath(dir, senderLastLetterTable),
		req.UID,
		now.Unix()-int64(req.MaxAge),
	)
}
