package ytc

import (
	"context"
	"fmt"
	"strconv"
	"strings"

	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/reqs"
)

type YasmsSmsHistoryRowBase struct {
	GlobalSmsID   string `yson:"global_sms_id"`
	Action        string `yson:"action"`
	UnixtimeMicro int64  `yson:"unixtime"`
}

type YasmsSmsHistoryRow struct {
	YasmsSmsHistoryRowBase

	UID   *uint64 `yson:"uid"`
	Phone *uint64 `yson:"phone"`

	Data YasmsSmsHistoryData `yson:"data"`
}

type YasmsSmsHistoryData struct {
	Extra         map[string]interface{} `yson:"extra"`
	EncryptedText *EncryptedData         `yson:"encrypted_text"`
}

const (
	yasmsSmsHistoryTable        = "yasms/sms_history/sms_history"
	yasmsSmsHistoryByPhoneTable = "yasms/sms_history_by_phone/sms_history_by_phone"
	yasmsSmsHistoryByUIDTable   = "yasms/sms_history_by_uid/sms_history_by_uid"
)

type YasmsSmsHistorySelectType = int

const (
	YasmsSmsHistoryBase YasmsSmsHistorySelectType = iota
	YasmsSmsHistoryCommon
	YasmsSmsHistoryFull
)

func (c *Client) SelectYasmsSmsHistory(
	ctx context.Context,
	req *reqs.SmsByGlobalIDRequest,
	selectType YasmsSmsHistorySelectType,
	collector Collector[YasmsSmsHistoryRow],
) error {
	var rowsTotal uint64
	query := buildYasmsSmsHistoryQuery(req, selectType, c.dir)
	if err := c.selectAll(ctx, query, basicScanner(&rowsTotal, collector)); err != nil {
		return err
	}

	c.unistat.yasmsSmsHistoryRows.Add(float64(rowsTotal))
	return nil
}

func (c *Client) SelectYasmsSmsHistoryByFields(
	ctx context.Context,
	req *reqs.SmsByFieldsRequest,
	selectType YasmsSmsHistorySelectType,
	collector Collector[YasmsSmsHistoryRow],
) error {
	query := buildYasmsSmsHistoryByFieldsQuery(req, selectType, c.dir)
	if query == "" {
		return nil
	}

	var rowsTotal uint64
	if err := c.selectAll(ctx, query, basicScanner(&rowsTotal, collector)); err != nil {
		return err
	}

	c.unistat.yasmsSmsHistoryIndexRows.Add(float64(rowsTotal))
	return nil
}

func buildYasmsSmsHistoryQuery(req *reqs.SmsByGlobalIDRequest, selectType YasmsSmsHistorySelectType, dir string) string {
	return fmt.Sprintf(`
%s
FROM [%s]
WHERE global_sms_id = %s
ORDER BY unixtime DESC
LIMIT %d
`,
		chooseYasmsSmsHistoryFields(selectType),
		buildNodePath(dir, yasmsSmsHistoryTable),
		escapeString(req.GlobalSmsID),
		req.Limit,
	)
}

func buildYasmsSmsHistoryByFieldsQuery(req *reqs.SmsByFieldsRequest, selectType YasmsSmsHistorySelectType, dir string) string {
	conditions := make([]string, 0, 4)
	if req.Phone != nil {
		phone, err := strconv.ParseUint(strings.TrimPrefix(*req.Phone, "+"), 10, 64)
		if err != nil {
			return ""
		}

		conditions = append(conditions, fmt.Sprintf("phone = %d", phone))
	}
	if req.UID != nil {
		conditions = append(conditions, fmt.Sprintf("uid = %d", *req.UID))
	}

	if len(conditions) == 0 {
		return ""
	}

	if req.ToTS != nil {
		conditions = append(conditions, fmt.Sprintf("%d < reversed_timestamp", reverseUnixtimeMicro(*req.ToTS)))
	}
	if req.FromTS != nil {
		conditions = append(conditions, fmt.Sprintf("reversed_timestamp <= %d", reverseUnixtimeMicro(*req.FromTS)))
	}

	table, key := chooseYasmsSmsHistoryIndex(req)

	return fmt.Sprintf(`
%s
FROM [%s]
JOIN [%s] USING global_sms_id,action,%s
WHERE %s
ORDER BY %s,reversed_timestamp
LIMIT %d
`,
		chooseYasmsSmsHistoryFields(selectType),
		buildNodePath(dir, table),
		buildNodePath(dir, yasmsSmsHistoryTable),
		key,
		strings.Join(conditions, " AND "),
		key,
		req.Limit,
	)
}

func chooseYasmsSmsHistoryFields(selectType YasmsSmsHistorySelectType) string {
	switch selectType {
	case YasmsSmsHistoryBase:
		return "global_sms_id,action,unixtime"
	case YasmsSmsHistoryCommon:
		return "global_sms_id,action,unixtime,uid,phone"
	default:
		return "global_sms_id,action,unixtime,uid,phone,data"
	}
}

func chooseYasmsSmsHistoryIndex(req *reqs.SmsByFieldsRequest) (string, string) {
	if req.Phone != nil {
		return yasmsSmsHistoryByPhoneTable, "phone"
	}
	return yasmsSmsHistoryByUIDTable, "uid"
}
