package ytc

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/historydb_api2/internal/errs"
	"a.yandex-team.ru/passport/infra/libs/go/ytsimple"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/passport/shared/golibs/unistat"
	"a.yandex-team.ru/yt/go/yson"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
)

type Config struct {
	Cluster    string `json:"cluster"`
	CypressDir string `json:"cypress_dir"`
	Timeout    uint64 `json:"timeout_ms"`
	UseTLS     bool   `json:"use_tls"`

	TableTTL TablesTTLConfig `json:"table_ttl"`
}

type Client struct {
	yc      yt.Client
	timeout yson.Duration
	dir     string
	unistat stats

	tablesTTLConfig TablesTTLConfig
}

type stats struct {
	responseTimings          *unistat.TimeStat
	requests                 *unistat.SignalDiff
	errs                     *unistat.SignalDiff
	decompressErrs           *unistat.SignalDiff
	decryptErrs              *unistat.SignalDiff
	mailUserHistoryRows      *unistat.SignalDiff
	pushRows                 *unistat.SignalDiff
	pushIndexRows            *unistat.SignalDiff
	authsRows                *unistat.SignalDiff
	failedAuthsRows          *unistat.SignalDiff
	lastAuthRows             *unistat.SignalDiff
	restoreRows              *unistat.SignalDiff
	yasmsSmsHistoryRows      *unistat.SignalDiff
	yasmsSmsHistoryIndexRows *unistat.SignalDiff
}

const TvmYtAlias = "yt"

func NewClient(cfg Config, tvmClient tvm.Client) (*Client, error) {
	cl, err := ythttp.NewClient(&yt.Config{
		Proxy:  cfg.Cluster,
		UseTLS: cfg.UseTLS,
		TVMFn: func(ctx context.Context) (string, error) {
			return tvmClient.GetServiceTicketForAlias(ctx, TvmYtAlias)
		},
		Logger:           logger.Log().With(log.Nil("YtClient")).Structured(),
		CompressionCodec: yt.ClientCodecGZIP,
	})
	if err != nil {
		return nil, xerrors.Errorf("Failed to create YT client: %w", err)
	}

	responseTimings, err := unistat.DefaultChunk.CreateTimeStats(
		"yt.response_time",
		unistat.CreateTimeBoundsFromMaxValue(10*time.Second),
	)
	if err != nil {
		return nil, xerrors.Errorf("Failed to create time stats: %w", err)
	}

	return &Client{
		yc:      cl,
		timeout: yson.Duration(time.Duration(cfg.Timeout) * time.Millisecond),
		dir:     cfg.CypressDir,
		unistat: stats{
			responseTimings:          responseTimings,
			requests:                 unistat.DefaultChunk.CreateSignalDiff("yt.requests"),
			errs:                     unistat.DefaultChunk.CreateSignalDiff("yt.errors"),
			decompressErrs:           unistat.DefaultChunk.CreateSignalDiff("yt.decompress_errors"),
			decryptErrs:              unistat.DefaultChunk.CreateSignalDiff("yt.decrypt_errors"),
			mailUserHistoryRows:      unistat.DefaultChunk.CreateSignalDiff("yt.rows.mail_user_history"),
			pushRows:                 unistat.DefaultChunk.CreateSignalDiff("yt.rows.push"),
			pushIndexRows:            unistat.DefaultChunk.CreateSignalDiff("yt.rows.push_index"),
			authsRows:                unistat.DefaultChunk.CreateSignalDiff("yt.rows.auths"),
			failedAuthsRows:          unistat.DefaultChunk.CreateSignalDiff("yt.rows.failed_auths"),
			lastAuthRows:             unistat.DefaultChunk.CreateSignalDiff("yt.rows.lastauth"),
			restoreRows:              unistat.DefaultChunk.CreateSignalDiff("yt.rows.restore"),
			yasmsSmsHistoryRows:      unistat.DefaultChunk.CreateSignalDiff("yt.rows.yasms_sms_history"),
			yasmsSmsHistoryIndexRows: unistat.DefaultChunk.CreateSignalDiff("yt.rows.yasms_sms_history_index"),
		},
		tablesTTLConfig: cfg.TableTTL,
	}, nil
}

func (c *Client) collectRequestUnistat(start time.Time, err error) {
	c.unistat.requests.Inc()
	c.unistat.responseTimings.Insert(time.Since(start))
	if err != nil {
		c.unistat.errs.Inc()
	}
}

func (c *Client) selectAll(ctx context.Context, query string, scanner func(reader yt.TableReader) error) (err error) {
	start := time.Now()
	defer c.collectRequestUnistat(start, err)

	err = ytsimple.SelectAll(ctx, c.yc, query, c.timeout, scanner)
	return
}

type CollectibleRow interface {
	YasmsSmsHistoryRow
}

type Collector[Row CollectibleRow] interface {
	CollectFromYt(row *Row) error
}

func basicScanner[Row CollectibleRow](rowsTotal *uint64, collector Collector[Row]) func(reader yt.TableReader) error {
	return func(reader yt.TableReader) error {
		row := Row{}
		if err := ytsimple.ScanRow(reader, &row); err != nil {
			return &errs.TemporaryError{
				Message: fmt.Sprintf("Failed to fetch row from YT: %v", err),
			}
		}
		*rowsTotal += 1

		return collector.CollectFromYt(&row)
	}
}
