import json
import os
import yatest.common as yc

from ctypes import c_uint64
from passport.infra.recipes.common import log, start_daemon
from yatest.common import network


C_INT64_MAX = c_uint64(-1).value >> 1


def reverse_timestamp(ts):
    return C_INT64_MAX - ts


def deep_sorted(item):
    def key_func(x):
        if isinstance(x, dict):
            return sorted((key, key_func(value)) for key, value in x.items())
        return x

    if isinstance(item, dict):
        return {key: deep_sorted(value) for key, value in item.items()}
    if isinstance(item, list):
        return sorted((deep_sorted(x) for x in item), key=key_func)
    if isinstance(item, tuple):
        return tuple(deep_sorted(x) for x in item)
    return item


def pretty_json(value, indent=2, **kwargs):
    return json.dumps(json.loads(value), indent=indent, **kwargs)


class HistoryDBApi2Fixture:
    def __init__(self, yt_proxy, auths_yt_threshold=0):
        self._pm = network.PortManager()

        self.http_port = self._pm.get_tcp_port()
        self.unistat_port = self._pm.get_tcp_port()
        self.oldapi_port = self._pm.get_tcp_port()

        tvm_secret_path = yc.output_path('tvm.secret')
        tvm_cache_dir = yc.output_path('tvm_cache')
        ping_path = yc.output_path("ping.html")
        oauth_token_path = yc.output_path('yt.token')
        config_path = yc.output_path("config.json")
        grants_path = yc.output_path("consumer_grants.json")
        restore_keys_path = yc.output_path('restore.keys')
        yasms_private_keys_path = yc.output_path('yasms_private.keys')

        try:
            with open('./tvmapi.port') as f:
                tvm_port = int(f.read())
        except IOError:
            log('Could not find tvm port file: ./tvmapi.port')

        with open(tvm_secret_path, 'w') as f:
            f.write("bAicxJVa5uVY7MjDlapthw")

        with open(restore_keys_path, 'w') as f:
            f.write('{"1":"d3afIKJ+MB2ZnAMxwsNQN729osL9meb33YP/5wilfCU="}')

        with open(yasms_private_keys_path, 'w') as f:
            f.write('{"1":"MWUSYx6eY547SSNiOYijE+cv9B1Beg+bfmCGefam5vE="}')

        historydb_api2_config = {
            "http_common": {
                "listen_address": "localhost",
                "port": self.http_port,
                "cert": "",
                "key": "",
            },
            "http_unistat": {
                "listen_address": "localhost",
                "port": self.unistat_port,
            },
            "logger": {
                "file": yc.output_path("common.log"),
            },
            "service": {
                "common": {
                    "ping_file": ping_path,
                    "access_log": {
                        "file": yc.output_path("access.log"),
                        "disable_printing_level": True,
                    },
                    "auths_yt_threshold": auths_yt_threshold,
                    "yasms_sms_history_from_yt": True,
                    "push_subscription_width": 86400,
                },
                "grants": {
                    "path": grants_path,
                    "period_sec": 300,
                },
                "hbase_api": {
                    "consumer_name": "localhost",
                    "host": "localhost",
                    "port": self.oldapi_port,
                    "retries": 3,
                    "timeout_ms": 60000,
                },
                "max_time_depth_sec": {
                    "users_history": 5356800,
                    "corp_users_history": 2678400,
                },
                "tvm": {
                    "self_tvm_id": 1000501,
                    "secret_filepath": tvm_secret_path,
                    "disk_cache_dir": tvm_cache_dir,
                    "check_service_tickets": True,
                    "blackbox_env": "test",
                    "destinations": {
                        "yt": 1000502,
                    },
                    "tvm_host": "http://127.0.0.1",
                    "tvm_port": tvm_port,
                },
                "yt": {
                    "cluster": yt_proxy,
                    "cypress_dir": "//home",
                    "timeout_ms": 30000,
                    "use_tls": False,
                    "table_ttl": {
                        "users_history_table_ttl__sec": 5356800,
                        "corp_users_history_table_ttl__sec": 2678400,
                        "push_table_ttl__sec": 8035200,
                        "auths_table_ttl__sec": 5356800,
                        "future_frontier__sec": 86400,
                    },
                },
                "keys": {
                    "restore": {
                        "keys_file": restore_keys_path,
                        "keys_encoding": "base64",
                    },
                    "yasms_private": {
                        "keys_file": yasms_private_keys_path,
                        "keys_encoding": "base64",
                    },
                },
            },
        }

        grants_config = {
            "localhost": {
                "client": {},
                "grants": {
                    "account": [
                        "auths",
                        "events",
                        "lastauth",
                    ],
                    "push": [
                        "base",
                    ],
                    "yasms": [
                        "sms_history",
                        "sms_laststatus",
                    ],
                },
                "networks": [
                    "127.0.0.1",
                    "::1",
                ],
            },
            "with_tvm": {
                "client": {
                    "client_id": 1000503,
                    "client_name": "with_tvm",
                },
                "grants": {
                    "account": [
                        "auths",
                        "events",
                        "lastauth",
                    ],
                    "push": [
                        "base",
                    ],
                    "push_subscription": [
                        "base",
                    ],
                    "yasms": [
                        "sms_history",
                        "sms_laststatus",
                    ],
                },
                "networks": [
                    "127.0.0.1",
                    "::1",
                ],
            },
            "wo_grants": {
                "client": {},
                "grants": {},
                "networks": [
                    "127.0.0.1",
                    "::1",
                ],
            },
            "wo_grants_with_tvm": {
                "client": {
                    "client_id": 1000503,
                    "client_name": "with_tvm",
                },
                "grants": {},
                "networks": [
                    "127.0.0.1",
                    "::1",
                ],
            },
        }

        if not os.path.isdir(tvm_cache_dir):
            os.mkdir(tvm_cache_dir)

        with open(ping_path, 'w') as f:
            f.write("Pong")

        with open(oauth_token_path, 'w') as f:
            f.write("")

        with open(config_path, 'w') as f:
            json.dump(historydb_api2_config, f, indent=2)

        with open(grants_path, 'w') as f:
            json.dump(grants_config, f, indent=2)

        command = [
            yc.build_path('passport/infra/daemons/historydb_api2/ut/oldapi_mock/oldapi_server'),
            str(self.oldapi_port),
        ]
        self._oldapi_process = start_daemon(command, os.environ.copy(), self.oldapi_port)

        command = [
            yc.build_path('passport/infra/daemons/historydb_api2/cmd/historydb_api2'),
            '-c',
            config_path,
        ]
        self._process = start_daemon(command, os.environ.copy(), self.http_port)

    def stop(self):
        self._pm.release()
        self._process.terminate()
        self._oldapi_process.terminate()
