package main

import (
	"encoding/json"
	"fmt"
	"net/http"
	"os"
	"strconv"
)

func SimpleHandler(response interface{}) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		body, err := json.Marshal(response)
		if err != nil {
			fmt.Printf("Failed to encode json response: %s", err)
		}

		_, err = w.Write(body)
		if err != nil {
			fmt.Printf("Failed to write response: %s", err)
		}
	}
}

func AuthsHandler(w http.ResponseWriter, r *http.Request) {
	query := r.URL.Query()
	uid, err := strconv.ParseUint(query.Get("uid"), 10, 64)
	if err != nil {
		fmt.Printf("Failed to parse uid query param: %s", err)
	}
	fromTS, err := strconv.ParseUint(query.Get("from_ts"), 10, 64)
	if err != nil {
		fmt.Printf("Failed to parse from_ts query param: %s", err)
	}
	toTS, err := strconv.ParseUint(query.Get("to_ts"), 10, 64)
	if err != nil {
		fmt.Printf("Failed to parse from_ts query param: %s", err)
	}

	var response interface{}
	if uid == 100500 && fromTS <= 1555555555 && toTS >= 1555555555 {
		response = map[string]interface{}{
			"status": "ok",
			"uid":    uid,
			"auths": []map[string]interface{}{
				{
					"timestamp":   1555555555.555555,
					"type":        "web",
					"status":      "failed",
					"client_name": "bb",
				},
			},
		}
	} else if uid == 17 && fromTS <= 1555555555 && toTS >= 1555555555 {
		response = map[string]interface{}{
			"status": "ok",
			"uid":    uid,
			"history": []map[string]interface{}{
				{
					"timestamp": 1555545600,
					"auths": []map[string]interface{}{
						{
							"count": 666,
							"auth": map[string]interface{}{
								"authtype": "hbase",
								"status":   "successful",
							},
						},
					},
				},
			},
		}
	} else {
		response = map[string]interface{}{
			"status":  "ok",
			"uid":     uid,
			"auths":   []interface{}{},
			"history": []interface{}{},
		}
	}

	SimpleHandler(response)(w, r)
}

func RunServer(port uint16) {
	fmt.Printf("Starting \"old-api\" server at port %d\n", port)

	http.HandleFunc("/", SimpleHandler("Mock"))
	http.HandleFunc("/ping", SimpleHandler("Pong"))
	http.HandleFunc("/2/auths/", AuthsHandler)
	http.HandleFunc("/2/auths/failed/", AuthsHandler)
	http.HandleFunc("/2/auths/aggregated/runtime/", AuthsHandler)

	err := http.ListenAndServe(fmt.Sprintf(":%d", port), nil)
	if err != http.ErrServerClosed {
		fmt.Println(err)
	}
}

func main() {
	if len(os.Args) < 2 {
		fmt.Printf("Usage: %s <port>\n", os.Args[0])
		return
	}

	port, err := strconv.ParseUint(os.Args[1], 10, 16)
	if err != nil {
		fmt.Printf("Invalid argument <port>: must be uint, got %s\n", os.Args[1])
		return
	}

	RunServer(uint16(port))
}
