import yt.wrapper

from passport.infra.daemons.historydb_api2.ut.common import reverse_timestamp
import passport.infra.daemons.lbchdb.ut_medium.common.schemes as schemes

NON_EXISTING_PUSH_ID = "non_existing_push_id"
ONE_SUB_PUSH_ID = "one_subscription_push_id"
NO_SUB_PUSH_ID = "no_subscription_push_id"
MULTIPLE_SUBS_PUSH_ID = "multiple_subscriptions_push_id"
PAST_MONTH_PUSH_ID = "past_month_push_id"
MULTIPLE_MONTH_PUSH_ID = "multiple_month_push_id"

DAY_SECONDS = 24 * 60 * 60


class PushTestFixture:
    def __init__(self):
        ts = schemes.create_push_tables(4)

        self._items = {}

        def add_rows(table, rows):
            self.__add_items(rows)

            def insert_rows(dir, prep_rows):
                yt.wrapper.insert_rows(table.path(dir), prep_rows, atomicity="none")

            insert_rows(schemes.PUSH_TABLE_DIR, self.__make_push_rows(rows))
            insert_rows(schemes.PUSH_BY_UID_TABLE_DIR, self.__make_by_uid_rows(rows))
            insert_rows(schemes.PUSH_BY_DEVICE_ID_TABLE_DIR, self.__make_by_device_id_rows(rows))
            insert_rows(schemes.PUSH_BY_APP_ID_TABLE_DIR, self.__make_by_app_id_rows(rows))

        current_table = schemes.MonthlyTable(ts)
        self._reserve_table_ts = current_table.next().ts()
        self._first_table_ts = current_table.ts()
        add_rows(current_table, self.__first_table_rows(self._first_table_ts))

        current_table = current_table.prev()
        self._second_table_ts = current_table.ts()
        add_rows(current_table, self.__second_table_rows(self._second_table_ts))

        current_table = current_table.prev()
        self._third_table_ts = current_table.ts()
        add_rows(current_table, self.__third_table_rows(self._third_table_ts))

    def __add_items(self, rows):
        for row in rows:
            subs = self._items.setdefault(row["push_id"], {})

            if row["subscription_id"] in subs:
                raise RuntimeError(
                    "Incorrect push test setup: push_id '%s' has multiple subscriptions with same id '%s'"
                    % (
                        row["push_id"],
                        row["subscription_id"],
                    )
                )

            subs[row["subscription_id"]] = {
                "push_id": row["push_id"],
                "subscription_id": row["subscription_id"],
                "uid": row["uid"],
                "unixtime": row["unixtime"],
                "device_id": row.get("device_id", ""),
                "app_id": row.get("app_id", ""),
                "push_service": row.get("data", {}).get("push_service", ""),
                "push_event": row.get("data", {}).get("push_event", ""),
                "details": row.get("data", {}).get("details", ""),
                "context": row.get("data", {}).get("context", ""),
                "status": row.get("data", {}).get("status", ""),
            }

    @staticmethod
    def __make_push_rows(rows):
        return [
            {
                "push_id": row["push_id"],
                "subscription_id": row["subscription_id"],
                "uid": row["uid"],
                "unixtime": row["unixtime"],
                "device_id": row.get("device_id", ""),
                "app_id": row.get("app_id", ""),
                "data": row.get("data", {}),
            }
            for row in rows
        ]

    @staticmethod
    def __make_by_uid_rows(rows):
        return [
            {
                "uid": row["uid"],
                "reversed_timestamp": reverse_timestamp(row["unixtime"]),
                "push_id": row["push_id"],
                "subscription_id": row["subscription_id"],
            }
            for row in rows
        ]

    @staticmethod
    def __make_by_device_id_rows(rows):
        return [
            {
                "device_id": row["device_id"],
                "reversed_timestamp": reverse_timestamp(row["unixtime"]),
                "push_id": row["push_id"],
                "subscription_id": row["subscription_id"],
            }
            for row in rows
            if "device_id" in row
        ]

    @staticmethod
    def __make_by_app_id_rows(rows):
        return [
            {
                "uid": row["uid"],
                "app_id": row["app_id"],
                "reversed_timestamp": reverse_timestamp(row["unixtime"]),
                "push_id": row["push_id"],
                "subscription_id": row["subscription_id"],
            }
            for row in rows
            if "app_id" in row
        ]

    @staticmethod
    def __first_table_rows(table_ts):
        return [
            {
                "push_id": ONE_SUB_PUSH_ID,
                "subscription_id": "foo",
                "uid": 123,
                "device_id": "123.device.foo",
                "app_id": "foo",
                "unixtime": table_ts,
                "data": {
                    "context": "some_context",
                    "status": "some_status",
                    "push_service": "some_service",
                    "push_event": "some_event",
                    "details": "some_details",
                },
            },
            {
                "push_id": NO_SUB_PUSH_ID,
                "subscription_id": "n/a",
                "uid": 123,
                "unixtime": table_ts + DAY_SECONDS,
            },
            {
                "push_id": MULTIPLE_SUBS_PUSH_ID,
                "subscription_id": "foo_with_status",
                "uid": 100500,
                "device_id": "100500.device.foo",
                "app_id": "foo",
                "unixtime": table_ts + 2 * DAY_SECONDS,
                "data": {
                    "status": "some_status",
                },
            },
            {
                "push_id": MULTIPLE_SUBS_PUSH_ID,
                "subscription_id": "bar_with_context",
                "uid": 100500,
                "device_id": "100500.device.bar",
                "app_id": "bar",
                "unixtime": table_ts + 3 * DAY_SECONDS,
                "data": {
                    "context": "some_context",
                },
            },
            {
                "push_id": MULTIPLE_SUBS_PUSH_ID,
                "subscription_id": "bar_without_data",
                "uid": 100500,
                "device_id": "100500.device.bar",
                "app_id": "bar",
                "unixtime": table_ts + 4 * DAY_SECONDS,
            },
        ]

    @staticmethod
    def __second_table_rows(table_ts):
        return [
            {
                "push_id": PAST_MONTH_PUSH_ID,
                "subscription_id": "foo",
                "uid": 123,
                "device_id": "123.device.foo",
                "app_id": "foo",
                "unixtime": table_ts,
            },
            {
                "push_id": MULTIPLE_MONTH_PUSH_ID,
                "subscription_id": "foo",
                "uid": 100500,
                "device_id": "100500.device.foo",
                "app_id": "foo",
                "unixtime": table_ts + DAY_SECONDS,
            },
        ]

    @staticmethod
    def __third_table_rows(table_ts):
        return [
            {
                "push_id": MULTIPLE_MONTH_PUSH_ID,
                "subscription_id": "bar",
                "uid": 100500,
                "device_id": "100500.device.bar",
                "app_id": "bar",
                "unixtime": table_ts + 27 * DAY_SECONDS,
            },
        ]

    def by_push_id_requests(self):
        return [
            {
                "params": {
                    "push": NON_EXISTING_PUSH_ID,
                },
                "items": [],
            },
            {
                "params": {
                    "push": ONE_SUB_PUSH_ID,
                },
                "items": [
                    self._items[ONE_SUB_PUSH_ID]["foo"],
                ],
            },
            {
                "params": {
                    "push": NO_SUB_PUSH_ID,
                },
                "items": [
                    self._items[NO_SUB_PUSH_ID]["n/a"],
                ],
            },
            {
                "params": {
                    "push": MULTIPLE_SUBS_PUSH_ID,
                },
                "items": [
                    self._items[MULTIPLE_SUBS_PUSH_ID]["foo_with_status"],
                    self._items[MULTIPLE_SUBS_PUSH_ID]["bar_with_context"],
                    self._items[MULTIPLE_SUBS_PUSH_ID]["bar_without_data"],
                ],
            },
            {
                "params": {
                    "push": PAST_MONTH_PUSH_ID,
                },
                "items": [
                    self._items[PAST_MONTH_PUSH_ID]["foo"],
                ],
            },
            {
                "params": {
                    "push": MULTIPLE_MONTH_PUSH_ID,
                },
                "items": [
                    self._items[MULTIPLE_MONTH_PUSH_ID]["foo"],
                    self._items[MULTIPLE_MONTH_PUSH_ID]["bar"],
                ],
            },
            {
                "params": {
                    "push": ONE_SUB_PUSH_ID,
                    "uid": 1,
                },
                "items": [],
            },
            {
                "params": {
                    "push": ONE_SUB_PUSH_ID,
                    "uid": 123,
                },
                "items": [
                    self._items[ONE_SUB_PUSH_ID]["foo"],
                ],
            },
            {
                "params": {
                    "push": MULTIPLE_MONTH_PUSH_ID,
                    "subscription": "non_existent",
                },
                "items": [],
            },
            {
                "params": {
                    "push": MULTIPLE_MONTH_PUSH_ID,
                    "subscription": "foo",
                },
                "items": [
                    self._items[MULTIPLE_MONTH_PUSH_ID]["foo"],
                ],
            },
            {
                "params": {
                    "push": MULTIPLE_MONTH_PUSH_ID,
                    "device": "non_existent",
                },
                "items": [],
            },
            {
                "params": {
                    "push": MULTIPLE_MONTH_PUSH_ID,
                    "device": "100500.device.foo",
                },
                "items": [
                    self._items[MULTIPLE_MONTH_PUSH_ID]["foo"],
                ],
            },
            {
                "params": {
                    "push": MULTIPLE_MONTH_PUSH_ID,
                    "app": "non_existent",
                },
                "items": [],
            },
            {
                "params": {
                    "push": MULTIPLE_MONTH_PUSH_ID,
                    "app": "foo",
                },
                "items": [
                    self._items[MULTIPLE_MONTH_PUSH_ID]["foo"],
                ],
            },
        ]

    def by_uid_requests(self):
        return [
            {
                "params": {
                    "uid": 123,
                    "from_ts": self._first_table_ts,
                    "to_ts": self._reserve_table_ts,
                },
                "items": [
                    self._items[NO_SUB_PUSH_ID]["n/a"],
                    self._items[ONE_SUB_PUSH_ID]["foo"],
                ],
            },
            {
                "params": {
                    "uid": 123,
                    "from_ts": self._first_table_ts,
                    "to_ts": self._first_table_ts + DAY_SECONDS,
                },
                "items": [
                    self._items[ONE_SUB_PUSH_ID]["foo"],
                ],
            },
            {
                "params": {
                    "uid": 123,
                    "from_ts": self._second_table_ts,
                    "to_ts": self._first_table_ts,
                },
                "items": [
                    self._items[PAST_MONTH_PUSH_ID]["foo"],
                ],
            },
            {
                "params": {
                    "uid": 123,
                    "from_ts": 0,
                    "to_ts": 20000000000,
                    "limit": 1,
                },
                "items": [
                    self._items[NO_SUB_PUSH_ID]["n/a"],
                ],
            },
            {
                "params": {
                    "uid": 100500,
                    "device": "100500.device.foo",
                    "from_ts": 0,
                    "to_ts": 20000000000,
                },
                "items": [
                    self._items[MULTIPLE_SUBS_PUSH_ID]["foo_with_status"],
                    self._items[MULTIPLE_MONTH_PUSH_ID]["foo"],
                ],
            },
            {
                "params": {
                    "uid": 100500,
                    "subscription": "foo",
                    "from_ts": 0,
                    "to_ts": 20000000000,
                },
                "items": [
                    self._items[MULTIPLE_MONTH_PUSH_ID]["foo"],
                ],
            },
        ]

    def by_device_id_requests(self):
        return [
            {
                "params": {
                    "device": "100500.device.foo",
                    "from_ts": self._first_table_ts,
                    "to_ts": self._reserve_table_ts,
                },
                "items": [
                    self._items[MULTIPLE_SUBS_PUSH_ID]["foo_with_status"],
                ],
            },
            {
                "params": {
                    "device": "100500.device.foo",
                    "from_ts": 0,
                    "to_ts": 20000000000,
                },
                "items": [
                    self._items[MULTIPLE_SUBS_PUSH_ID]["foo_with_status"],
                    self._items[MULTIPLE_MONTH_PUSH_ID]["foo"],
                ],
            },
            {
                "params": {
                    "device": "100500.device.bar",
                    "from_ts": 0,
                    "to_ts": 20000000000,
                    "limit": 1,
                },
                "items": [
                    self._items[MULTIPLE_SUBS_PUSH_ID]["bar_without_data"],
                ],
            },
            {
                "params": {
                    "device": "100500.device.bar",
                    "subscription": "bar_with_context",
                    "from_ts": 0,
                    "to_ts": 20000000000,
                },
                "items": [
                    self._items[MULTIPLE_SUBS_PUSH_ID]["bar_with_context"],
                ],
            },
        ]

    def by_app_id_requests(self):
        return [
            {
                "params": {
                    "uid": 100500,
                    "app": "foo",
                    "from_ts": self._first_table_ts,
                    "to_ts": self._reserve_table_ts,
                },
                "items": [
                    self._items[MULTIPLE_SUBS_PUSH_ID]["foo_with_status"],
                ],
            },
            {
                "params": {
                    "uid": 100500,
                    "app": "foo",
                    "from_ts": 0,
                    "to_ts": 20000000000,
                },
                "items": [
                    self._items[MULTIPLE_SUBS_PUSH_ID]["foo_with_status"],
                    self._items[MULTIPLE_MONTH_PUSH_ID]["foo"],
                ],
            },
            {
                "params": {
                    "uid": 100500,
                    "app": "bar",
                    "from_ts": 0,
                    "to_ts": 20000000000,
                    "limit": 1,
                },
                "items": [
                    self._items[MULTIPLE_SUBS_PUSH_ID]["bar_without_data"],
                ],
            },
            {
                "params": {
                    "uid": 100500,
                    "app": "bar",
                    "subscription": "bar_with_context",
                    "from_ts": 0,
                    "to_ts": 20000000000,
                },
                "items": [
                    self._items[MULTIPLE_SUBS_PUSH_ID]["bar_with_context"],
                ],
            },
        ]
